package com.paypal.controllers.webhooks;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.paypal.api.payments.Event;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.service.PayPalReplenishmentService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import com.paypal.hybris.facade.facades.PayPalCheckoutFacade;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.site.BaseSiteService;
import java.io.IOException;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;

@Controller
@RequestMapping(value = "/paypal/webhook")
public class WebhookListener {

    private static final String ORDER_APPROVED_EVENT_TYPE = "CHECKOUT.ORDER.APPROVED";
    private static final String PAYMENT_AUTHORIZATION_VOIDED_EVENT_TYPE = "PAYMENT.AUTHORIZATION.VOIDED";
    private static final String WEBHOOK_RESOURCE_TYPE_CAPTURE = "capture";
    private static final String BILLING_AGREEMENT_EVENT_TYPE = "BILLING_AGREEMENTS.AGREEMENT.CANCELLED";
    private static final String RESOURCE = "resource";

    private static final Logger LOG = Logger.getLogger(WebhookListener.class);

    @Resource(name = "payPalCheckoutFacade")
    private PayPalCheckoutFacade payPalCheckoutFacade;

    @Resource(name = "payPalCartService")
    private PayPalCartService cartService;

    @Resource(name = "paymentService")
    private PayPalPaymentService payPalPaymentService;

    @Resource(name = "payPalPaymentTransactionsService")
    private PaymentTransactionsService paymentTransactionsService;
    
     @Resource(name = "baseSiteService")
    private BaseSiteService baseSiteService;

     @Resource(name = "paymentInfoService")
     private PayPalPaymentInfoService payPalPaymentInfoService;

     @Resource(name = "payPalReplenishmentService")
     private PayPalReplenishmentService payPalReplenishmentService;

    @Secured({ "ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/order/approved")
    public ResponseEntity<String> orderApproved(final HttpServletRequest request,
        final HttpServletResponse response) throws IOException, InvalidCartException {
        if ("POST".equalsIgnoreCase(request.getMethod())) {
            final JsonObject parsedRequest = parseRequestToJson(request);
            final String eventId = parsedRequest.get("id").getAsString();
            final Event event = payPalPaymentService.getEventById(eventId);
            if (ORDER_APPROVED_EVENT_TYPE.equalsIgnoreCase(event.getEventType())) {
                final String orderId = parsedRequest.get(RESOURCE).getAsJsonObject().get("id").getAsString();
                final CartModel cart = cartService.getCartByPayPalOrderId(orderId);
                if (cart != null) {
                    baseSiteService.setCurrentBaseSite(cart.getSite(), true);
                    payPalCheckoutFacade.prepareCartForCheckout(cart);
                    payPalCheckoutFacade.authorizePayment(cart);
                    final OrderData orderData = payPalCheckoutFacade.placeOrderByCart(cart);
                    LOG.info(orderData.getCode() + " was placed due to webhook event: " + event.getId());
                }
            }
        }
        return ResponseEntity.ok("200");
    }

    @Secured({ "ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/capture")
    public ResponseEntity<String> captureTransaction(final HttpServletRequest request,
        final HttpServletResponse response) throws IOException {
        if ("POST".equalsIgnoreCase(request.getMethod())) {
            final JsonObject parsedRequest = parseRequestToJson(request);
            final String eventId = parsedRequest.get("id").getAsString();
            final Event event = payPalPaymentService.getEventById(eventId);
            if (WEBHOOK_RESOURCE_TYPE_CAPTURE.equalsIgnoreCase(event.getResourceType())) {
                final JsonObject resource = parsedRequest.get(RESOURCE).getAsJsonObject();
                final String amount = resource.get("amount").getAsJsonObject().get("value").getAsString();
                final String captureId = resource.get("id").getAsString();
                final String status = resource.get("status").getAsString();
                String authorizationId = null;
                final JsonArray linksArray = resource.get("links").getAsJsonArray();
                for (final JsonElement link : linksArray) {
                    final JsonObject linkJson = link.getAsJsonObject();
                    final String rel = linkJson.get("rel").getAsString();
                    if ("up".equals(rel)) {
                        final String href = linkJson.get("href").getAsString();
                        authorizationId = href
                            .substring(href.lastIndexOf("authorizations/") + "authorizations/".length());
                    }
                }
                paymentTransactionsService.updatePaymentTransaction(authorizationId, captureId, status, amount);
            }
        }
        return ResponseEntity.ok("200");
    }

    @Secured({ "ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/authorization/voided")
    public ResponseEntity<String> paymentAuthorizationVoided(final HttpServletRequest request,
                                                final HttpServletResponse response) throws IOException {
        if ("POST".equalsIgnoreCase(request.getMethod())) {
            final JsonObject parsedRequest = parseRequestToJson(request);
            final String eventId = parsedRequest.get("id").getAsString();
            final Event event = payPalPaymentService.getEventById(eventId);
            if (PAYMENT_AUTHORIZATION_VOIDED_EVENT_TYPE.equalsIgnoreCase(event.getEventType())) {
                final JsonObject resource = parsedRequest.get(RESOURCE).getAsJsonObject();
                final String cancelId = resource.get("id").getAsString();

                paymentTransactionsService.voidPaymentTransaction(cancelId);
            }
        }
        return ResponseEntity.ok("200");
    }

    @Secured({ "ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/billing-agreement/cancelled")
    public ResponseEntity<String> agreementCancelled(final HttpServletRequest request,
        final HttpServletResponse response) throws IOException {
        if ("POST".equalsIgnoreCase(request.getMethod())) {
            final JsonObject parsedRequest = parseRequestToJson(request);
            final String eventId = parsedRequest.get("id").getAsString();
            final Event event = payPalPaymentService.getEventById(eventId);
            if (BILLING_AGREEMENT_EVENT_TYPE.equalsIgnoreCase(event.getEventType())) {
                final JsonObject resource = parsedRequest.get(RESOURCE).getAsJsonObject();
                final String payerId = resource.get("payer").getAsJsonObject().get("payer_info").getAsJsonObject().get("payer_id").getAsString();
                final String billingAgreementId = resource.get("id").getAsString();
                payPalReplenishmentService.disableReplenishmentCronJobByBillingAgreement(billingAgreementId);
                payPalPaymentInfoService.disable(payerId, billingAgreementId);
                LOG.info("Payment method with billing agreement id: " + billingAgreementId +
                    " and PayPal payer id: " + payerId + " has been disabled by payment method webhook");
            }
        }
        return ResponseEntity.ok("200");
    }

    private JsonObject parseRequestToJson(final HttpServletRequest request) throws IOException {
        return JsonParser.parseReader(request.getReader()).getAsJsonObject();
    }
}

