/*

 */
package com.paypal.hybris.controllers;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_ORDER_ID_PLACEHOLDER;

import com.braintree.hybris.data.ws.PayPalBillingAgreementPaymentForNewUserRequestWsDTO;
import com.braintree.hybris.data.ws.PayPalBillingAgreementTokenWsDTO;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalBillingAgreementData;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.data.ws.AddingPaymentMethodAndAddressStatusWsData;
import com.paypal.hybris.data.ws.PayPalConnectLoginData;
import com.paypal.hybris.data.ws.PayPalConnectWsData;
import com.paypal.hybris.facade.facades.PayPalCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import de.hybris.platform.ycommercewebservices.v2.controller.BaseCommerceController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.Map;
import javax.annotation.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}/connect")
@CacheControl(directive = CacheControlDirective.NO_CACHE)
@Api(tags = "PayPal Connect")
public class PayPalConnectController extends BaseCommerceController {

    private static final Logger LOG = LoggerFactory.getLogger(PayPalConnectController.class);

    private static final String ACCESS_TOKEN_KEY = "accessToken";
    private static final String AUTHORIZATION_CODE_KEY = "authorizationCode";
    private static final String DEFAULT_PAYMENT_METHOD = "paypal";

    @Resource(name = "payPalConnectService")
    private PayPalConnectService payPalConnectService;

    @Resource(name = "payPalCustomerAccountService")
    private PayPalCustomerAccountService payPalCustomerAccountService;

    @Resource(name = "payPalRegistrationUserFacade")
    private PayPalRegistrationUserFacade userFacade;

    @Resource(name = "payPalCheckoutFacade")
    private PayPalCheckoutFacade checkoutFacade;

    @Resource(name = "payPalBillingAgreementPaymentRequestValidator")
    private Validator payPalBillingAgreementPaymentRequestValidator;

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/register")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "register endpoint for connect with PayPal", value = "register user by accessToken")
    @ApiBaseSiteIdAndUserIdParam
    public ResponseEntity<String> doRegister(
        @ApiParam(value = "JSON object which contains access token", required = true) @RequestBody final Map<String, String> body)
        throws DuplicateUidException {
        String payerId = userFacade.registerPayPalUser(body.get(ACCESS_TOKEN_KEY));
        return new ResponseEntity<>(payerId, HttpStatus.OK);
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/exchange")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "exchange authorization code", value = "exchange authorization code to access token")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalConnectWsData exchangeAuthorizationCode(
        @ApiParam(value = "JSON object which contains authorization code", required = true) @RequestBody final Map<String, String> body) {
        final PayPalConnectWsData connectWsData = new PayPalConnectWsData();
        final String accessToken = payPalConnectService
            .exchangeAuthorizationCodeToAccessToken(body.get(AUTHORIZATION_CODE_KEY));
        final PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(accessToken);
        connectWsData.setAccessToken(accessToken);
        connectWsData.setIsRegistered(payPalCustomerAccountService.isCustomerWithPayerIdExist(userData.getPayer_id()));
        return connectWsData;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/login")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "login endpoint for connect with PayPal", value = "Generate login data for user by accessToken")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalConnectLoginData login(
        @ApiParam(value = "JSON object which contains accessToken", required = true) @RequestBody final Map<String, String> body) {
        final PayPalConnectUserData userData = payPalConnectService
            .getUserDataByAccessToken(body.get(ACCESS_TOKEN_KEY));
        final CustomerData customer = payPalCustomerAccountService.getCustomerByPayerId(userData.getPayer_id());
        final String password = payPalCustomerAccountService.setTempPassword(userData.getPayer_id());
        final PayPalConnectLoginData loginData = new PayPalConnectLoginData();
        loginData.setLogin(customer.getUid());
        loginData.setPassword(password);
        return loginData;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/afterLogin")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "After login action", value = "clear password field")
    @ApiBaseSiteIdAndUserIdParam
    public HttpStatus afterLogin(
        @ApiParam(value = "JSON object which contains access token", required = true) @RequestBody final Map<String, String> body) {
        final PayPalConnectUserData userData = payPalConnectService
            .getUserDataByAccessToken(body.get(ACCESS_TOKEN_KEY));
        payPalCustomerAccountService.clearTempPassword(userData.getPayer_id());
        return HttpStatus.OK;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/createPaymentDetails")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public AddingPaymentMethodAndAddressStatusWsData createPayPalPaymentAndSaveAddressForNewUser(
        @ApiParam(value = "Request body parameter that contains" +
            " billingAgreementToken and isSavePaymentInfo", required = true) @RequestBody final PayPalBillingAgreementPaymentForNewUserRequestWsDTO payPalBillingAgreementPaymentRequest,
        @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
            allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
        @ApiParam(value = "Payment method type") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_PAYMENT_METHOD) final String paymentMethodType) {
        validate(payPalBillingAgreementPaymentRequest.getBillingAgreementRequest(), "payPalRequest", payPalBillingAgreementPaymentRequestValidator);
        AddingPaymentMethodAndAddressStatusWsData result = new AddingPaymentMethodAndAddressStatusWsData();
        result.setIsAddressAdded(Boolean.FALSE);
        result.setIsPaymentMethodAdded(Boolean.FALSE);
        final String billingAgreementToken = payPalBillingAgreementPaymentRequest.getBillingAgreementRequest()
            .getBillingAgreementTokenId();
        PayPalConnectUserData userData = payPalConnectService
            .getUserDataByAccessToken(payPalBillingAgreementPaymentRequest.getAccessToken());
        try {
            final PayPalBillingAgreementData billingAgreement = checkoutFacade
                .createBillingAgreement(billingAgreementToken);
            checkoutFacade.createPayPalPaymentSubscriptionForNewUser(billingAgreement, PAYPAL_ORDER_ID_PLACEHOLDER,
                    paymentMethodType, userData.getPayer_id());
            result.setIsPaymentMethodAdded(Boolean.TRUE);
        } catch (Exception ex) {
            LOG.error("Error during adding payment method for new customer", ex);
            result.setErrorMessagePaymentMethod(ex.getMessage());
        }
        if(userData.getAddress() != null) {
            try {
                checkoutFacade.addAddressForNewUser(userData.getAddress(), userData.getPayer_id());
                result.setIsAddressAdded(Boolean.TRUE);
            } catch (Exception ex) {
                LOG.error("Error during adding address for new customer", ex);
                result.setErrorMessageAddress(ex.getMessage());
            }
        }
        return result;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/createBillingAgreementToken")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "create billing agreement token", value = "Create new billing agreement tok", notes = "Create and save new billing agreement token")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalBillingAgreementTokenWsDTO createBillingAgreementToken() {
        boolean skipShipping = !checkoutFacade.hasNoDeliveryAddress();
        return GenericBuilder.of(PayPalBillingAgreementTokenWsDTO::new)
            .with(PayPalBillingAgreementTokenWsDTO::setTokenId,
                checkoutFacade.createBillingAgreementToken(skipShipping)).build();
    }

}
