package com.paypal.action;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.orderprocessing.model.OrderProcessModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.action.AbstractAction;
import de.hybris.platform.task.RetryLaterException;
import de.hybris.platform.ticket.enums.CsTicketCategory;
import de.hybris.platform.ticket.enums.CsTicketPriority;
import de.hybris.platform.ticket.events.model.CsCustomerEventModel;
import de.hybris.platform.ticket.model.CsTicketModel;
import de.hybris.platform.ticket.service.TicketBusinessService;
import de.hybris.platform.util.localization.Localization;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;


public class PayPalCheckTransactionReviewStatusAction extends AbstractAction<OrderProcessModel> {

    private static final String ORDER_IN_REVIEW_TITLE = "message.ticket.orderinreview.title";

    private TicketBusinessService ticketBusinessService;

    public enum Transition {
        OK, NOK, WAIT;

        public static Set<String> getStringValues() {
            final Set<String> res = new HashSet<String>();
            for (final Transition transitions : Transition.values()) {
                res.add(transitions.toString());
            }
            return res;
        }
    }

    @Override
    public Set<String> getTransitions() {
        return Transition.getStringValues();
    }

    @Override
    public final String execute(final OrderProcessModel process) throws RetryLaterException, Exception {
        return executeAction(process).toString();
    }

    protected Transition executeAction(final OrderProcessModel process) {
        final OrderModel order = process.getOrder();
        if (order != null) {
            return order.getPaymentTransactions().stream()
                .map(transactionModel -> checkPaymentTransaction(transactionModel, order))
                .filter(transition -> !Transition.OK.equals(transition))
                .findFirst()
                .orElse(Transition.OK);
        }

        return Transition.OK;
    }

    protected Transition checkPaymentTransaction(PaymentTransactionModel transaction, OrderModel orderModel) {
        final List<PaymentTransactionEntryModel> transactionEntries = transaction.getEntries();
        PaymentInfoModel paymentInfo = transaction.getInfo();
        for (int index = transactionEntries.size() - 1; index >= 0; index--) {
            final PaymentTransactionEntryModel entry = transactionEntries.get(index);

            if (isReviewDecision(entry)) {
                if (isReviewAccepted(entry)) {
                    orderModel.setStatus(OrderStatus.PAYMENT_AUTHORIZED);
                    getModelService().save(orderModel);
                    return Transition.OK;
                } else {
                    orderModel.setStatus(OrderStatus.PAYMENT_NOT_AUTHORIZED);
                    getModelService().save(orderModel);
                    return Transition.NOK;
                }
            } else if (isAuthorization(entry)) {
                if (isAuthorizationInReview(entry)) {
                    final String ticketTitle = Localization.getLocalizedString(ORDER_IN_REVIEW_TITLE);
                    final String ticketMessage = Localization.getLocalizedString(ORDER_IN_REVIEW_TITLE,
                        new Object[]{orderModel.getCode()});
                    createTicket(ticketTitle, ticketMessage, orderModel, CsTicketCategory.FRAUD, CsTicketPriority.HIGH);

                    orderModel.setStatus(OrderStatus.SUSPENDED);
                    getModelService().save(orderModel);
                    return Transition.WAIT;
                } else if (paymentInfo instanceof PayPalCreditCardPaymentInfoModel
                    && (PAYPAL_INTENT_AUTHORIZE.equals(((PayPalCreditCardPaymentInfoModel) paymentInfo).getIntent()))) {
                    setOrderStatus(orderModel, OrderStatus.PAYMENT_NOT_CAPTURED);
                    return Transition.WAIT;
                } else {
                    return Transition.OK;
                }
            } else if (isCaptured(entry)) {
                return Transition.OK;
            }

            // Continue onto next entry
        }
        return Transition.OK;
    }

    protected CsTicketModel createTicket(final String subject, final String description, final OrderModel order,
        final CsTicketCategory category, final CsTicketPriority priority) {
        final CsTicketModel newTicket = GenericBuilder.of(CsTicketModel::new)
            .with(CsTicketModel::setHeadline, subject)
            .with(CsTicketModel::setCategory, category)
            .with(CsTicketModel::setPriority, priority)
            .with(CsTicketModel::setOrder, order)
            .with(CsTicketModel::setCustomer, order.getUser())
            .build();

        final CsCustomerEventModel newTicketEvent = new CsCustomerEventModel();
        newTicketEvent.setText(description);

        modelService.attach(newTicket);

        return getTicketBusinessService().createTicket(newTicket, newTicketEvent);
    }

    protected boolean isReviewDecision(final PaymentTransactionEntryModel entry) {
        return PaymentTransactionType.REVIEW_DECISION.equals(entry.getType());
    }

    protected boolean isReviewAccepted(final PaymentTransactionEntryModel entry) {
        return TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());
    }

    protected boolean isAuthorization(final PaymentTransactionEntryModel entry) {
        return PaymentTransactionType.AUTHORIZATION.equals(entry.getType());
    }

    protected boolean isAuthorizationInReview(final PaymentTransactionEntryModel entry) {
        return TransactionStatus.REVIEW.name().equals(entry.getTransactionStatus());
    }

    protected boolean isCaptured(PaymentTransactionEntryModel entry) {
        return PaymentTransactionType.CAPTURE.equals(entry.getType()) || PaymentTransactionType.PARTIAL_CAPTURE
            .equals(entry.getType());
    }

    protected TicketBusinessService getTicketBusinessService() {
        return ticketBusinessService;
    }

    public void setTicketBusinessService(final TicketBusinessService ticketBusinessService) {
        this.ticketBusinessService = ticketBusinessService;
    }
}
