/*

 */
package com.paypal.hybris.facade.facades.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.LOGIN_PAGE_DISABLE_FUNDING_LIST;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_DISABLE_FUNDING_CART_PAGE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_DISABLE_FUNDING_MARK_PAGE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_DISABLE_FUNDING_MINI_CART;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_ORDER_ID_PLACEHOLDER;

import com.paypal.enums.PayPalPageType;
import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.CreditCardPaymentMethodData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalBillingAgreementData;
import com.paypal.hybris.data.PayPalCheckoutData;
import com.paypal.hybris.data.PayPalConfigurationData;
import com.paypal.hybris.data.PayPalConnectAddressData;
import com.paypal.hybris.data.PayPalFullAddressData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalPaymentMethodData;
import com.paypal.hybris.facade.converters.impl.DefaultPayPalAddressDataConverter;
import com.paypal.hybris.facade.facades.PayPalCheckoutFacade;
import de.hybris.platform.acceleratorfacades.order.impl.DefaultAcceleratorCheckoutFacade;
import de.hybris.platform.catalog.CatalogVersionService;
import de.hybris.platform.catalog.model.CatalogVersionModel;
import de.hybris.platform.commercefacades.address.AddressVerificationFacade;
import de.hybris.platform.commercefacades.address.data.AddressVerificationResult;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.address.AddressVerificationDecision;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.media.MediaService;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * This method is a default implementation of the PayPalCheckoutFacade interface
 */
public class DefaultPayPalCheckoutFacade extends DefaultAcceleratorCheckoutFacade implements PayPalCheckoutFacade {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalCheckoutFacade.class);
    private static final String EMPTY = "";
    private static final String SPACE = " ";

    private PayPalCustomerAccountService payPalCustomerAccountService;
    private PayPalPaymentService payPalPaymentService;
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;
    private I18NService i18NService;
    private CommonI18NService commonI18NService;
    private CartService cartService;
    private DefaultPayPalAddressDataConverter defaultPayPalAddressDataConverter;
    private AddressVerificationFacade addressVerificationFacade;
    private UserFacade userFacade;
    private Populator<PayPalAddressDetailsData, AddressData> payPalAddressDataPopulator;
    private Populator<AddressData, PayPalAddressDetailsData> payPalAddressDataReversePopulator;
    private CatalogVersionService catalogVersionService;
    private MediaService mediaService;
    private DefaultPayPalButtonRenderDecisionMakerFacade decisionMakerFacade;
    private PayPalCustomerAccountDao payPalCustomerAccountDao;
    private CustomerAccountService customerAccountService;

    @Override
    public boolean processExpressCheckout(final PayPalOrderDetailsData orderDetails, String paymentMethodType) {
        final AddressData shippingAddress = createAddressDataFromPayPalAddress(orderDetails.getShippingAddress(), true,
            getCurrentUserForCheckout());
        if (isPayPalAddressValid(shippingAddress) || isPickupInStore()) {
            removeDeliveryAddressFromCart();
            setDeliveryAddressForCheckout(shippingAddress);
            setCheapestDeliveryModeForCheckout();
            updatePayPalOrder(orderDetails);
            final CCPaymentInfoData payPalPaymentSubscription = createPayPalPaymentSubscriptionForExpressCheckout(
                orderDetails, paymentMethodType);
            setPaymentDetails(payPalPaymentSubscription.getId());
            return true;
        }
        return false;
    }

    @Override
    public CCPaymentInfoData processBillingAgreementExpressCheckout() {
        if (hasNoDeliveryAddress()) {
            setDefaultDeliveryAddressForCheckout();
        }
        if (hasNoDeliveryMode()) {
            setCheapestDeliveryModeForCheckout();
        }
        if (hasNoPaymentInfo()) {
            setPayPalPaymentInfoIfAvailable();
            updateCheckoutPaymentInfoOrderId(PAYPAL_ORDER_ID_PLACEHOLDER);
        } else {
            prepareCartForCheckout();
            processSessionPaymentInfoForExpressCheckout();
        }
        PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = (PayPalCreditCardPaymentInfoModel) getCart()
            .getPaymentInfo();
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public void updateCheckoutPaymentInfoOrderId(final String orderId) {
        PaymentInfoModel paymentInfo = getCart().getPaymentInfo();
        if (paymentInfo instanceof PayPalCreditCardPaymentInfoModel) {
            ((PayPalCreditCardPaymentInfoModel) paymentInfo).setSubscriptionId(orderId);
            getModelService().save(paymentInfo);
        }
    }

    @Override
    public boolean isPickupInStore() {
        if (getCheckoutCart().getDeliveryOrderGroups() != null) {
            return getCheckoutCart().getDeliveryOrderGroups().isEmpty();
        }
        return false;
    }

    @Override
    public boolean isLocalPaymentFlow() {
        if (getCheckoutCart().getPaymentInfo() != null) {
            return getCheckoutCart().getPaymentInfo().getCardType().equals(CreditCardType.LOCAL_PAYMENT.getCode())
                && defaultPayPalConfigurationService.isCommitEnabled();
        }
        return false;
    }

    @Override
    public PayPalOrderDetailsData getPayPalOrderDetails(final String payPalOrderId) {
        return payPalPaymentService.getOrderDetails(payPalOrderId);
    }

    @Override
    public void setDeliveryAddressForCheckout(AddressData addressData) {
        userFacade.addAddress(addressData);
        setDeliveryAddress(addressData);
    }

    @Override
    public String createBillingAgreementToken(final boolean skipShippingAddress) {
        return payPalPaymentService.createBillingAgreementToken(skipShippingAddress).getToken_id();
    }

    @Override
    public PayPalBillingAgreementData createBillingAgreement(final String billingAgreementToken) {
        return payPalPaymentService.createBillingAgreement(billingAgreementToken);
    }

    @Override
    public String getCurrentSessionUserUid() {
        return getCartService().getSessionCart().getUser().getUid();
    }

    @Override
    public String createPayPalOrder() {
        CartData cartData = getCheckoutCart();
        PayPalAddressDetailsData addressDetailsData = null;
        if (!isPickupInStore()) {
            addressDetailsData = new PayPalAddressDetailsData();
            payPalAddressDataReversePopulator.populate(getCheckoutCart().getDeliveryAddress(), addressDetailsData);
        }
        return getPayPalPaymentService()
            .createOrder(cartData.getTotalPrice().getCurrencyIso(),
                getIntegerPriceValue(cartData), addressDetailsData);
    }

    @Override
    public CCPaymentInfoData createPayPalPaymentSubscriptionForExpressCheckout(
        final PayPalOrderDetailsData orderDetails, String paymentMethodType) {
        CreditCardType paymentType = setCreditCardType(paymentMethodType);
        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
            .with(CCPaymentInfoData::setPayPalOrderId, orderDetails.getOrderId())
            .with(CCPaymentInfoData::setSubscriptionId, orderDetails.getOrderId())
            .with(CCPaymentInfoData::setPayerId, orderDetails.getBuyerId())
            .with(CCPaymentInfoData::setAccountHolderName, orderDetails.getBuyer())
            .with(CCPaymentInfoData::setSaved, false)
            .with(CCPaymentInfoData::setPayerEmail, orderDetails.getShippingAddress().getEmail())
            .with(CCPaymentInfoData::setDefaultPaymentInfo, false)
            .with(CCPaymentInfoData::setCardType, paymentType.name())
            .build();

        final CustomerModel customer = getCurrentUserForCheckout();
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
            .createPayPalPaymentSubscription(customer, paymentInfoData,
                createBillingAddress(orderDetails.getBillingAddress(), customer));
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public CCPaymentInfoData createPayPalPaymentSubscriptionForMarkCheckout(final String orderId,
        CreditCardType paymentType) {
        if ((paymentType).equals(CreditCardType.LOCAL_PAYMENT) && !(defaultPayPalConfigurationService.getBuyerCountry()
            .equals(getCartFacade().getSessionCart().getDeliveryAddress().getCountry().getIsocode()))) {
            paymentType = setCreditCardType(CreditCardType.CARD.getCode());
        }

        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
            .with(CCPaymentInfoData::setPayPalOrderId, orderId)
            .with(CCPaymentInfoData::setSubscriptionId, orderId)
            .with(CCPaymentInfoData::setAccountHolderName, "")
            .with(CCPaymentInfoData::setSaved, false)
            .with(CCPaymentInfoData::setDefaultPaymentInfo, false)
            .with(CCPaymentInfoData::setCardType, paymentType.name())
            .build();

        AddressModel billingAddress = getModelService().create(AddressModel.class);
        if (!isPickupInStore()) {
            getAddressReversePopulator()
                .populate(getCartFacade().getSessionCart().getDeliveryAddress(), billingAddress);
        }
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
            .createPayPalPaymentSubscription(getCurrentUserForCheckout(), paymentInfoData, billingAddress);
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }


    @Override
    public CCPaymentInfoData updatePayPalPaymentSubscription(PayPalOrderDetailsData orderDetailsData,
        CCPaymentInfoData paymentInfoData) {
        paymentInfoData.setPayerId(orderDetailsData.getBuyerId());
        paymentInfoData.setAccountHolderName(orderDetailsData.getBuyer());
        paymentInfoData.setPayerEmail(orderDetailsData.getShippingAddress().getEmail());

        CreditCardPaymentInfoModel paymentInfoModel = getCustomerAccountService()
            .getCreditCardPaymentInfoForCode(getCurrentUserForCheckout(), paymentInfoData.getId());
        final CustomerModel customer = getCurrentUserForCheckout();
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
            .updatePayPalPaymentSubscription(customer, paymentInfoData,
                createBillingAddress(orderDetailsData.getBillingAddress(), customer),
                (PayPalCreditCardPaymentInfoModel) paymentInfoModel);
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public CCPaymentInfoData createPayPalPaymentSubscription(final PayPalBillingAgreementData billingAgreementDetails,
        final String payPalOrderId, boolean savePaymentInfo, final String paymentMethodType) {
        if (hasNoDeliveryAddress()) {
            setDeliveryAddressForCheckout(
                createAddressDataFromPayPalAddress(billingAgreementDetails.getShippingAddress(), true, getCurrentUserForCheckout()));
        }
        if (hasNoDeliveryMode()) {
            setCheapestDeliveryModeForCheckout();
        }
        if (savePaymentInfo) {
            final PayPalCreditCardPaymentInfoModel savedPayPalPayment = isAccountAlreadySaved(
                billingAgreementDetails.getPayerEmail());
            if (savedPayPalPayment != null && !savedPayPalPayment.getDuplicate()) {
                return getCreditCardPaymentInfoConverter().convert(savedPayPalPayment);
            }
        }
        boolean shouldBeDefault = userFacade.getCCPaymentInfos(true).isEmpty();
        CreditCardType paymentType = setCreditCardType(paymentMethodType);
        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
            .with(CCPaymentInfoData::setPayerId, billingAgreementDetails.getPayerId())
            .with(CCPaymentInfoData::setAccountHolderName, billingAgreementDetails.getPayerName())
            .with(CCPaymentInfoData::setPayerEmail, billingAgreementDetails.getPayerEmail())
            .with(CCPaymentInfoData::setBillingAgreementId, billingAgreementDetails.getId())
            .with(CCPaymentInfoData::setSubscriptionId, payPalOrderId)
            .with(CCPaymentInfoData::setSaved, savePaymentInfo)
            .with(CCPaymentInfoData::setDefaultPaymentInfo, shouldBeDefault && savePaymentInfo)
            .with(CCPaymentInfoData::setCardType, paymentType.name())
            .build();
        final CustomerModel customer = getCurrentUserForCheckout();
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
            .createPayPalPaymentSubscription(customer, paymentInfoData,
                createBillingAddress(billingAgreementDetails.getBillingAddress(), customer));
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public CCPaymentInfoData createPayPalPaymentSubscriptionForNewUser(final PayPalBillingAgreementData billingAgreementDetails,
        final String payPalOrderId, final String paymentMethodType, final String payerId) {
        CreditCardType paymentType = setCreditCardType(paymentMethodType);
        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
            .with(CCPaymentInfoData::setPayerId, billingAgreementDetails.getPayerId())
            .with(CCPaymentInfoData::setAccountHolderName, billingAgreementDetails.getPayerName())
            .with(CCPaymentInfoData::setPayerEmail, billingAgreementDetails.getPayerEmail())
            .with(CCPaymentInfoData::setBillingAgreementId, billingAgreementDetails.getId())
            .with(CCPaymentInfoData::setSubscriptionId, payPalOrderId)
            .with(CCPaymentInfoData::setSaved, true)
            .with(CCPaymentInfoData::setDefaultPaymentInfo, true)
            .with(CCPaymentInfoData::setCardType, paymentType.name())
            .build();
        final CustomerModel customer = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
            .createPayPalPaymentSubscription(customer, paymentInfoData,
                createBillingAddress(billingAgreementDetails.getBillingAddress(), customer));
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public void addAddressForNewUser(final PayPalConnectAddressData addressData, final String payerId) {
        final AddressModel newAddress = getModelService().create(AddressModel.class);

        final CountryModel countryModel = getCommonI18NService().getCountry(addressData.getCountry());

        final CustomerModel customer = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);

        if (StringUtils.isNotBlank(addressData.getRegion())) {
            final String regionIsoCode = String.format("%s-%s", countryModel.getIsocode(), addressData.getRegion());
            final RegionModel regionModel = getCommonI18NService().getRegion(countryModel, regionIsoCode);
            newAddress.setRegion(regionModel);
            getModelService().save(regionModel);
        }

        List<String> fullName = Arrays.asList(customer.getName().split(SPACE));
        newAddress.setFirstname(fullName.stream().findFirst().orElse(EMPTY));
        newAddress.setLastname(fullName.stream().skip(1).collect(Collectors.joining(" ")));
        newAddress.setEmail(customer.getContactEmail());
        newAddress.setCountry(countryModel);
        newAddress.setTown(addressData.getLocality());
        newAddress.setLine1(addressData.getStreet_address());
        newAddress.setPostalcode(addressData.getPostal_code());
        newAddress.setShippingAddress(true);

        customer.setDefaultShipmentAddress(newAddress);
        customer.setDefaultPaymentAddress(newAddress);

        getCustomerAccountService().saveAddressEntry(customer, newAddress);
    }

    @Override
    public boolean authorizePayment(final CartModel cartModel) {
        final CreditCardPaymentInfoModel creditCardPaymentInfoModel = cartModel == null ? null
            : (CreditCardPaymentInfoModel) cartModel.getPaymentInfo();
        if (creditCardPaymentInfoModel != null
            && StringUtils.isNotBlank(creditCardPaymentInfoModel.getSubscriptionId())) {
            final CommerceCheckoutParameter parameter = createCommerceCheckoutParameter(cartModel, true);
            parameter.setPaymentProvider(getPaymentProvider());
            final PaymentTransactionEntryModel paymentTransactionEntryModel = getCommerceCheckoutService()
                .authorizePayment(parameter);

            return paymentTransactionEntryModel != null
                && (TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())
                || TransactionStatus.REVIEW.name().equals(paymentTransactionEntryModel.getTransactionStatus()));
        }
        return false;
    }

    @Override
    public OrderData placeOrderByCart(CartModel cartModel) throws InvalidCartException {
        if (cartModel != null) {
            beforePlaceOrder(cartModel);
            final OrderModel orderModel = placeOrder(cartModel);
            afterPlaceOrder(cartModel, orderModel);
            if (orderModel != null) {
                return getOrderConverter().convert(orderModel);
            }
        }
        return null;
    }

    @Override
    protected void afterPlaceOrder(CartModel cartModel, OrderModel orderModel) {
        if (orderModel != null) {
            getModelService().remove(cartModel);
            getModelService().refresh(orderModel);
        }
    }

    @Override
    public void prepareCartForCheckout(final CartModel cartModel) {
        if (cartModel != null) {
            getCommerceCheckoutService().calculateCart(createCommerceCheckoutParameter(cartModel, true));
        }
    }

    @Override
    public PayPalCheckoutData getPayPalCheckoutData(final String pageType) {
        PayPalCheckoutData payPalCheckoutData = generateConfigurationData(pageType);

        payPalCheckoutData.setPayPalPaymentMethod(generatePayPalMethodData());
        payPalCheckoutData.setCreditCardPaymentMethod(generateCreditCardPaymentMethod());
        if (PayPalPageType.MARK.toString().equalsIgnoreCase(pageType)
            && cartService.hasSessionCart()
            && cartService.getSessionCart().getDeliveryAddress() != null) {

            final PayPalFullAddressData payPalAddress = defaultPayPalAddressDataConverter
                .convert(cartService.getSessionCart().getDeliveryAddress());
            payPalCheckoutData.setShippingAddressOverride(payPalAddress);
        }

        return payPalCheckoutData;
    }

    @Override
    public CreditCardType setCreditCardType(String fundingSource) {
        CreditCardType paymentType;
        if (CreditCardType.PAYPAL.name().equalsIgnoreCase(fundingSource)
            || CreditCardType.PAYLATER.name().equalsIgnoreCase(fundingSource)
            || CreditCardType.CREDIT.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.PAYPAL;
        } else if (CreditCardType.VENMO.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.VENMO;
        } else if (CreditCardType.CARD.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.CARD;
        } else {
            paymentType = CreditCardType.LOCAL_PAYMENT;
        }

        return paymentType;

    }

    private boolean isPayPalAddressValid(final AddressData addressData) {
        final AddressVerificationResult<AddressVerificationDecision> verificationResult = addressVerificationFacade
            .verifyAddressData(addressData);
        return AddressVerificationDecision.ACCEPT.getDecisionString()
            .equals(verificationResult.getDecision().getDecisionString());
    }

    private void removeDeliveryAddressFromCart() {
        final AddressData previousSelectedAddress = getCheckoutCart().getDeliveryAddress();
        if (Objects.nonNull(previousSelectedAddress) && !previousSelectedAddress.isVisibleInAddressBook()) {
            getUserFacade().removeAddress(previousSelectedAddress);
        }
        removeDeliveryAddress();
    }

    private void processSessionPaymentInfoForExpressCheckout() {
        final PaymentInfoModel sessionPaymentInfo = getCart().getPaymentInfo();
        if (sessionPaymentInfo instanceof PayPalCreditCardPaymentInfoModel) {
            final String payPalOrderId = ((PayPalCreditCardPaymentInfoModel) sessionPaymentInfo).getSubscriptionId();
            if (PAYPAL_ORDER_ID_PLACEHOLDER.equals(payPalOrderId)) {
                return;
            }
            final PayPalOrderDetailsData payPalOrderDetails = getPayPalOrderDetails(payPalOrderId);
            if (payPalOrderDetails != null) {
                updatePayPalOrder(payPalOrderDetails);
            } else if (setPaymentInfoIfAvailable()) {
                updateCheckoutPaymentInfoOrderId(PAYPAL_ORDER_ID_PLACEHOLDER);
            } else {
                removePaymentInfoFromCart();
            }
        }
    }

    private boolean removePaymentInfoFromCart() {
        final CartModel cartModel = getCart();
        if (cartModel != null) {
            cartModel.setPaymentInfo(null);
            getModelService().save(cartModel);
            getModelService().refresh(cartModel);
        }
        return true;
    }

    private void updatePayPalOrder(final PayPalOrderDetailsData orderDetailsData) {
        final BigDecimal payPalOrderAmountDouble = BigDecimal.valueOf(Double.parseDouble(orderDetailsData.getAmount()));
        final BigDecimal cartTotalPrice = BigDecimal.valueOf(getCart().getTotalPrice());
        if (!cartTotalPrice.equals(payPalOrderAmountDouble)) {
            getPayPalPaymentService()
                .updateOrderAmountDetails(orderDetailsData.getOrderId(), getCart().getCurrency().getIsocode(),
                    String.valueOf(cartTotalPrice));
        }
    }

    private boolean setPayPalPaymentInfoIfAvailable() {
        boolean result = false;
        final CartModel cartModel = getCart();
        if (cartModel != null && cartModel.getPaymentInfo() == null) {
            final CustomerModel currentUser = getCurrentUserForCheckout();
            if (cartModel.getUser().equals(currentUser)) {
                final PaymentInfoModel defaultPaymentInfo = currentUser.getDefaultPaymentInfo();
                if (decisionMakerFacade.isPaymentMethodSupportStaticImage(defaultPaymentInfo)) {
                    final CommerceCheckoutParameter parameter = createCommerceCheckoutParameter(cartModel, true);
                    parameter.setPaymentInfo(defaultPaymentInfo);

                    result = getCommerceCheckoutService().setPaymentInfo(parameter);
                } else {
                    final List<CreditCardPaymentInfoModel> creditCardPaymentInfos = getCustomerAccountService()
                        .getCreditCardPaymentInfos(currentUser, true);
                    if (creditCardPaymentInfos != null && !creditCardPaymentInfos.isEmpty()) {
                        final CommerceCheckoutParameter parameter = createCommerceCheckoutParameter(cartModel, true);
                        Optional<CreditCardPaymentInfoModel> lastSaved = creditCardPaymentInfos.stream()
                            .filter(decisionMakerFacade::isPaymentMethodSupportStaticImage)
                            .reduce((first, second) -> second);

                        lastSaved.ifPresent(parameter::setPaymentInfo);
                        result = lastSaved.isPresent() && getCommerceCheckoutService().setPaymentInfo(parameter);
                    }
                }
            }
        }
        return result;
    }

    private String getIntegerPriceValue(final CartData cartData) {
        try {
            String priceValue;
            if (isNonDecimalCurrency(cartData)) {
                priceValue = cartData.getTotalPriceWithTax().getValue().toBigInteger().toString();
            } else {
                priceValue = cartData.getTotalPriceWithTax().getValue().toString();
            }
            return priceValue;
        } catch (ClassCastException e) {
            LOG.error("[PayPal Checkout Facade Impl] Errors during currency converting: " + e.getMessage(),
                e);
            throw new ClassCastException(e.getMessage());
        }
    }

    private Boolean isNonDecimalCurrency(final CartData cartData) {
        String paypalNonDecimalCurrency = defaultPayPalConfigurationService.getNonDecimalCurrency();
        return paypalNonDecimalCurrency.contains(cartData.getTotalPrice().getCurrencyIso());
    }

    private PayPalCreditCardPaymentInfoModel isAccountAlreadySaved(final String payerEmail) {
        Collection<PaymentInfoModel> paymentInfos = getCurrentUserForCheckout().getPaymentInfos();
        for (final PaymentInfoModel payment : paymentInfos) {
            if (payment instanceof PayPalCreditCardPaymentInfoModel) {
                final String savedPaymentInfoEmail = ((PayPalCreditCardPaymentInfoModel) payment).getPayerEmail();
                if (payment.isSaved() && savedPaymentInfoEmail.equals(payerEmail)) {
                    return (PayPalCreditCardPaymentInfoModel) payment;
                }
            }
        }
        return null;
    }

    private AddressModel createBillingAddress(final PayPalAddressDetailsData addressDetailsData, final CustomerModel customer) {
        final AddressModel billingAddress = getModelService().create(AddressModel.class);
        final AddressData billingAddressData = createAddressDataFromPayPalAddress(addressDetailsData, false, customer);
        getAddressReversePopulator().populate(billingAddressData, billingAddress);
        billingAddress.setEmail(billingAddressData.getEmail());
        return billingAddress;
    }

    private AddressData createAddressDataFromPayPalAddress(final PayPalAddressDetailsData payPalAddressDetailsData,
        boolean isShipping, final CustomerModel customer) {
        final AddressData addressData = new AddressData();
        getPayPalAddressDataPopulator().populate(payPalAddressDetailsData, addressData);

        final TitleModel title = customer.getTitle();
        if (title != null) {
            addressData.setTitle(title.getName());
            addressData.setTitleCode(title.getCode());
        }
        boolean hasNoDefaultAddress = userFacade.getDefaultAddress() == null && isShipping;
        addressData.setDefaultAddress(hasNoDefaultAddress);
        addressData.setVisibleInAddressBook(hasNoDefaultAddress);
        addressData.setShippingAddress(isShipping);
        addressData.setBillingAddress(!isShipping);
        return addressData;
    }

    private PayPalPaymentMethodData generatePayPalMethodData() {
        PayPalPaymentMethodData paymentMethodData = new PayPalPaymentMethodData();
        paymentMethodData.setBillingAgreementDescription(
            (String) getDefaultPayPalConfigurationService().getCustomBillingAgreementDescription());
        paymentMethodData.setPayPalExpressEnabled(getDefaultPayPalConfigurationService().isExpressCheckoutEnabled());
        return paymentMethodData;
    }

    private PayPalCheckoutData generateConfigurationData(final String pageType) {
        final PayPalCheckoutData payPalCheckoutData = new PayPalCheckoutData();
        PayPalConfigurationData payPalConfigurationData = new PayPalConfigurationData();
        payPalConfigurationData.setIntent(getDefaultPayPalConfigurationService().getPayPalIntent());
        payPalConfigurationData.setCommit(getDefaultPayPalConfigurationService().isCommitEnabled());
        payPalConfigurationData.setPayPalSdkUrl(getDefaultPayPalConfigurationService().getPayPalSdkJsUrl());
        payPalConfigurationData
            .setPartnerAttributionId(getDefaultPayPalConfigurationService().getPayPalPartnerAttributionId());
        payPalConfigurationData.setEnvironmentType(getDefaultPayPalConfigurationService().getEnvironmentType());
        payPalConfigurationData.setCurrency(getI18NService().getCurrentJavaCurrency().getCurrencyCode());
        payPalConfigurationData.setStaticImageUrl(getImageUrlByCode("/images/cart/paypal-express-checkout.png"));
        payPalConfigurationData.setVenmoIconUrl(getImageUrlByCode("venmoLogo"));
        payPalConfigurationData.setIsVenmoEnabled(getDefaultPayPalConfigurationService().isVenmoEnabled());
        payPalConfigurationData
            .setIsPayLaterEnabledForNonUsCountries(getDefaultPayPalConfigurationService().isPayLaterEnabledForNonUsCountries());
        payPalConfigurationData.setIsBillingAgreementEnabled(defaultPayPalConfigurationService.isBillingAgreementEnabled());
        payPalConfigurationData.setNonDecimalCurrency(defaultPayPalConfigurationService.getNonDecimalCurrency());
        switch (PayPalPageType.valueOf(pageType)) {
            case CART:
                setPayPalConfigDataIfCart(payPalConfigurationData);
                break;
            case MINICART:
                payPalConfigurationData.setDisableFunding(decisionMakerFacade.getDisableFunding(PAYPAL_DISABLE_FUNDING_MINI_CART));
                break;
            case MARK:
                setPayPalConfigDataIfMark(payPalConfigurationData);
                break;
            case LOGIN:
                setPayPalConfigDataIfLogin(payPalConfigurationData);
                break;
            default:
                //Default action is not provided in this switch-case block
                break;
        }
        payPalConfigurationData.setBuyerCountry(getDefaultPayPalConfigurationService().getBuyerCountry());

        if (cartService.hasSessionCart()) {
            CartModel cart = cartService.getSessionCart();
            payPalCheckoutData.setEnableShippingAddress(!isOnlyPickUp());
            payPalCheckoutData.setAmount(cart.getTotalPrice());
        } else {
            payPalCheckoutData.setEnableShippingAddress(false);
        }
        payPalCheckoutData.setConfigurationData(payPalConfigurationData);
        payPalCheckoutData.getConfigurationData().setClient_id(getDefaultPayPalConfigurationService().getClientID());

        return payPalCheckoutData;
    }

    private void setPayPalConfigDataIfCart(PayPalConfigurationData payPalConfigurationData) {
        payPalConfigurationData.setIsBillingAgreementFlowEnabled(
            decisionMakerFacade.isExpressCheckoutBillingAgreementFlow());
        payPalConfigurationData.setIsExpressCheckoutStaticImageFlow(
            decisionMakerFacade.isExpressCheckoutStaticImageFlow(getCartFacade().getSessionCart()));
        payPalConfigurationData.setDisableFunding(decisionMakerFacade.getDisableFunding(PAYPAL_DISABLE_FUNDING_CART_PAGE));
        payPalConfigurationData.setBuyerCountry(getDefaultPayPalConfigurationService().getBuyerCountry());

    }

    private void setPayPalConfigDataIfMark(PayPalConfigurationData payPalConfigurationData) {
        payPalConfigurationData
            .setIsBillingAgreementFlowEnabled(decisionMakerFacade.isMarkCheckoutBillingAgreementFlow());
        payPalConfigurationData.setDisableFunding(decisionMakerFacade.getDisableFunding(PAYPAL_DISABLE_FUNDING_MARK_PAGE));
        payPalConfigurationData.setLocale(getDefaultPayPalConfigurationService().getLPMLocale());
        payPalConfigurationData.setBuyerCountry(getDefaultPayPalConfigurationService().getBuyerCountry());

    }

    private void setPayPalConfigDataIfLogin(PayPalConfigurationData payPalConfigurationData) {
        payPalConfigurationData.setIsBillingAgreementFlowEnabled(Boolean.TRUE);
        payPalConfigurationData.setLocale(getDefaultPayPalConfigurationService().getLPMLocale());
        payPalConfigurationData.setBuyerCountry(getDefaultPayPalConfigurationService().getBuyerCountry());
        payPalConfigurationData.setDisableFunding(LOGIN_PAGE_DISABLE_FUNDING_LIST);
        payPalConfigurationData.setPayPalConnectScript(defaultPayPalConfigurationService.getPayPalConnectScript());
        payPalConfigurationData.setPayPalConnectScopes(defaultPayPalConfigurationService.getPayPalConnectScopes());
        payPalConfigurationData.setPayPalConnectResponseType(defaultPayPalConfigurationService.getPayPalConnectResponseType());
        payPalConfigurationData.setPayPalConnectReturnUrl(defaultPayPalConfigurationService.getPayPalConnectReturnUrl());
    }

    private String getImageUrlByCode(String code) {
        String imgUrl = null;
        if (StringUtils.isNotEmpty(code)) {
            for (final CatalogVersionModel catalogVersionModel : catalogVersionService.getSessionCatalogVersions()) {
                try {
                    final MediaModel media = mediaService.getMedia(catalogVersionModel, code);
                    imgUrl = Optional.ofNullable(media).map(MediaModel::getDownloadURL).orElse(null);
                } catch (final UnknownIdentifierException ignore) {
                    // Ignore this exception
                }
            }
        }
        return imgUrl;
    }

    private CreditCardPaymentMethodData generateCreditCardPaymentMethod() {
        CreditCardPaymentMethodData paymentMethodData = new CreditCardPaymentMethodData();
        paymentMethodData.setPaymentsImagesUrl(defaultPayPalConfigurationService.getAcceptedPaymentMethodImages());
        return paymentMethodData;
    }

    private Boolean isOnlyPickUp() {
        return getCheckoutCart().getDeliveryOrderGroups().isEmpty();
    }

    public PayPalCustomerAccountService getPayPalCustomerAccountService() {
        return payPalCustomerAccountService;
    }


    public I18NService getI18NService() {
        return i18NService;
    }

    public void setI18NService(I18NService i18NService) {
        this.i18NService = i18NService;

    }

    public void setPayPalCustomerAccountService(PayPalCustomerAccountService payPalCustomerAccountService) {
        this.payPalCustomerAccountService = payPalCustomerAccountService;
    }

    public AddressVerificationFacade getAddressVerificationFacade() {
        return addressVerificationFacade;
    }

    public void setAddressVerificationFacade(AddressVerificationFacade addressVerificationFacade) {
        this.addressVerificationFacade = addressVerificationFacade;
    }


    public PayPalPaymentService getPayPalPaymentService() {
        return payPalPaymentService;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public DefaultPayPalConfigurationService getDefaultPayPalConfigurationService() {
        return defaultPayPalConfigurationService;
    }

    public void setDefaultPayPalConfigurationService(DefaultPayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public UserFacade getUserFacade() {
        return userFacade;
    }

    public void setUserFacade(UserFacade userFacade) {
        this.userFacade = userFacade;
    }

    public Populator<PayPalAddressDetailsData, AddressData> getPayPalAddressDataPopulator() {
        return payPalAddressDataPopulator;
    }

    public void setPayPalAddressDataPopulator(
        Populator<PayPalAddressDetailsData, AddressData> payPalAddressDataPopulator) {
        this.payPalAddressDataPopulator = payPalAddressDataPopulator;
    }

    public Populator<AddressData, PayPalAddressDetailsData> getPayPalAddressDataReversePopulator() {
        return payPalAddressDataReversePopulator;
    }

    public void setPayPalAddressDataReversePopulator(
        Populator<AddressData, PayPalAddressDetailsData> payPalAddressDataReversePopulator) {
        this.payPalAddressDataReversePopulator = payPalAddressDataReversePopulator;
    }

    public void setDefaultPayPalAddressDataConverter(DefaultPayPalAddressDataConverter defaultPayPalAddressDataConverter) {
        this.defaultPayPalAddressDataConverter = defaultPayPalAddressDataConverter;
    }

    public DefaultPayPalAddressDataConverter getDefaultPayPalAddressDataConverter() {
        return defaultPayPalAddressDataConverter;
    }

    public MediaService getMediaService() {
        return mediaService;
    }

    public void setMediaService(MediaService mediaService) {
        this.mediaService = mediaService;
    }

    public CatalogVersionService getCatalogVersionService() {
        return catalogVersionService;
    }

    public void setCatalogVersionService(CatalogVersionService catalogVersionService) {
        this.catalogVersionService = catalogVersionService;
    }

    /**
     * @return the cartService
     */
    @Override
    public CartService getCartService() {
        return cartService;
    }

    /**
     * @param cartService the cartService to set
     */
    @Override
    public void setCartService(final CartService cartService) {
        this.cartService = cartService;
    }

    public DefaultPayPalButtonRenderDecisionMakerFacade getDecisionMakerFacade() {
        return decisionMakerFacade;
    }

    public void setDecisionMakerFacade(DefaultPayPalButtonRenderDecisionMakerFacade decisionMakerFacade) {
        this.decisionMakerFacade = decisionMakerFacade;
    }

    public PayPalCustomerAccountDao getPayPalCustomerAccountDao() {
        return payPalCustomerAccountDao;
    }

    public void setPayPalCustomerAccountDao(PayPalCustomerAccountDao payPalCustomerAccountDao) {
        this.payPalCustomerAccountDao = payPalCustomerAccountDao;
    }

    @Override
    public CustomerAccountService getCustomerAccountService() {
        return customerAccountService;
    }

    @Override
    public void setCustomerAccountService(CustomerAccountService customerAccountService) {
        this.customerAccountService = customerAccountService;
    }

    @Override
    public CommonI18NService getCommonI18NService() {
        return commonI18NService;
    }

    @Override
    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }
}
