/**
 *
 */
package com.paypal.hybris.core.service.impl;

import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import com.paypal.hybris.core.dao.PaymentTransactionsDao;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordercancel.OrderCancelEntry;
import de.hybris.platform.ordercancel.OrderCancelRecordsHandler;
import de.hybris.platform.ordercancel.OrderCancelRequest;
import de.hybris.platform.ordercancel.OrderCancelResponse;
import de.hybris.platform.ordercancel.OrderCancelResponse.ResponseStatus;
import de.hybris.platform.ordercancel.OrderStatusChangeStrategy;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.ArrayList;

import java.util.List;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultPaymentTransactionsServiceTest {

    private static final String TRANSACTION_ID = "transactionId";
    private static final String CAPTURE_ID = "captureId";
    private static final String AUTHORIZATION_ID = "authorizationId";
    private static final String STATUS = "status";
    private static final String AMOUNT = "4";

    @Mock
    PaymentTransactionsDao transactionsDao;

    @Mock
    ModelService modelService;

    @Mock
    OrderCancelRecordsHandler orderCancelRecordsHandler;

    @Mock
    OrderStatusChangeStrategy completeCancelStatusChangeStrategy;

    @Mock
    DefaultPayPalPaymentService paypalPaymentService;

    @InjectMocks
    private DefaultPaymentTransactionsService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void getTransactionEntryByIdNotNull() {

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(entryModel);

        assertNotNull(unit.getTransactionEntryById(TRANSACTION_ID));
    }

    @Test
    public void getTransactionEntryByIdIsNull() {

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(null);

        assertNull(unit.getTransactionEntryById(TRANSACTION_ID));
    }

    @Test
    public void updatePaymentTransactionCaptureTransactionEntryNull() {

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);

        when(transactionsDao.getTransactionEntryById(CAPTURE_ID)).thenReturn(null);

        unit.updatePaymentTransaction(AUTHORIZATION_ID, CAPTURE_ID, STATUS, AMOUNT);

        verify(modelService, never()).save(entryModel);

    }

    @Test
    public void updatePaymentTransactionCaptureTransactionEntryNotNullTypeCapture() {

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);

        when(transactionsDao.getTransactionEntryById(CAPTURE_ID)).thenReturn(entryModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);

        unit.updatePaymentTransaction(AUTHORIZATION_ID, CAPTURE_ID, STATUS, AMOUNT);

        verify(modelService).save(entryModel);

    }

    @Test
    public void updatePaymentTransactionCaptureTransactionEntryNotNullTypeAuthorization() {

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        PaymentTransactionModel transactionModel = Mockito.mock(PaymentTransactionModel.class);
        OrderModel order = Mockito.mock(OrderModel.class);

        when(transactionsDao.getTransactionEntryById(AUTHORIZATION_ID)).thenReturn(entryModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getPlannedAmount()).thenReturn(BigDecimal.TEN);
        when(transactionModel.getOrder()).thenReturn(order);

        unit.updatePaymentTransaction(AUTHORIZATION_ID, CAPTURE_ID, STATUS, AMOUNT);

        verify(modelService).save(order);

    }

    @Test
    public void voidPaymentTransactionWithTransactionEntryNull() {

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(null);

        unit.voidPaymentTransaction(TRANSACTION_ID);

        verify(modelService, never()).save(entryModel);

    }

    @Test
    public void voidPaymentTransactionWithAuthorizeTypeTransactionEntryNotNull() {

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        PaymentTransactionModel transactionModel = Mockito.mock(PaymentTransactionModel.class);
        PaymentTransactionEntryModel newTransactionEntryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        OrderModel order = Mockito.mock(OrderModel.class);
        OrderEntryModel orderEntry = Mockito.mock(OrderEntryModel.class);
        ArrayList<AbstractOrderEntryModel> orderEntries = new ArrayList<AbstractOrderEntryModel>();
        orderEntries.add(orderEntry);
        ArrayList<PaymentTransactionModel> paymentTransactions = new ArrayList<>();
        paymentTransactions.add(transactionModel);
        ArrayList<PaymentTransactionEntryModel> paymentTransactionEntries = new ArrayList<>();
        paymentTransactionEntries.add(entryModel);

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(entryModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(order);
        when(order.getEntries()).thenReturn(orderEntries);
        when(orderEntry.getOrder()).thenReturn(order);
        when(order.getPaymentTransactions()).thenReturn(paymentTransactions);
        when(transactionModel.getEntries()).thenReturn(paymentTransactionEntries);

        when(paypalPaymentService.getCancelableTransactions(order)).thenReturn(paymentTransactionEntries);
        when(paypalPaymentService.getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CANCEL)).thenReturn("newCode");
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(newTransactionEntryModel);
        when(paypalPaymentService.isOrderCanceled(order)).thenReturn(true);

        unit.voidPaymentTransaction(TRANSACTION_ID);

        verify(modelService).save(entryModel);
        verify(modelService).save(order);
        verify(modelService).save(newTransactionEntryModel);
        verify(modelService).refresh(order);

    }

    @Test
    public void makeInternalResponsePartialStatus() {

        OrderCancelRequest request = Mockito.mock(OrderCancelRequest.class);
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        OrderCancelEntry orderCancelEntry = Mockito.mock(OrderCancelEntry.class);
        AbstractOrderEntryModel abstractOrderEntryModel = Mockito.mock(AbstractOrderEntryModel.class);
        List<OrderCancelEntry> orderCancelEntryList = new ArrayList<>();
        orderCancelEntryList.add(orderCancelEntry);

        when(request.isPartialCancel()).thenReturn(true);
        when(request.getOrder()).thenReturn(orderModel);
        when(request.getEntriesToCancel()).thenReturn(orderCancelEntryList);
        when(orderCancelEntry.getOrderEntry()).thenReturn(abstractOrderEntryModel);
        when(abstractOrderEntryModel.getOrder()).thenReturn(orderModel);

        OrderCancelResponse response = unit.makeInternalResponse(request, true, (String) null);

        OrderCancelResponse.ResponseStatus actual = response.getResponseStatus();
        OrderCancelResponse.ResponseStatus expected = OrderCancelResponse.ResponseStatus.partial;

        assertEquals(expected, actual);
    }

    @Test
    public void makeInternalResponseFullStatus() {

        OrderCancelRequest request = Mockito.mock(OrderCancelRequest.class);
        OrderModel orderModel = Mockito.mock(OrderModel.class);

        when(request.isPartialCancel()).thenReturn(false);
        when(request.getOrder()).thenReturn(orderModel);

        OrderCancelResponse response = unit.makeInternalResponse(request, true, (String) null);

        OrderCancelResponse.ResponseStatus actual = response.getResponseStatus();
        OrderCancelResponse.ResponseStatus expected = OrderCancelResponse.ResponseStatus.full;

        assertEquals(expected, actual);
    }

    @Test
    public void makeInternalResponseErrorStatus() {

        OrderCancelRequest request = Mockito.mock(OrderCancelRequest.class);
        OrderModel orderModel = Mockito.mock(OrderModel.class);

        when(request.isPartialCancel()).thenReturn(false);
        when(request.getOrder()).thenReturn(orderModel);

        OrderCancelResponse response = unit.makeInternalResponse(request, false, (String) null);

        OrderCancelResponse.ResponseStatus actual = response.getResponseStatus();
        OrderCancelResponse.ResponseStatus expected = ResponseStatus.error;

        assertEquals(expected, actual);
    }

    @Test
    public void buildCancelRequestWithNullOrder() {

        assertNull(unit.buildCancelRequest(null));
    }

    @Test
    public void buildCancelRequestWithNotNullOrder() {

        OrderModel orderModel = Mockito.mock(OrderModel.class);
        OrderCancelEntry orderCancelEntry = Mockito.mock(OrderCancelEntry.class);
        AbstractOrderEntryModel abstractOrderEntryModel = Mockito.mock(AbstractOrderEntryModel.class);
        List<AbstractOrderEntryModel> abstractOrderEntryModelList = new ArrayList<>();
        abstractOrderEntryModelList.add(abstractOrderEntryModel);
        List<OrderCancelEntry> orderCancelEntryList = new ArrayList<>();
        orderCancelEntryList.add(orderCancelEntry);

        when(orderCancelEntry.getOrderEntry()).thenReturn(abstractOrderEntryModel);
        when(abstractOrderEntryModel.getOrder()).thenReturn(orderModel);
        when(orderModel.getEntries()).thenReturn(abstractOrderEntryModelList);

        assertNotNull(unit.buildCancelRequest(orderModel));
    }

    @Test
    public void createOrderCancelEntryListNotEmpty() {

        List<OrderCancelEntry> orderCancelEntryList = new ArrayList<>();
        AbstractOrderEntryModel entry = Mockito.mock(AbstractOrderEntryModel.class);

        unit.createOrderCancelEntry(orderCancelEntryList, entry);

        assertTrue(orderCancelEntryList.size() > 0);
    }
}
