/*

 */
package com.paypal.hybris.core.service.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.hybris.core.commands.impl.DefaultPayPalCaptureIntentCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateBillingAgreementCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateBillingAgreementTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateOrderCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetEventCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetOrderDetailsCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalReauthorizationRequestCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalUpdateOrderAmountCommand;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Currency;
import java.util.List;
import java.util.UUID;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;


public class DefaultPayPalPaymentServiceTest {

    private static final String CURRENCY = "USD";

    @Mock
    private CommandFactoryRegistry commandFactoryRegistry;

    @Mock
    private ModelService modelService;

    @Mock
    private CommonI18NService commonI18NService;

    @Spy
    @InjectMocks
    private DefaultPayPalPaymentService unit = new DefaultPayPalPaymentService();

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void createStubAuthorize() {
        final String merchantTransactionCode = UUID.randomUUID().toString();
        final String paymentProvider = UUID.randomUUID().toString();
        final String subscriptionId = UUID.randomUUID().toString();
        final String newEntryCode = UUID.randomUUID().toString();
        final BigDecimal amount = BigDecimal.valueOf(Math.random());
        Currency currency = Currency.getInstance(CURRENCY);
        PayPalCreditCardPaymentInfoModel paymentInfo = Mockito.mock(PayPalCreditCardPaymentInfoModel.class);
        PaymentTransactionModel transactionModel = Mockito.mock(PaymentTransactionModel.class);
        CurrencyModel currencyModel = Mockito.mock(CurrencyModel.class);

        when(modelService.create(PaymentTransactionModel.class)).thenReturn(transactionModel);
        when(commonI18NService.getCurrency(CURRENCY)).thenReturn(currencyModel);
        when(paymentInfo.getSubscriptionId()).thenReturn(subscriptionId);
        doReturn(newEntryCode).when(unit).getNewPaymentTransactionEntryCode(any(), any());
        when(transactionModel.getPlannedAmount()).thenReturn(amount);
        when(transactionModel.getCurrency()).thenReturn(currencyModel);

        PaymentTransactionEntryModel actualTransactionModel = unit.createStubAuthorize(merchantTransactionCode, amount, currency, paymentProvider, paymentInfo);

        assertEquals(transactionModel.getPlannedAmount(), actualTransactionModel.getAmount());
        assertEquals(transactionModel.getCurrency(), actualTransactionModel.getCurrency());
    }

    @Test
    public void shouldInvokeReauthorizeCommand() throws CommandNotSupportedException {
        final SubscriptionAuthorizationRequest expectedRequest = mock(SubscriptionAuthorizationRequest.class);
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalReauthorizationRequestCommand command = mock(DefaultPayPalReauthorizationRequestCommand.class);
        final AuthorizationResult result = mock(AuthorizationResult.class);
        when(expectedRequest.getPaymentProvider()).thenReturn(PAYPAL_PROVIDER_NAME);
        when(commandFactory.getPaymentProvider()).thenReturn(PAYPAL_PROVIDER_NAME);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalReauthorizationRequestCommand.class)).thenReturn(command);
        when(command.perform(expectedRequest)).thenReturn(result);

        unit.reauthorize(expectedRequest);

        verify(command).perform(expectedRequest);
    }

    @Test
    public void shouldInvokeCreateOrderCommand() throws CommandNotSupportedException {
        final String currency = UUID.randomUUID().toString();
        final String orderAmount = UUID.randomUUID().toString();
        final PayPalAddressDetailsData detailsData = mock(PayPalAddressDetailsData.class);

        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalCreateOrderCommand command = mock(DefaultPayPalCreateOrderCommand.class);

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalCreateOrderCommand.class)).thenReturn(command);

        unit.createOrder(currency, orderAmount, detailsData);

        verify(command).perform(any(PayPalOrderRequestData.class));
    }

    @Test
    public void shouldInvokeCaptureIntentCommand() throws CommandNotSupportedException {
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalCaptureIntentCommand command = mock(DefaultPayPalCaptureIntentCommand.class);
        final CaptureRequest expectedRequest = mock(CaptureRequest.class);

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalCaptureIntentCommand.class)).thenReturn(command);

        unit.doCaptureIntentRequest(expectedRequest);

        verify(command).perform(expectedRequest);
    }

    @Test
    public void shouldInvokeGetEventCommand() throws CommandNotSupportedException {
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalGetEventCommand command = mock(DefaultPayPalGetEventCommand.class);
        final String eventId = UUID.randomUUID().toString();

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalGetEventCommand.class)).thenReturn(command);

        unit.getEventById(eventId);

        verify(command).perform(eventId);
    }

    @Test
    public void shouldSetCancelStatus() {
        List<PaymentTransactionEntryModel> cancelableTransactions = new ArrayList<>();
        OrderModel order = new OrderModel();

        doReturn(cancelableTransactions).when(unit).getCancelableTransactions(order);
        doReturn(true).when(unit).isOrderCanceled(order);

        unit.doCancel(order);

        assertEquals(OrderStatus.CANCELLED, order.getStatus());
    }

    @Test
    public void shouldGetOrderDetails() throws CommandNotSupportedException {
        final String expectedRequest = UUID.randomUUID().toString();
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalGetOrderDetailsCommand command = mock(DefaultPayPalGetOrderDetailsCommand.class);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalGetOrderDetailsCommand.class)).thenReturn(command);

        unit.getOrderDetails(expectedRequest);

        verify(command).perform(expectedRequest);
    }

    @Test
    public void shouldCreateBillingAgreement() throws CommandNotSupportedException {
        final String expectedRequest = UUID.randomUUID().toString();
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalCreateBillingAgreementCommand command = mock(DefaultPayPalCreateBillingAgreementCommand.class);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalCreateBillingAgreementCommand.class)).thenReturn(command);

        unit.createBillingAgreement(expectedRequest);

        verify(command).perform(expectedRequest);
    }

    @Test
    public void shouldInvokeCommandUpdateOrderAmountDetails() throws CommandNotSupportedException {
        final PayPalOrderRequestData requestData = mock(PayPalOrderRequestData.class);
        when(requestData.getOrderId()).thenReturn(UUID.randomUUID().toString());
        when(requestData.getCurrency()).thenReturn(UUID.randomUUID().toString());
        when(requestData.getAmount()).thenReturn(UUID.randomUUID().toString());
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalUpdateOrderAmountCommand command = mock(DefaultPayPalUpdateOrderAmountCommand.class);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalUpdateOrderAmountCommand.class)).thenReturn(command);

        unit.updateOrderAmountDetails(requestData.getOrderId(), requestData.getCurrency(), requestData.getAmount());

        verify(command).perform(any());
    }

    @Test
    public void shouldInvokeCommandCreateBillingAgreementToken() throws CommandNotSupportedException {
        final CommandFactory commandFactory = mock(CommandFactory.class);
        final DefaultPayPalCreateBillingAgreementTokenCommand command = mock(DefaultPayPalCreateBillingAgreementTokenCommand.class);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalCreateBillingAgreementTokenCommand.class)).thenReturn(command);

        unit.createBillingAgreementToken(true);

        verify(command).perform(true);
    }
}
