/*

 */
package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.impl.DefaultCustomerAccountService;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.UUID;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;
import static org.junit.Assert.*;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.mock;

@UnitTest
public class DefaultPayPalCustomerAccountServiceTest extends DefaultCustomerAccountService {

    @Mock
    private CustomerModel customerModel;

    @Mock
    private CCPaymentInfoData ccPaymentInfoData;

    @Mock
    private ModelService modelService;

    @Mock
    private AddressModel addressModel;

    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Mock
    private CCPaymentInfoData ccRefreshPaymentInfoData;

    @Mock
    private PayPalCustomerAccountDao payPalCustomerAccountDao;

    @Mock
    private Converter<CustomerModel, CustomerData> customerConverter;

    @Mock
    private CustomerData customerData;

    @InjectMocks
    private DefaultPayPalCustomerAccountService unit = new DefaultPayPalCustomerAccountService();

    private PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel = new PayPalCreditCardPaymentInfoModel();


    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(customerModel.getUid()).thenReturn(UUID.randomUUID().toString());
        setUpPaymentData(this.ccPaymentInfoData);
        setUpPaymentData(this.ccRefreshPaymentInfoData);
    }

    private CCPaymentInfoData setUpPaymentData(CCPaymentInfoData paymentInfoData) {
        when(paymentInfoData.getPayerEmail()).thenReturn(UUID.randomUUID().toString());
        when(paymentInfoData.getAccountHolderName()).thenReturn(UUID.randomUUID().toString());
        when(paymentInfoData.getPayerId()).thenReturn(UUID.randomUUID().toString());
        when(paymentInfoData.getBillingAgreementId()).thenReturn(UUID.randomUUID().toString());
        when(paymentInfoData.getSubscriptionId()).thenReturn(UUID.randomUUID().toString());
        when(paymentInfoData.getCardType()).thenReturn(CreditCardType.PAYPAL.getCode());
        return paymentInfoData;
    }

    @Before
    public void setUpPayPalInfoModel() {
        payPalPaymentInfoModel.setPayerEmail(ccPaymentInfoData.getPayerEmail());
        payPalPaymentInfoModel.setPayPalOrderId(ccPaymentInfoData.getPayPalOrderId());
        payPalPaymentInfoModel.setPayerId(ccPaymentInfoData.getPayerId());
        payPalPaymentInfoModel.setBillingAgreementId(ccPaymentInfoData.getBillingAgreementId());
        payPalPaymentInfoModel.setSubscriptionId(ccPaymentInfoData.getSubscriptionId());
        payPalPaymentInfoModel.setCcOwner(ccPaymentInfoData.getAccountHolderName());
        payPalPaymentInfoModel.setType(CreditCardType.PAYPAL);
    }

    @Test
    public void shouldCorrectPopulatePayPalPaymentSubscription() {
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class))
            .thenReturn(new PayPalCreditCardPaymentInfoModel());
        when(modelService.clone(addressModel)).thenReturn(addressModel);

        final PayPalCreditCardPaymentInfoModel actualInfo = unit
            .createPayPalPaymentSubscription(customerModel, ccPaymentInfoData, addressModel);

        assertEquals(payPalPaymentInfoModel.getPayerEmail(), actualInfo.getPayerEmail());
        assertEquals(payPalPaymentInfoModel.getPayPalOrderId(), actualInfo.getPayPalOrderId());
        assertEquals(payPalPaymentInfoModel.getPayerId(), actualInfo.getPayerId());
        assertEquals(payPalPaymentInfoModel.getBillingAgreementId(), actualInfo.getBillingAgreementId());
        assertEquals(payPalPaymentInfoModel.getSubscriptionId(), actualInfo.getSubscriptionId());
        assertEquals(payPalPaymentInfoModel.getCcOwner(), actualInfo.getCcOwner());
        assertEquals(payPalPaymentInfoModel.getType(), CreditCardType.PAYPAL);
    }

    @Test
    public void shouldCorrectSavePayPalPaymentSubscription() {
        final PayPalCreditCardPaymentInfoModel mockInfo = mock(PayPalCreditCardPaymentInfoModel.class);
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class)).thenReturn(mockInfo);
        when(modelService.clone(addressModel)).thenReturn(addressModel);

        unit.createPayPalPaymentSubscription(customerModel, ccPaymentInfoData, addressModel);

        verify(modelService).saveAll(addressModel, mockInfo);
        verify(modelService, times(2)).refresh(customerModel);
    }

    @Test
    public void shouldCorrectUpdatePayPAlPaymentSubscription() {

        final PayPalCreditCardPaymentInfoModel updatedPayPalPaymentInfoModel = unit
            .updatePayPalPaymentSubscription(customerModel, ccRefreshPaymentInfoData, addressModel, payPalPaymentInfoModel);

        verify(modelService).saveAll(addressModel, payPalPaymentInfoModel);
        verify(modelService).saveAll(customerModel, payPalPaymentInfoModel);
        verify(modelService, times(2)).refresh(customerModel);

        assertNotEquals(updatedPayPalPaymentInfoModel.getPayerEmail(), ccPaymentInfoData.getPayerEmail());
        assertNotEquals(updatedPayPalPaymentInfoModel.getCcOwner(), ccPaymentInfoData.getAccountHolderName());
        assertNotEquals(updatedPayPalPaymentInfoModel.getPayerId(), ccPaymentInfoData.getPayerId());
        assertNotEquals(updatedPayPalPaymentInfoModel.getBillingAgreementId(), ccPaymentInfoData.getBillingAgreementId());
        assertNotEquals(updatedPayPalPaymentInfoModel.getSubscriptionId(), ccPaymentInfoData.getSubscriptionId());

        assertEquals(updatedPayPalPaymentInfoModel.getPayerEmail(), ccRefreshPaymentInfoData.getPayerEmail());
        assertEquals(updatedPayPalPaymentInfoModel.getCcOwner(), ccRefreshPaymentInfoData.getAccountHolderName());
        assertEquals(updatedPayPalPaymentInfoModel.getPayerId(), ccRefreshPaymentInfoData.getPayerId());
        assertEquals(updatedPayPalPaymentInfoModel.getBillingAgreementId(), ccRefreshPaymentInfoData.getBillingAgreementId());
        assertEquals(updatedPayPalPaymentInfoModel.getSubscriptionId(), ccRefreshPaymentInfoData.getSubscriptionId());
        assertEquals(updatedPayPalPaymentInfoModel.getType().getCode(), ccRefreshPaymentInfoData.getCardType());
    }

    @Test
    public void shouldGetCustomerByPayerId() {
        final String payerId = UUID.randomUUID().toString();
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId)).thenReturn(customerModel);
        when(customerConverter.convert(customerModel)).thenReturn(customerData);

        assertEquals(customerData, unit.getCustomerByPayerId(payerId));
    }

    @Test
    public void shouldNotGetCustomerByPayerId() {
        final String payerId = UUID.randomUUID().toString();
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId)).thenReturn(null);

        assertFalse(unit.isCustomerWithPayerIdExist(payerId));
    }
}
