/**
 *
 */
package com.paypal.hybris.core.service.impl;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.hybris.core.service.PayPalPaymentService;
import de.hybris.platform.commercefacades.product.data.PriceData;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commerceservices.order.CommercePaymentAuthorizationStrategy;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.SessionService;
import java.math.BigDecimal;
import java.util.Map;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultPayPalCommerceCheckoutServiceTest {

    private static final String DEFAULT_FLOW = "DEFAULT_FLOW";
    private static final String ORDER_PROCESS_FLOW = "ORDER_PROCESS_FLOW";
    private static final String TEST_INTENT = "TestIntent";
    private static final String PAYPAL_ORDER_PROCESS_FLOW = "ORDER_PROCESS_FLOW";
    private static final String PAYPAL_ORDER_ID_PLACEHOLDER = "ORDER_ID_PLACEHOLDER";
    private static final String PAYPAL_NONDECIMAL_CURRENCY = "JPY,HUF,TWD";
    private static final String CURRENCY = "USD";

    @Mock
    private ModelService modelService;

    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Mock
    private SessionService sessionService;

    @Mock
    private Map<String, CommercePaymentAuthorizationStrategy> strategyMap;

    @Mock
    private Converter<CartModel, CartData> cartConverter;

    @Mock
    private PayPalPaymentService payPalPaymentService;

    @InjectMocks
    DefaultPayPalCommerceCheckoutService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(TEST_INTENT);
    }

    @Test
    public void authorizePaymentAuthorizationAmountNull() {

        final CommerceCheckoutParameter parameter = Mockito.mock(CommerceCheckoutParameter.class);
        final CartModel cartModel = Mockito.mock(CartModel.class);
        final PaymentInfoModel paymentInfoModel = Mockito.mock(PaymentInfoModel.class);
        final CommercePaymentAuthorizationStrategy strategy = Mockito.mock(CommercePaymentAuthorizationStrategy.class);

        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(null);
        when(strategyMap.get(DEFAULT_FLOW)).thenReturn(strategy);

        unit.authorizePayment(parameter);

        verify(parameter).setAuthorizationAmount(any());
        verify(modelService, never()).refresh(cartModel);
        verify(strategy).authorizePaymentAmount(parameter);

    }

    @Test
    public void authorizePaymentAuthorizationAmountNotNull() {

        final CommerceCheckoutParameter parameter = Mockito.mock(CommerceCheckoutParameter.class);
        final BigDecimal authorizationAmount = Mockito.mock(BigDecimal.class);
        final CartModel cartModel = Mockito.mock(CartModel.class);
        final PayPalCreditCardPaymentInfoModel paymentInfoModel = Mockito.mock(PayPalCreditCardPaymentInfoModel.class);
        final CommercePaymentAuthorizationStrategy strategy = Mockito.mock(CommercePaymentAuthorizationStrategy.class);

        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(authorizationAmount);
        when(strategyMap.get(ORDER_PROCESS_FLOW)).thenReturn(strategy);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn("any");
        when(paymentInfoModel.getIntent()).thenReturn("any");

        unit.authorizePayment(parameter);

        verify(parameter, never()).setAuthorizationAmount(any());
        verify(strategy).authorizePaymentAmount(parameter);

    }

    @Test
    public void isPayPalOrderShouldBeCreatedTrue() {
        final CommerceCheckoutParameter parameter = Mockito.mock(CommerceCheckoutParameter.class);
        final CartModel cartModel = Mockito.mock(CartModel.class);
        final CartData cartData = Mockito.mock(CartData.class);
        final PriceData priceData = Mockito.mock(PriceData.class);
        final PayPalCreditCardPaymentInfoModel paymentInfoModel = Mockito.mock(PayPalCreditCardPaymentInfoModel.class);
        final CommercePaymentAuthorizationStrategy strategy = Mockito.mock(CommercePaymentAuthorizationStrategy.class);

        when(cartData.getTotalPrice()).thenReturn(priceData);
        when(cartData.getTotalPriceWithTax()).thenReturn(priceData);
        when(priceData.getValue()).thenReturn(BigDecimal.ONE);
        when(cartConverter.convert(cartModel)).thenReturn(cartData);
        when(paymentInfoModel.getIntent()).thenReturn(TEST_INTENT);
        when(paymentInfoModel.getSubscriptionId()).thenReturn(PAYPAL_ORDER_ID_PLACEHOLDER);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(null);
        when(strategyMap.get(PAYPAL_ORDER_PROCESS_FLOW)).thenReturn(strategy);
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(PAYPAL_NONDECIMAL_CURRENCY);
        when(priceData.getCurrencyIso()).thenReturn(CURRENCY);

        unit.authorizePayment(parameter);

        verify(paymentInfoModel).setIntent(TEST_INTENT);
        verify(modelService).refresh(cartModel);
    }
}
