package com.paypal.hybris.core.service.impl;

import com.google.common.base.Predicate;
import com.paypal.hybris.core.dao.PaymentTransactionsDao;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.security.PrincipalModel;
import de.hybris.platform.ordercancel.*;
import de.hybris.platform.ordercancel.exceptions.OrderCancelRecordsHandlerException;
import de.hybris.platform.ordercancel.model.OrderCancelRecordEntryModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.log4j.Logger;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * This class is a default implementation of the PaymentTransactionsService interface
 */
public class DefaultPaymentTransactionsService implements PaymentTransactionsService {

    private static final String REVIEW_DECISION = "_ReviewDecision";
    private static final String TRANSACTION_STATUS_VOIDED = "CANCELED";

    private static final Logger LOG = Logger.getLogger(DefaultPaymentTransactionsService.class);

    private PaymentTransactionsDao transactionsDao;
    private ModelService modelService;
    private BusinessProcessService businessProcessService;
    private DefaultPayPalPaymentService paypalPaymentService;
    private OrderCancelRecordsHandler orderCancelRecordsHandler;
    private OrderStatusChangeStrategy completeCancelStatusChangeStrategy;

    @Override
    public PaymentTransactionEntryModel getTransactionEntryById(String transactionId) {
        return transactionsDao.getTransactionEntryById(transactionId);
    }

    /**
     * This method is used to update payment transaction
     * @param authorizationId authorizationId
     * @param captureId captureId
     * @param status status
     * @param amount amount
     */
    public void updatePaymentTransaction(String authorizationId, String captureId, String status, String amount) {
        final PaymentTransactionEntryModel captureTransactionEntry = getTransactionEntryById(captureId);
        if (captureTransactionEntry != null && PaymentTransactionType.CAPTURE
            .equals(captureTransactionEntry.getType())) {
            captureTransactionEntry.setTransactionStatus(status);
            captureTransactionEntry.setTransactionStatusDetails(status);
            modelService.save(captureTransactionEntry);
        } else {
            final PaymentTransactionEntryModel authorizationEntry = getTransactionEntryById(authorizationId);
            if (authorizationEntry != null && PaymentTransactionType.AUTHORIZATION
                .equals(authorizationEntry.getType())) {
                PaymentTransactionModel paymentTransaction = authorizationEntry.getPaymentTransaction();
                PaymentTransactionEntryModel newCaptureTransactionEntry = createCaptureTransactionEntry(captureId,
                    status,
                    paymentTransaction, amount);
                modelService.attach(newCaptureTransactionEntry);
                modelService.save(newCaptureTransactionEntry);
                executeWebhookPaymentCaptureOperation((OrderModel) paymentTransaction.getOrder());
            }
        }
    }

    /**
     * This method is used to void payment transaction
     * @param cancelId cancel id
     */
    public void voidPaymentTransaction(String cancelId) {
        final PaymentTransactionEntryModel authorizationEntry = getTransactionEntryById(cancelId);
        if (authorizationEntry != null && PaymentTransactionType.AUTHORIZATION
                .equals(authorizationEntry.getType())) {
            PaymentTransactionModel paymentTransaction = authorizationEntry.getPaymentTransaction();

            try {
                OrderCancelRequest orderCancelRequest = buildCancelRequest((OrderModel) paymentTransaction.getOrder());
                OrderCancelRecordEntryModel result = this.orderCancelRecordsHandler.createRecordEntry(orderCancelRequest, new PrincipalModel());

                OrderModel order = orderCancelRequest.getOrder();
                this.doCancel(order);
                this.modelService.refresh(order);

                if (!OrderUtils.hasLivingEntries(order) && this.completeCancelStatusChangeStrategy != null) {
                    this.completeCancelStatusChangeStrategy
                            .changeOrderStatusAfterCancelOperation(result, true);
                }

                this.orderCancelRecordsHandler
                        .updateRecordEntry(this.makeInternalResponse(orderCancelRequest, true, (String) null));
            }
            catch (OrderCancelRecordsHandlerException e){
                LOG.info(e.getMessage(), e);
            }
        }

    }

    protected OrderCancelResponse makeInternalResponse(OrderCancelRequest request, boolean success, String message) {
        return request.isPartialCancel() ?
                new OrderCancelResponse(request.getOrder(), request.getEntriesToCancel(),
                        success ? OrderCancelResponse.ResponseStatus.partial : OrderCancelResponse.ResponseStatus.error,
                        message) :
                new OrderCancelResponse(request.getOrder(),
                        success ? OrderCancelResponse.ResponseStatus.full : OrderCancelResponse.ResponseStatus.error, message);
    }

    protected OrderCancelRequest buildCancelRequest(OrderModel order) {
        if (order != null) {
            List<OrderCancelEntry> orderCancelEntries = new ArrayList<>();
            order.getEntries()
                    .forEach(entry -> createOrderCancelEntry(orderCancelEntries, entry));
            OrderCancelRequest orderCancelRequest = new OrderCancelRequest(order, orderCancelEntries);
            return orderCancelRequest;

        }
        return null;
    }

    protected void createOrderCancelEntry(List<OrderCancelEntry> orderCancelEntries, Object entry) {
        AbstractOrderEntryModel orderEntryToCancel = (AbstractOrderEntryModel) entry;
        OrderCancelEntry orderCancelEntry = new OrderCancelEntry(orderEntryToCancel);
        orderCancelEntries.add(orderCancelEntry);
    }

    private void doCancel(OrderModel order) {
        List<PaymentTransactionEntryModel> cancelableTransactions = this.paypalPaymentService.getCancelableTransactions(order);
        for (PaymentTransactionEntryModel cancelableTransaction : cancelableTransactions) {
            cancel(cancelableTransaction);
        }
        if (this.paypalPaymentService.isOrderCanceled(order)) {
            order.setStatus(OrderStatus.CANCELLED);
            this.modelService.save(order);
            LOG.info("Order: " + order.getCode() + " has been cancelled by webhook!\n Order status: " + order.getStatus());
        }
    }

    private void cancel(PaymentTransactionEntryModel transactionEntry) {
        PaymentTransactionType transactionType = PaymentTransactionType.CANCEL;
        String newEntryCode = this.paypalPaymentService.getNewPaymentTransactionEntryCode(transactionEntry.getPaymentTransaction(), transactionType);
        PaymentTransactionEntryModel entry = this.modelService.create(PaymentTransactionEntryModel.class);
        if (transactionEntry.getCurrency() != null) {
            entry.setCurrency(transactionEntry.getCurrency());
        }
        entry.setType(transactionType);
        entry.setTime(new Date());
        entry.setPaymentTransaction(transactionEntry.getPaymentTransaction());
        entry.setRequestId(transactionEntry.getRequestId());
        entry.setAmount(transactionEntry.getAmount());
        entry.setRequestToken(transactionEntry.getRequestToken());
        entry.setTransactionStatus(TransactionStatus.ACCEPTED.toString());
        entry.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.toString());
        entry.setCode(newEntryCode);

        transactionEntry.setTransactionStatus(TRANSACTION_STATUS_VOIDED);
        this.modelService.save(transactionEntry);
        this.modelService.save(entry);
    }

    private PaymentTransactionEntryModel createCaptureTransactionEntry(String captureId, String status,
        PaymentTransactionModel paymentTransaction, String requestAmount) {
        final BigDecimal amount = new BigDecimal(requestAmount);
        boolean isFullCapture = paymentTransaction.getPlannedAmount().compareTo(amount) <= 0;
        return GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setAmount, amount)
            .with(PaymentTransactionEntryModel::setCurrency, paymentTransaction.getCurrency())
            .with(PaymentTransactionEntryModel::setType,
                isFullCapture ? PaymentTransactionType.CAPTURE : PaymentTransactionType.PARTIAL_CAPTURE)
            .with(PaymentTransactionEntryModel::setRequestId, paymentTransaction.getRequestId())
            .with(PaymentTransactionEntryModel::setTime, new Date())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, paymentTransaction)
            .with(PaymentTransactionEntryModel::setTransactionStatus, status)
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails, status)
            .with(PaymentTransactionEntryModel::setCode, captureId)
            .build();
    }

    protected void executeWebhookPaymentCaptureOperation(OrderModel order) {
        order.getOrderProcess().stream()
            .filter(process -> process.getCode().startsWith(order.getStore().getSubmitOrderProcessCode()))
            .forEach(filteredProcess -> businessProcessService
                .triggerEvent(filteredProcess.getOrder().getCode() + REVIEW_DECISION));
        LOG.info(String.format("Payment Capture Webhook -> Completed. %s triggered.", "ReviewDecision"));
        if (isOrderFullyCaptured(order)) {
            order.setStatus(OrderStatus.PAYMENT_CAPTURED);
        }
        modelService.save(order);
    }

    private boolean isOrderFullyCaptured(OrderModel orderModel) {
        modelService.refresh(orderModel);
        Predicate<PaymentTransactionEntryModel> captureEntries = entry ->
            (PaymentTransactionType.PARTIAL_CAPTURE.equals(entry.getType()) || PaymentTransactionType.CAPTURE
                .equals(entry.getType()))
                && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());

        BigDecimal result = orderModel.getPaymentTransactions().stream()
            .flatMap(transaction -> transaction.getEntries().stream())
            .filter(captureEntries).map(PaymentTransactionEntryModel::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);

        return result.compareTo(getAuthorizedAmount(orderModel)) >= 0;
    }

    private BigDecimal getAuthorizedAmount(OrderModel orderModel) {
        modelService.refresh(orderModel);
        Predicate<PaymentTransactionEntryModel> authorizationEntries = (entry -> (PaymentTransactionType.AUTHORIZATION
            .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()));

        return orderModel.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(authorizationEntries).map(PaymentTransactionEntryModel::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    public void setTransactionsDao(PaymentTransactionsDao transactionsDao) {
        this.transactionsDao = transactionsDao;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void setBusinessProcessService(BusinessProcessService businessProcessService) {
        this.businessProcessService = businessProcessService;
    }

    public void setPaypalPaymentService(DefaultPayPalPaymentService paypalPaymentService) {
        this.paypalPaymentService = paypalPaymentService;
    }

    public OrderStatusChangeStrategy getCompleteCancelStatusChangeStrategy() {
        return this.completeCancelStatusChangeStrategy;
    }

    public void setCompleteCancelStatusChangeStrategy(OrderStatusChangeStrategy completeCancelStatusChangeStrategy) {
        this.completeCancelStatusChangeStrategy = completeCancelStatusChangeStrategy;
    }

    public OrderCancelRecordsHandler getOrderCancelRecordsHandler() {
        return this.orderCancelRecordsHandler;
    }

    public void setOrderCancelRecordsHandler(OrderCancelRecordsHandler orderCancelRecordsHandler) {
        this.orderCancelRecordsHandler = orderCancelRecordsHandler;
    }

}
