package com.paypal.hybris.core.service.impl;

import com.paypal.api.payments.Event;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateBillingAgreementTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetEventCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetOrderDetailsCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalReauthorizationRequestCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalUpdateOrderAmountCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateBillingAgreementCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateOrderCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCaptureIntentCommand;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalBillingAgreementData;
import com.paypal.hybris.data.PayPalBillingAgreementTokenData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.request.FollowOnRefundRequest;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.commands.result.RefundResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.impl.DefaultPaymentServiceImpl;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import org.apache.log4j.Logger;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;

/**
 * This class is a default implementation of the PayPalPaymentService interface
 */
public class DefaultPayPalPaymentService extends DefaultPaymentServiceImpl implements PayPalPaymentService {

    private static final String TRANSACTION_STATUS_VOIDED = "CANCELED";
    private static final String REFUND_ERROR = "Refund is not available";
    private static final String UPDATING_ORDER_ERROR = "[PayPal Payment Service] Errors during updating order details: ";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalPaymentService.class);

    private CommandFactoryRegistry commandFactoryRegistry;

    @Override
    public PaymentTransactionEntryModel refund(final PaymentTransactionEntryModel selectedEntry,
        final BigDecimal amount) {
        final PaymentTransactionModel transaction = selectedEntry.getPaymentTransaction();
        if (!(transaction.getInfo() instanceof PayPalCreditCardPaymentInfoModel)) {
            return super.refundFollowOn(transaction, amount);
        } else {
            if (isAvailableRefund(selectedEntry)) {
                PaymentTransactionType transactionType = PaymentTransactionType.REFUND_PARTIAL;
                String newEntryCode = this
                    .getNewPaymentTransactionEntryCode(selectedEntry.getPaymentTransaction(), transactionType);
                final Currency currency = Currency.getInstance(selectedEntry.getCurrency().getIsocode());
                final FollowOnRefundRequest followOnRefundRequest = new FollowOnRefundRequest(newEntryCode,
                    selectedEntry.getRequestId(),
                    selectedEntry.getRequestToken(), currency, amount,
                    selectedEntry.getPaymentTransaction().getPaymentProvider());

                PaymentTransactionEntryModel entry = null;

                try {
                    final RefundResult refundResult = this.getCardPaymentService()
                        .refundFollowOn(followOnRefundRequest);
                    entry = createRefundTransaction(refundResult, newEntryCode, transactionType, transaction,
                        selectedEntry.getRequestId());
                } catch (Exception e) {
                    String message = "Exception, message : " + e.getMessage();
                    LOG.error(message, e);
                    entry = createFailedRefundTransaction(selectedEntry, amount, transactionType, newEntryCode,
                        e.getMessage());
                }

                this.getModelService().attach(entry);
                this.getModelService().save(entry);
                this.getModelService().refresh(transaction);
                return entry;

            }
            throw new AdapterException(REFUND_ERROR);
        }
    }

    @Override
    public PaymentTransactionEntryModel capture(PaymentTransactionModel transaction) {
        PaymentTransactionEntryModel auth = null;
        Iterator var4 = transaction.getEntries().iterator();

        while (var4.hasNext()) {
            PaymentTransactionEntryModel pte = (PaymentTransactionEntryModel) var4.next();
            if (PaymentTransactionType.AUTHORIZATION.equals(pte.getType())) {
                auth = pte;
                break;
            }
        }

        if (auth == null) {
            throw new AdapterException("Could not capture without authorization");
        } else {
            PaymentTransactionType transactionType = PaymentTransactionType.CAPTURE;
            PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) transaction.getInfo();
            String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, transactionType);
            CaptureResult result;
            if (PAYPAL_INTENT_CAPTURE.equalsIgnoreCase(paymentInfo.getIntent())) {
                result = doCaptureIntentRequest(new CaptureRequest(newEntryCode, transaction.getRequestId(),
                    paymentInfo.getBillingAgreementId(), Currency.getInstance(auth.getCurrency().getIsocode()),
                    auth.getAmount(),
                    transaction.getPaymentProvider(), auth.getSubscriptionID()));
            } else {
                result = this.getCardPaymentService()
                    .capture(new CaptureRequest(newEntryCode, transaction.getRequestId(),
                        paymentInfo.getBillingAgreementId(), Currency.getInstance(auth.getCurrency().getIsocode()),
                        auth.getAmount(),
                        transaction.getPaymentProvider(), auth.getSubscriptionID()));
            }

            final CurrencyModel currency = result.getCurrency() != null
                ? this.getCommonI18NService().getCurrency(result.getCurrency().getCurrencyCode()) : null;

            PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setAmount, result.getTotalAmount())
                .with(PaymentTransactionEntryModel::setCurrency, currency)
                .with(PaymentTransactionEntryModel::setType, transactionType)
                .with(PaymentTransactionEntryModel::setRequestId, result.getRequestId())
                .with(PaymentTransactionEntryModel::setRequestToken, result.getRequestToken())
                .with(PaymentTransactionEntryModel::setTime,
                    result.getRequestTime() == null ? new Date() : result.getRequestTime())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
                .with(PaymentTransactionEntryModel::setTransactionStatus, result.getTransactionStatus().toString())
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                    result.getTransactionStatusDetails().toString())
                .with(PaymentTransactionEntryModel::setCode, newEntryCode)
                .build();

            this.getModelService().attach(entry);
            this.getModelService().save(entry);
            return entry;
        }
    }

    @Override
    public PaymentTransactionEntryModel createStubAuthorize(final String merchantTransactionCode,
        final BigDecimal amount, final Currency currency, final String paymentProvider,
        PayPalCreditCardPaymentInfoModel paymentInfo) {
        final PaymentTransactionModel transaction = this.getModelService().create(PaymentTransactionModel.class);
        transaction.setCode(merchantTransactionCode);
        transaction.setPlannedAmount(amount);
        transaction.setCurrency(this.getCommonI18NService().getCurrency(currency.getCurrencyCode()));
        transaction.setPaymentProvider(paymentProvider);
        transaction.setRequestId(paymentInfo.getSubscriptionId());
        return stubAuthorize(transaction);
    }

    @Override
    public AuthorizationResult reauthorize(SubscriptionAuthorizationRequest request) {
        try {
            CommandFactory commandFactory;
            commandFactory = this.commandFactoryRegistry.getFactory(request.getPaymentProvider());
            DefaultPayPalReauthorizationRequestCommand command = commandFactory
                .createCommand(DefaultPayPalReauthorizationRequestCommand.class);
            AuthorizationResult result = command.perform(request);
            result.setPaymentProvider(commandFactory.getPaymentProvider());
            return result;
        } catch (CommandNotSupportedException var5) {
            throw new AdapterException(var5.getMessage(), var5);
        }
    }

    @Override
    public PayPalOrderDetailsData getOrderDetails(final String payPalOrderId) {
        try {
            LOG.debug("GetOrderDetails PayPal orderID: " + payPalOrderId);
            final DefaultPayPalGetOrderDetailsCommand command = getCommandFactory()
                .createCommand(DefaultPayPalGetOrderDetailsCommand.class);

            return command.perform(payPalOrderId);
        } catch (final Exception exception) {
            LOG.error("[PayPal Payment Service] Errors during getting order details: " + exception.getMessage(),
                exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public Boolean updateOrderAmountDetails(final String orderId, final String currency, final String orderAmount) {
        final PayPalOrderRequestData orderRequestData = GenericBuilder.of(PayPalOrderRequestData::new)
            .with(PayPalOrderRequestData::setAmount, orderAmount)
            .with(PayPalOrderRequestData::setCurrency, currency)
            .with(PayPalOrderRequestData::setOrderId, orderId)
            .build();
        try {
            LOG.debug(
                "UpdateOrderDetails orderId: " + orderId + ", orderAmount: " + orderAmount + ", currency: " + currency);
            final DefaultPayPalUpdateOrderAmountCommand command = getCommandFactory()
                .createCommand(DefaultPayPalUpdateOrderAmountCommand.class);

            return command.perform(orderRequestData);
        } catch (final Exception exception) {
            LOG.error(UPDATING_ORDER_ERROR + exception.getMessage(),
                exception);
            throw new AdapterException(exception.getMessage());
        }
    }


    @Override
    public PayPalBillingAgreementTokenData createBillingAgreementToken(final boolean skipShippingAddress) {
        try {
            LOG.debug("Creating billing agreement token");
            final DefaultPayPalCreateBillingAgreementTokenCommand command = getCommandFactory()
                .createCommand(DefaultPayPalCreateBillingAgreementTokenCommand.class);
            return command.perform(skipShippingAddress);
        } catch (final Exception exception) {
            LOG.error(UPDATING_ORDER_ERROR + exception.getMessage(),
                exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public PayPalBillingAgreementData createBillingAgreement(String billingAgreementToken) {
        LOG.debug("Creating billing agreement for token: " + billingAgreementToken);
        try {
            final DefaultPayPalCreateBillingAgreementCommand command = getCommandFactory()
                .createCommand(DefaultPayPalCreateBillingAgreementCommand.class);
            return command.perform(billingAgreementToken);
        } catch (CommandNotSupportedException exception) {
            LOG.error(UPDATING_ORDER_ERROR + exception.getMessage(),
                exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public String createOrder(final String currency, final String orderAmount,
        PayPalAddressDetailsData deliveryAddress) {
        final PayPalOrderRequestData orderRequestData = GenericBuilder.of(PayPalOrderRequestData::new)
            .with(PayPalOrderRequestData::setAmount, orderAmount)
            .with(PayPalOrderRequestData::setCurrency, currency)
            .with(PayPalOrderRequestData::setShippingAddress, deliveryAddress)
            .build();
        try {
            LOG.debug("CreateOrder with details: orderAmount: " + orderAmount + ", currency: " + currency);
            final DefaultPayPalCreateOrderCommand command = getCommandFactory()
                .createCommand(DefaultPayPalCreateOrderCommand.class);

            return command.perform(orderRequestData);
        } catch (final Exception exception) {
            LOG.error(UPDATING_ORDER_ERROR + exception.getMessage(),
                exception);
            throw new AdapterException(exception.getMessage());
        }
    }


    @Override
    public CaptureResult doCaptureIntentRequest(CaptureRequest request) {
        LOG.debug("Do Capture intent request :" + request);
        try {
            final DefaultPayPalCaptureIntentCommand command = getCommandFactory()
                .createCommand(DefaultPayPalCaptureIntentCommand.class);
            return command.perform(request);
        } catch (CommandNotSupportedException exception) {
            LOG.error("[PayPal Payment Service] Errors during capture intent request: " + exception.getMessage(),
                exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    /**
     * This method is used to get event by id
     * @param eventId event id
     * @return event
     */
    public Event getEventById(String eventId) {
        try {
            CommandFactory commandFactory = getCommandFactoryRegistry().getFactory(PAYPAL_PROVIDER_NAME);
            DefaultPayPalGetEventCommand command = commandFactory.createCommand(DefaultPayPalGetEventCommand.class);
            return command.perform(eventId);
        } catch (CommandNotSupportedException e) {
            String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            throw new AdapterException();
        }
    }

    /**
     * This method is used to cancel order
     * @param order order
     */
    public void doCancel(OrderModel order) {
        List<PaymentTransactionEntryModel> cancelableTransactions = getCancelableTransactions(order);
        for (PaymentTransactionEntryModel cancelableTransaction : cancelableTransactions) {
            PaymentTransactionEntryModel canceledTransaction = this.cancel(cancelableTransaction);
            if (TransactionStatus.ACCEPTED.name().equalsIgnoreCase(canceledTransaction.getTransactionStatus())) {
                cancelableTransaction.setTransactionStatus(TRANSACTION_STATUS_VOIDED);
                getModelService().save(cancelableTransaction);
            }
        }
        if (isOrderCanceled(order)) {
            order.setStatus(OrderStatus.CANCELLED);
            getModelService().save(order);
        }
    }

    /**
     * This method is used to get CancelableTransactions
     * @param order order
     * @return list of PaymentTransactionEntryModel
     */
    public List<PaymentTransactionEntryModel> getCancelableTransactions(OrderModel order) {
        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(authorizationEntries()).collect(Collectors.toList());

    }

    /**
     * This method is used to check if order is canceled
     * @param order order
     * @return checking result
     */
    public boolean isOrderCanceled(OrderModel order) {
        getModelService().refresh(order);

        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .noneMatch(authorizationEntries());
    }

    private boolean isAvailableRefund(final PaymentTransactionEntryModel selectedEntry) {
        return (PaymentTransactionType.CAPTURE.equals(selectedEntry.getType())
            || PaymentTransactionType.PARTIAL_CAPTURE.equals(selectedEntry.getType()))
            && TransactionStatus.ACCEPTED.name().equals(selectedEntry.getTransactionStatus());
    }

    private PaymentTransactionEntryModel createRefundTransaction(final RefundResult refundResult,
        final String entryCode,
        final PaymentTransactionType transactionType, final PaymentTransactionModel paymentTransaction,
        final String captureRequestId) {
        final CurrencyModel refundCurrency = refundResult.getCurrency() != null
            ? this.getCommonI18NService().getCurrency(refundResult.getCurrency().getCurrencyCode()) : null;

        return GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setCurrency, refundCurrency)
            .with(PaymentTransactionEntryModel::setType, transactionType)
            .with(PaymentTransactionEntryModel::setTime,
                refundResult.getRequestTime() == null ? new Date() : refundResult.getRequestTime())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, paymentTransaction)
            .with(PaymentTransactionEntryModel::setAmount, refundResult.getTotalAmount())
            .with(PaymentTransactionEntryModel::setRequestId, refundResult.getRequestId())
            .with(PaymentTransactionEntryModel::setRequestToken, refundResult.getRequestToken())
            .with(PaymentTransactionEntryModel::setSubscriptionID, captureRequestId)
            .with(PaymentTransactionEntryModel::setTransactionStatus, refundResult.getTransactionStatus().toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                refundResult.getTransactionStatusDetails().toString())
            .with(PaymentTransactionEntryModel::setCode, entryCode)
            .build();
    }

    private PaymentTransactionEntryModel createFailedRefundTransaction(final PaymentTransactionEntryModel selectedEntry,
        final BigDecimal amount, final PaymentTransactionType transactionType, final String entryCode,
        final String errorMessage) {
        return GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setCurrency, selectedEntry.getCurrency())
            .with(PaymentTransactionEntryModel::setType, transactionType)
            .with(PaymentTransactionEntryModel::setTime, new Date())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, selectedEntry.getPaymentTransaction())
            .with(PaymentTransactionEntryModel::setAmount, amount)
            .with(PaymentTransactionEntryModel::setRequestId, selectedEntry.getRequestId())
            .with(PaymentTransactionEntryModel::setRequestToken, selectedEntry.getRequestToken())
            .with(PaymentTransactionEntryModel::setSubscriptionID, selectedEntry.getRequestId())
            .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.REJECTED.name())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails, errorMessage)
            .with(PaymentTransactionEntryModel::setCode, entryCode)
            .build();
    }

    private PaymentTransactionEntryModel stubAuthorize(final PaymentTransactionModel transaction) {
        final PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        final String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);
        this.getModelService().save(transaction);

        final PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setAmount, transaction.getPlannedAmount())
            .with(PaymentTransactionEntryModel::setCurrency, transaction.getCurrency())
            .with(PaymentTransactionEntryModel::setType, paymentTransactionType)
            .with(PaymentTransactionEntryModel::setTime, new Date())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
            .with(PaymentTransactionEntryModel::setRequestId, transaction.getRequestId())
            .with(PaymentTransactionEntryModel::setRequestToken, transaction.getRequestToken())
            .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.ACCEPTED.toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                TransactionStatusDetails.SUCCESFULL.toString())
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .build();

        this.getModelService().attach(entry);
        this.getModelService().save(entry);
        this.getModelService().refresh(transaction);
        return entry;
    }

    private Predicate<PaymentTransactionEntryModel> authorizationEntries() {
        return entry -> (PaymentTransactionType.AUTHORIZATION.equals(entry.getType()))
            && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());
    }

    private CommandFactory getCommandFactory() {
        return commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME);
    }

    public CommandFactoryRegistry getCommandFactoryRegistry() {
        return this.commandFactoryRegistry;
    }

    public void setCommandFactoryRegistry(CommandFactoryRegistry commandFactoryRegistry) {
        this.commandFactoryRegistry = commandFactoryRegistry;
    }
}
