package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.service.PayPalManualMultiCaptureService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.methods.CardPaymentService;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.Date;

/**
 * This class is a default implementation of the PayPalManualMultiCaptureService interface
 */
public class DefaultPayPalManualMultiCaptureService implements PayPalManualMultiCaptureService {

    private ModelService modelService;
    private I18NService i18nService;
    private CardPaymentService cardPaymentService;

    /**
     * This method is used to do payment transaction multi capture
     * @param amount amount
     * @param selectedEntry selectedEntry
     * @param transactionType transactionType
     * @return PaymentTransactionEntryModel
     * @throws AdapterException if smth get wrong
     */
    public PaymentTransactionEntryModel doMultiCapture(BigDecimal amount, PaymentTransactionEntryModel selectedEntry,
        PaymentTransactionType transactionType) throws AdapterException {
        CaptureRequest request = prepareCaptureRequest(amount, selectedEntry, transactionType, selectedEntry.getSubscriptionID());
        CaptureResult captureResult;
        captureResult = processCapture(request);
        return createCapturePaymentTransactionEntry(captureResult, selectedEntry, transactionType);
    }

    private CaptureRequest prepareCaptureRequest(BigDecimal amount, PaymentTransactionEntryModel selectedEntry,
        PaymentTransactionType transactionType, String subscriptionId) {
        final String merchantTransactionCode = getNewPaymentTransactionEntryCode(selectedEntry.getPaymentTransaction(),
            transactionType);
        final String requestId = selectedEntry.getRequestId();
        final String requestToken = selectedEntry.getRequestToken();
        final Currency currency = Currency.getInstance(selectedEntry.getCurrency().getIsocode());
        final String paymentProvider = selectedEntry.getPaymentTransaction().getPaymentProvider();

         return new CaptureRequest(merchantTransactionCode, requestId, requestToken,
            currency, amount, paymentProvider, subscriptionId);
    }

    private CaptureResult processCapture(CaptureRequest request) {
        CaptureResult captureResult = cardPaymentService.capture(request);
        return captureResult;
    }

    private PaymentTransactionEntryModel createCapturePaymentTransactionEntry(CaptureResult captureResult,
        PaymentTransactionEntryModel selectedEntry, PaymentTransactionType transactionType) {
        final PaymentTransactionModel paymentTransaction = selectedEntry.getPaymentTransaction();
        final String newEntryCode = getNewPaymentTransactionEntryCode(selectedEntry.getPaymentTransaction(),
            transactionType);
        final CurrencyModel currency = captureResult.getCurrency() != null ? getI18nService()
            .getCurrency(captureResult.getCurrency().getCurrencyCode()) : null;

        final PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setAmount, captureResult.getTotalAmount())
            .with(PaymentTransactionEntryModel::setCurrency, currency)
            .with(PaymentTransactionEntryModel::setType, transactionType)
            .with(PaymentTransactionEntryModel::setTime, captureResult.getRequestTime() == null ? new Date()
                : captureResult.getRequestTime())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, paymentTransaction)
            .with(PaymentTransactionEntryModel::setRequestId, captureResult.getRequestId())
            .with(PaymentTransactionEntryModel::setRequestToken, captureResult.getRequestToken())
            .with(PaymentTransactionEntryModel::setSubscriptionID, captureResult.getReconciliationId())
            .with(PaymentTransactionEntryModel::setTransactionStatus, captureResult.getTransactionStatus().toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                captureResult.getTransactionStatusDetails().toString())
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .build();

        getModelService().attach(entry);
        getModelService().save(entry);
        getModelService().refresh(paymentTransaction);
        return entry;
    }

    private String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
        PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                + 1);
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public I18NService getI18nService() {
        return i18nService;
    }

    public void setI18nService(I18NService i18nService) {
        this.i18nService = i18nService;
    }

    public CardPaymentService getCardPaymentService() {
        return cardPaymentService;
    }

    public void setCardPaymentService(CardPaymentService cardPaymentService) {
        this.cardPaymentService = cardPaymentService;
    }
}
