package com.paypal.hybris.core.service.impl;

import com.google.common.base.Predicate;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalManualAuthorizationService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.log4j.Logger;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.Date;
import java.util.UUID;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;

/**
 * This class is a default implementation of the PayPalManualAuthorizationService interface
 */
public class DefaultPayPalManualAuthorizationService implements PayPalManualAuthorizationService {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalManualAuthorizationService.class);

    private PayPalPaymentService payPalPaymentService;
    private I18NService i18nService;
    private ModelService modelService;
    private CommandFactoryRegistry commandFactoryRegistry;

    /**
     * This method is used to do authorization
     * @param order order
     * @param amount amount
     * @return payment transaction model
     */
    public PaymentTransactionEntryModel doAuthorization(final OrderModel order, final BigDecimal amount) {
        try {
            final String payPalOrderId = createOrderForGivenAmount(order.getCurrency(), amount);
            LOG.error("New order Id --  " + payPalOrderId);
            final SubscriptionAuthorizationRequest request = prepareAuthorizationRequest(order, amount, payPalOrderId);
            final AuthorizationResult result = executeSubscriptionAuthorizeCommand(request);
            final PaymentTransactionEntryModel paymentTransactionEntry = createPaymentTransactionEntry(result, order,
                payPalOrderId);
            updatePlanetAmount(order);
            return paymentTransactionEntry;
        } catch (AdapterException exception) {
            String message = "Exception, message : " + exception.getMessage();
            LOG.error(message, exception);
            return handlePayPalSDKError(order, exception.getMessage(), amount);
        }
    }

    private String createOrderForGivenAmount(final CurrencyModel currency, final BigDecimal amount) {

        return getPayPalPaymentService()
            .createOrder(currency.getIsocode(), amount.toString(), null);

    }

    private SubscriptionAuthorizationRequest prepareAuthorizationRequest(final OrderModel order,
        final BigDecimal amount,
        final String payPalOrderId) {
        final String merchantTransactionCode = generateCode(order);
        final Currency currency = getI18nService().getBestMatchingJavaCurrency(order.getCurrency().getIsocode());
        final BillingInfo shippingInfo = createBillingInfo(order.getDeliveryAddress());
        final String cv2 = ((PayPalCreditCardPaymentInfoModel) order.getPaymentInfo()).getBillingAgreementId();

        return new SubscriptionAuthorizationRequest(merchantTransactionCode,
            payPalOrderId, currency, amount, shippingInfo, cv2, PAYPAL_PROVIDER_NAME);
    }

    private String generateCode(final OrderModel order) {
        return order.getUser().getUid() + "-" + UUID.randomUUID();
    }

    protected BillingInfo createBillingInfo(final AddressModel address) {
        if (address == null) {
            return null;
        } else {
            final String country = address.getCountry() != null ? address.getCountry().getIsocode() : null;
            final String region = address.getRegion() != null ? address.getRegion().getName() : null;

            return GenericBuilder.of(BillingInfo::new)
                .with(BillingInfo::setCity, address.getTown())
                .with(BillingInfo::setCountry, country)
                .with(BillingInfo::setEmail, address.getEmail())
                .with(BillingInfo::setFirstName, address.getFirstname())
                .with(BillingInfo::setLastName, address.getFirstname())
                .with(BillingInfo::setPhoneNumber, address.getPhone1())
                .with(BillingInfo::setPostalCode, address.getPostalcode())
                .with(BillingInfo::setRegion, region)
                .with(BillingInfo::setStreet1, address.getStreetname())
                .with(BillingInfo::setStreet2, address.getStreetnumber())
                .build();
        }
    }

    private AuthorizationResult executeSubscriptionAuthorizeCommand(final SubscriptionAuthorizationRequest request) {

        try {
            SubscriptionAuthorizationCommand command = getCommandFactory()
                .createCommand(SubscriptionAuthorizationCommand.class);

            return command.perform(request);
        } catch (CommandNotSupportedException var5) {
            throw new AdapterException(var5.getMessage(), var5);
        }
    }

    private PaymentTransactionEntryModel createPaymentTransactionEntry(final AuthorizationResult result,
        final OrderModel order, final String payPalOrderId) {
        final PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        final PaymentTransactionModel transaction = order.getPaymentTransactions().stream().findFirst().get();

        final String newEntryCode = getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);
        final CurrencyModel currency = result.getCurrency() != null ? this.getI18nService()
            .getCurrency(result.getCurrency().getCurrencyCode()) : null;

        PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setCurrency, currency)
            .with(PaymentTransactionEntryModel::setAmount, result.getTotalAmount())
            .with(PaymentTransactionEntryModel::setType, paymentTransactionType)
            .with(PaymentTransactionEntryModel::setTime, result.getAuthorizationTime() == null ? new Date()
                : result.getAuthorizationTime())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
            .with(PaymentTransactionEntryModel::setRequestId, result.getRequestId())
            .with(PaymentTransactionEntryModel::setRequestToken, result.getRequestToken())
            .with(PaymentTransactionEntryModel::setTransactionStatus, result.getTransactionStatus().toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                result.getTransactionStatusDetails().toString())
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .with(PaymentTransactionEntryModel::setSubscriptionID, payPalOrderId)
            .build();
        getModelService().attach(entry);
        getModelService().save(entry);
        getModelService().refresh(transaction);
        return entry;
    }

    private PaymentTransactionEntryModel handlePayPalSDKError(final OrderModel order, String message,
        final BigDecimal amount) {
        PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        PaymentTransactionModel transaction = order.getPaymentTransactions().stream().findFirst().get();
        String newEntryCode = getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);
        PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
            .with(PaymentTransactionEntryModel::setType, paymentTransactionType)
            .with(PaymentTransactionEntryModel::setAmount, amount)
            .with(PaymentTransactionEntryModel::setCurrency, transaction.getCurrency())
            .with(PaymentTransactionEntryModel::setTime, new Date())
            .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.ERROR.toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails, message)
            .build();
        modelService.attach(entry);
        getModelService().save(entry);
        getModelService().refresh(transaction);
        return entry;
    }

    /**
     *This method is used to get new PaymentTransaction EntryCode
     * @param transaction transaction
     * @param paymentTransactionType transaction type
     * @return payment transaction entry code
     */
    public String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
        PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                + 1);
    }

    private void updatePlanetAmount(final OrderModel order) {
        PaymentTransactionModel transaction = order.getPaymentTransactions().stream().findFirst().get();
        transaction.setPlannedAmount(getAuthorizedAmount(transaction));
        getModelService().save(transaction);
    }

    private BigDecimal getAuthorizedAmount(final PaymentTransactionModel transaction) {
        Predicate<PaymentTransactionEntryModel> authorizationEntries = (entry -> (PaymentTransactionType.AUTHORIZATION
            .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()));

        double authorizedAmount = transaction.getEntries().stream()
            .filter(authorizationEntries).mapToDouble(entry -> entry.getAmount().doubleValue()).sum();
        return BigDecimal.valueOf(authorizedAmount);
    }

    public PayPalPaymentService getPayPalPaymentService() {
        return payPalPaymentService;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public I18NService getI18nService() {
        return i18nService;
    }

    public void setI18nService(I18NService i18nService) {
        this.i18nService = i18nService;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    private CommandFactory getCommandFactory() {
        return commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME);
    }

    public CommandFactoryRegistry getCommandFactoryRegistry() {
        return commandFactoryRegistry;
    }

    public void setCommandFactoryRegistry(CommandFactoryRegistry commandFactoryRegistry) {
        this.commandFactoryRegistry = commandFactoryRegistry;
    }
}
