package com.paypal.hybris.core.service.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHARACTERS_FOR_PASSWORD;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_EMPTY_STRING;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_TEMP_PASSWORD_LENGTH;

import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.impl.DefaultCustomerAccountService;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import java.security.SecureRandom;
import java.util.UUID;

/**
 * This class is a default implementation of the PayPalCustomerAccountService interface
 */
public class DefaultPayPalCustomerAccountService extends DefaultCustomerAccountService implements
    PayPalCustomerAccountService {

    private static final int ZERO = 0;

    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;
    private PayPalCustomerAccountDao payPalCustomerAccountDao;
    private Converter<CustomerModel, CustomerData> customerConverter;
    private ModelService modelService;

    @Override
    public PayPalCreditCardPaymentInfoModel createPayPalPaymentSubscription(final CustomerModel customerModel,
        final CCPaymentInfoData ccPaymentInfoData, final AddressModel billingAddress) {
        final PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel = getModelService()
            .create(PayPalCreditCardPaymentInfoModel.class);

        populateDumbCCInfo(payPalPaymentInfoModel, customerModel);
        populatePayPalPaymentInfo(payPalPaymentInfoModel, ccPaymentInfoData);

        payPalPaymentInfoModel.setBillingAddress(billingAddress);
        payPalPaymentInfoModel.setIntent(defaultPayPalConfigurationService.getPayPalIntent());
        if (billingAddress.getOwner() == null) {
            billingAddress.setOwner(payPalPaymentInfoModel);
        }
        getModelService().saveAll(billingAddress, payPalPaymentInfoModel);
        getModelService().refresh(customerModel);

        addPaymentInfo(customerModel, payPalPaymentInfoModel);
        if (ccPaymentInfoData.isDefaultPaymentInfo()) {
            customerModel.setDefaultPaymentInfo(payPalPaymentInfoModel);
            getModelService().save(customerModel);
        }
        return payPalPaymentInfoModel;
    }

    @Override
    public PayPalCreditCardPaymentInfoModel updatePayPalPaymentSubscription(final CustomerModel customerModel,
        final CCPaymentInfoData ccPaymentInfoData, final AddressModel billingAddress,
        PayPalCreditCardPaymentInfoModel paymentInfo) {
        populatePayPalPaymentInfo(paymentInfo, ccPaymentInfoData);

        paymentInfo.setBillingAddress(billingAddress);
        if (billingAddress.getOwner() == null) {
            billingAddress.setOwner(paymentInfo);
        }
        getModelService().saveAll(billingAddress, paymentInfo);
        getModelService().refresh(customerModel);

        addPaymentInfo(customerModel, paymentInfo);
        return paymentInfo;
    }

    @Override
    public CustomerData getCustomerByPayerId(String payerId) {
        return customerConverter.convert(payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId));
    }

    @Override
    public boolean isCustomerWithPayerIdExist(String payerId) {
        return payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId) != null;
    }

    @Override
    public String setTempPassword(String payerId) {
        final CustomerModel customerModel = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        final String tempPassword = generateRandomPassword(PAYPAL_TEMP_PASSWORD_LENGTH);
        customerModel.setPassword(tempPassword);
        modelService.save(customerModel);
        return tempPassword;
    }

    @Override
    public void clearTempPassword(String payerId) {
        final CustomerModel customerModel = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        customerModel.setPassword(PAYPAL_EMPTY_STRING);
        modelService.save(customerModel);
    }

    private void populatePayPalPaymentInfo(final PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel,
        final CCPaymentInfoData ccPaymentInfoData) {
        payPalPaymentInfoModel.setPayerEmail(ccPaymentInfoData.getPayerEmail());
        payPalPaymentInfoModel.setPayPalOrderId(ccPaymentInfoData.getPayPalOrderId());
        payPalPaymentInfoModel.setPayerId(ccPaymentInfoData.getPayerId());
        payPalPaymentInfoModel.setBillingAgreementId(ccPaymentInfoData.getBillingAgreementId());
        payPalPaymentInfoModel.setSaved(ccPaymentInfoData.isSaved());
        payPalPaymentInfoModel.setSubscriptionId(ccPaymentInfoData.getSubscriptionId());
        payPalPaymentInfoModel.setCcOwner(ccPaymentInfoData.getAccountHolderName());
        payPalPaymentInfoModel.setType(CreditCardType.valueOf(ccPaymentInfoData.getCardType().toUpperCase()));
    }

    private void populateDumbCCInfo(final CreditCardPaymentInfoModel creditCardPaymentInfoModel,
        final CustomerModel customerModel) {
        creditCardPaymentInfoModel.setCode(customerModel.getUid() + "_" + UUID.randomUUID());
        creditCardPaymentInfoModel.setUser(customerModel);
        creditCardPaymentInfoModel.setValidToMonth(String.valueOf(ZERO));
        creditCardPaymentInfoModel.setValidToYear(String.valueOf(ZERO));
        creditCardPaymentInfoModel.setNumber(getMaskedCardNumber("****************"));
    }

    public DefaultPayPalConfigurationService getDefaultPayPalConfigurationService() {
        return defaultPayPalConfigurationService;
    }

    public void setDefaultPayPalConfigurationService(DefaultPayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    private static String generateRandomPassword(final int len) {
        final SecureRandom random = new SecureRandom();
        final StringBuilder sb = new StringBuilder();
        for (int i = 0; i < len; i++) {
            final int randomIndex = random.nextInt(PAYPAL_CHARACTERS_FOR_PASSWORD.length());
            sb.append(PAYPAL_CHARACTERS_FOR_PASSWORD.charAt(randomIndex));
        }
        return sb.toString();
    }

    public PayPalCustomerAccountDao getPayPalCustomerAccountDao() {
        return payPalCustomerAccountDao;
    }

    public void setPayPalCustomerAccountDao(PayPalCustomerAccountDao payPalCustomerAccountDao) {
        this.payPalCustomerAccountDao = payPalCustomerAccountDao;
    }

    public Converter<CustomerModel, CustomerData> getCustomerConverter() {
        return customerConverter;
    }

    public void setCustomerConverter(
        Converter<CustomerModel, CustomerData> customerConverter) {
        this.customerConverter = customerConverter;
    }

    @Override
    public ModelService getModelService() {
        return modelService;
    }

    @Override
    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }
}
