package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCommerceCheckoutService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.strategy.PayPalCommercePaymentProviderStrategy;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commerceservices.order.CommercePaymentAuthorizationStrategy;
import de.hybris.platform.commerceservices.order.impl.DefaultCommerceCheckoutService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.session.SessionService;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.BooleanUtils;

import java.util.Map;
import org.apache.log4j.Logger;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.*;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNull;


public class DefaultPayPalCommerceCheckoutService extends DefaultCommerceCheckoutService implements
    PayPalCommerceCheckoutService {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalCommerceCheckoutService.class);

    private Map<String, CommercePaymentAuthorizationStrategy> strategyMap;
    private SessionService sessionService;
    private PayPalPaymentService payPalPaymentService;
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;
    private Converter<CartModel, CartData> cartConverter;
    private PayPalCommercePaymentProviderStrategy commercePaymentProviderStrategy;


    @Override
    public String getPaymentProvider(CartModel cart) {

        return getCommercePaymentProviderStrategy().getPaymentProvider(cart);
    }

    @Override
    public PaymentTransactionEntryModel authorizePayment(CommerceCheckoutParameter parameter) {
        final CartModel cartModel = parameter.getCart();
        validateParameterNotNull(cartModel, "Cart model cannot be null");
        validateParameterNotNull(cartModel.getPaymentInfo(), "Payment information on cart cannot be null");
        refreshOrderIntent(cartModel.getPaymentInfo());
        if (parameter.getAuthorizationAmount() == null) {
            parameter.setAuthorizationAmount(calculateAuthAmount(cartModel));
        }
        final String authorizationFlow = getAuthorizationFlow(cartModel.getPaymentInfo());
        if (isPayPalOrderShouldBeCreated(authorizationFlow, cartModel.getPaymentInfo())) {
            createPayPalOrder(cartModel);
        }
        return getAuthorizationStrategy(authorizationFlow).authorizePaymentAmount(parameter);
    }

    private void createPayPalOrder(final CartModel cartModel) {
        final PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) cartModel
            .getPaymentInfo();
        final CartData cartData = cartConverter.convert(cartModel);
        final String orderId = payPalPaymentService
            .createOrder(cartData.getTotalPrice().getCurrencyIso(),
                getIntegerPriceValue(cartData), null);
        paymentInfo.setSubscriptionId(orderId);
        getModelService().save(paymentInfo);
        getModelService().refresh(cartModel);
    }

    private String getIntegerPriceValue(final CartData cartData) {
        try {
            String priceValue;
            if (isNonDecimalCurrency(cartData)) {
                priceValue = cartData.getTotalPriceWithTax().getValue().toBigInteger().toString();
            } else {
                priceValue = cartData.getTotalPriceWithTax().getValue().toString();
            }
            return priceValue;
        } catch (ClassCastException e) {
            LOG.error("[Default PayPal Commerce Checkout Service] Errors during currency converting: " + e.getMessage(),
                e);
            throw new ClassCastException(e.getMessage());
        }
    }

    private Boolean isNonDecimalCurrency(final CartData cartData) {
        final String paypalNonDecimalCurrency = defaultPayPalConfigurationService.getNonDecimalCurrency();
        return paypalNonDecimalCurrency.contains(cartData.getTotalPrice().getCurrencyIso());
    }

    private boolean isPayPalOrderShouldBeCreated(final String flow, final PaymentInfoModel paymentInfoModel) {
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel) {
            return (PAYPAL_ORDER_PROCESS_FLOW.equals(flow) || PAYPAL_INTENT_CAPTURE_PROCESS_FLOW.equals(flow))
                && PAYPAL_ORDER_ID_PLACEHOLDER
                .equals(((PayPalCreditCardPaymentInfoModel) paymentInfoModel).getSubscriptionId());
        }
        return false;
    }

    private String getAuthorizationFlow(PaymentInfoModel paymentInfoModel) {
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel) {
            if (BooleanUtils.isTrue(sessionService.getAttribute(PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE))) {
                return PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE;
            } else if (((PayPalCreditCardPaymentInfoModel) paymentInfoModel).getIntent()
                .equalsIgnoreCase(PAYPAL_INTENT_CAPTURE)) {
                return PAYPAL_INTENT_CAPTURE_PROCESS_FLOW;
            } else {
                return PAYPAL_ORDER_PROCESS_FLOW;
            }
        }
        return DEFAULT_FLOW;
    }

    private CommercePaymentAuthorizationStrategy getAuthorizationStrategy(final String flow) {
        if (StringUtils.equals(flow, PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE)) {
            sessionService.removeAttribute(PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE);
            return strategyMap.get(PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE);
        } else if (StringUtils.equals(flow, PAYPAL_INTENT_CAPTURE_PROCESS_FLOW)) {
            return strategyMap.get(PAYPAL_INTENT_CAPTURE_PROCESS_FLOW);
        } else if (StringUtils.equals(flow, PAYPAL_ORDER_PROCESS_FLOW)) {
            return strategyMap.get(PAYPAL_ORDER_PROCESS_FLOW);
        }
        return strategyMap.get(DEFAULT_FLOW);
    }

    private void refreshOrderIntent(PaymentInfoModel paymentInfoModel) {
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel) {
            ((PayPalCreditCardPaymentInfoModel) paymentInfoModel)
                .setIntent(getDefaultPayPalConfigurationService().getPayPalIntent());
        }
        getModelService().save(paymentInfoModel);
    }

    public Map<String, CommercePaymentAuthorizationStrategy> getStrategyMap() {
        return strategyMap;
    }

    public void setStrategyMap(
        Map<String, CommercePaymentAuthorizationStrategy> strategyMap) {
        this.strategyMap = strategyMap;
    }

    public SessionService getSessionService() {
        return sessionService;
    }

    public void setSessionService(SessionService sessionService) {
        this.sessionService = sessionService;
    }

    public PayPalPaymentService getPayPalPaymentService() {
        return payPalPaymentService;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public DefaultPayPalConfigurationService getDefaultPayPalConfigurationService() {
        return defaultPayPalConfigurationService;
    }

    public void setDefaultPayPalConfigurationService(DefaultPayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public Converter<CartModel, CartData> getCartConverter() {
        return cartConverter;
    }

    public void setCartConverter(
        Converter<CartModel, CartData> cartConverter) {
        this.cartConverter = cartConverter;
    }

    @Override
    public PayPalCommercePaymentProviderStrategy getCommercePaymentProviderStrategy() {
        return commercePaymentProviderStrategy;
    }

    public void setCommercePaymentProviderStrategy(PayPalCommercePaymentProviderStrategy commercePaymentProviderStrategy) {
        this.commercePaymentProviderStrategy = commercePaymentProviderStrategy;
    }
}
