package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.orders.Authorization;
import com.paypal.orders.AuthorizeRequest;
import com.paypal.orders.Order;
import com.paypal.orders.OrdersAuthorizeRequest;
import com.paypal.orders.PaymentCollection;
import com.paypal.orders.PaymentSource;
import com.paypal.orders.PurchaseUnit;
import com.paypal.orders.Token;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.Optional;
import java.util.function.Predicate;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.BILLING_AGREEMENT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;


public class DefaultPayPalSubscriptionAuthorizationCommand extends PayPalAbstractCommand implements
    SubscriptionAuthorizationCommand {

    private static final String DEFAULT = "default";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalSubscriptionAuthorizationCommand.class);

    @Override
    public AuthorizationResult perform(SubscriptionAuthorizationRequest request) {
        OrdersAuthorizeRequest ordersAuthorizeRequest = Optional.ofNullable(request.getSubscriptionID())
            .map(OrdersAuthorizeRequest::new).orElseThrow(()
                -> new IllegalArgumentException(
                " Subscription ID is undefined, actual Subscription ID is: " + request.getSubscriptionID()));

        if (request.getCv2() != null) {
            ordersAuthorizeRequest.requestBody(prepareRequestBody(request));
        }
        HttpResponse<Order> response = null;
        try {
            response = createClient().execute(ordersAuthorizeRequest);
        } catch (IOException e) {
            LOG.error("Authorization failed: ", e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
        return translateResponse(response, request);
    }

    private AuthorizationResult translateResponse(HttpResponse<Order> response,
        SubscriptionAuthorizationRequest request) {
        String resultStatus = response.result().status();

        Authorization authorization = response.result().purchaseUnits().stream().filter(isDefault())
            .map(PurchaseUnit::payments)
            .findFirst()
            .map(PaymentCollection::authorizations)
            .orElseThrow(() -> new IllegalArgumentException(" Payments are empty "))
            .stream().filter(isCreated())
            .findFirst().orElseThrow(() -> new IllegalArgumentException(" No Authorization with status created "));

        AuthorizationResult result = getAuthorizationResult(request, resultStatus, authorization);

        return result;
    }

    private AuthorizationResult getAuthorizationResult(SubscriptionAuthorizationRequest request, String resultStatus,
        Authorization authorization) {
        final TransactionStatus transactionStatus = getTransactionStatusMap()
            .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
            getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

        return GenericBuilder.of(AuthorizationResult::new)
            .with(AuthorizationResult::setRequestId, authorization.id())
            .with(AuthorizationResult::setAuthorizationCode, authorization.id())
            .with(AuthorizationResult::setPaymentProvider, request.getPaymentProvider())
            .with(AuthorizationResult::setCurrency, request.getCurrency())
            .with(AuthorizationResult::setTotalAmount, request.getTotalAmount())
            .with(AuthorizationResult::setReconciliationId, request.getSubscriptionID())
            .with(AuthorizationResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
            .with(AuthorizationResult::setTransactionStatus, transactionStatus)
            .with(AuthorizationResult::setTransactionStatusDetails, transactionStatusDetails)
            .build();
    }

    private AuthorizeRequest prepareRequestBody(SubscriptionAuthorizationRequest request) {
        AuthorizeRequest authorizeRequest = new AuthorizeRequest();
        PaymentSource paymentSource = new PaymentSource();
        Token token = new Token();
        token.id(request.getCv2());
        token.type(BILLING_AGREEMENT);
        paymentSource.token(token);
        authorizeRequest.paymentSource(paymentSource);
        return authorizeRequest;
    }

    private static Predicate<PurchaseUnit> isDefault() {
        return unit -> unit != null && DEFAULT.equals(unit.referenceId());
    }

    private Predicate<Authorization> isCreated() {
        return authorization -> authorization != null && CREATED_STATUS_RESULT.equals(authorization.status());
    }

}
