package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.payments.Authorization;
import com.paypal.payments.AuthorizationsReauthorizeRequest;
import com.paypal.payments.Money;
import com.paypal.payments.ReauthorizeRequest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.RoundingMode;
import java.util.Optional;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;


public class DefaultPayPalReauthorizationRequestCommand extends PayPalAbstractCommand
    implements SubscriptionAuthorizationCommand {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalReauthorizationRequestCommand.class);

    @Override
    public AuthorizationResult perform(SubscriptionAuthorizationRequest request) {
        AuthorizationsReauthorizeRequest authorizationsReauthorizeRequest = Optional
            .ofNullable(request.getSubscriptionID())
            .map(AuthorizationsReauthorizeRequest::new).orElseThrow(()
                -> new IllegalArgumentException(
                " Subscription ID is undefined, actual Subscription ID is: " + request.getSubscriptionID()));

        authorizationsReauthorizeRequest.requestBody(prepareRequestBody(request));
        HttpResponse<Authorization> response = null;
        try {
            response = createClient().execute(authorizationsReauthorizeRequest);
        } catch (IOException e) {
            LOG.error("Reauthorization failed: ", e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
        return translateResponse(response, request);
    }

    private AuthorizationResult translateResponse(HttpResponse<Authorization> response,
        SubscriptionAuthorizationRequest request) {

        AuthorizationResult result = getAuthorizationResult(request, response.result());
        return result;
    }

    private AuthorizationResult getAuthorizationResult(SubscriptionAuthorizationRequest request,
        Authorization response) {
        final TransactionStatus transactionStatus =
            CREATED_STATUS_RESULT.equals(response.status()) ? TransactionStatus.ACCEPTED
                : TransactionStatus.REJECTED;
        final TransactionStatusDetails transactionStatusDetails = CREATED_STATUS_RESULT.equals(response.status()) ?
            TransactionStatusDetails.SUCCESFULL : TransactionStatusDetails.PROCESSOR_DECLINE;

        return GenericBuilder.of(AuthorizationResult::new)
            .with(AuthorizationResult::setRequestId, request.getSubscriptionID())
            .with(AuthorizationResult::setAuthorizationCode, response.id())
            .with(AuthorizationResult::setPaymentProvider, request.getPaymentProvider())
            .with(AuthorizationResult::setCurrency, request.getCurrency())
            .with(AuthorizationResult::setTotalAmount, request.getTotalAmount())
            .with(AuthorizationResult::setReconciliationId, request.getSubscriptionID())
            .with(AuthorizationResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
            .with(AuthorizationResult::setTransactionStatus, transactionStatus)
            .with(AuthorizationResult::setTransactionStatusDetails, transactionStatusDetails)
            .build();
    }

    private ReauthorizeRequest prepareRequestBody(SubscriptionAuthorizationRequest request) {
        ReauthorizeRequest reauthorizeRequest = new ReauthorizeRequest();
        Money money = new Money();
        money.currencyCode(request.getCurrency().getCurrencyCode());
        money.value(String.valueOf(request.getTotalAmount().setScale(2, RoundingMode.HALF_UP)));
        reauthorizeRequest.amount(money);
        return reauthorizeRequest;
    }

}
