package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.orders.AddressPortable;
import com.paypal.orders.Order;
import com.paypal.orders.OrdersGetRequest;
import com.paypal.orders.PurchaseUnit;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.Command;
import org.apache.log4j.Logger;

import java.io.IOException;


public class DefaultPayPalGetOrderDetailsCommand extends PayPalAbstractCommand
    implements Command<String, PayPalOrderDetailsData> {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalGetOrderDetailsCommand.class);

    @Override
    public PayPalOrderDetailsData perform(final String payPalOrderId) {
        OrdersGetRequest ordersGetRequest = new OrdersGetRequest(payPalOrderId);
        HttpResponse<Order> response = null;
        try {
            response = createClient().execute(ordersGetRequest);
        } catch (IOException e) {
            LOG.error("Getting order details failed. OrderId: " + payPalOrderId + "\nException: " + e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
        return translateResponse(response);
    }

    private PayPalOrderDetailsData translateResponse(HttpResponse<Order> payPalResponse) {
        if (payPalResponse == null) {
            return null;
        }
        final Order payPalOrder = payPalResponse.result();
        final PurchaseUnit purchaseUnit = payPalOrder.purchaseUnits().get(0);
        final AddressPortable payPalBillingAddress = payPalOrder.payer().addressPortable();
        final String firstName = payPalOrder.payer().name().givenName();
        final String lastName = payPalOrder.payer().name().surname();
        final String email = payPalOrder.payer().email();
        final PayPalAddressDetailsData billingAddress = convertPayPalAddress(payPalBillingAddress, email, firstName,
            lastName);
        AddressPortable payPalShippingAddress = new AddressPortable();
        String shippingFirstName = firstName;
        String shippingLastName = lastName;
        if (purchaseUnit.shippingDetail() != null) {
            payPalShippingAddress = purchaseUnit.shippingDetail().addressPortable();
            final String fullName = purchaseUnit.shippingDetail().name().fullName();
            int delimiter = fullName.lastIndexOf(' ');
            if (delimiter != -1) {
                shippingFirstName = fullName.substring(0, delimiter);
                shippingLastName = fullName.substring(delimiter + 1);
            }
        }
        final PayPalAddressDetailsData shippingAddress = convertPayPalAddress(payPalShippingAddress, email,
            shippingFirstName, shippingLastName);

        return GenericBuilder.of(PayPalOrderDetailsData::new)
            .with(PayPalOrderDetailsData::setOrderId, payPalOrder.id())
            .with(PayPalOrderDetailsData::setBuyerId, payPalOrder.payer().payerId())
            .with(PayPalOrderDetailsData::setBuyer, firstName + " " + lastName)
            .with(PayPalOrderDetailsData::setAmount, purchaseUnit.amountWithBreakdown().value())
            .with(PayPalOrderDetailsData::setShippingAddress, shippingAddress)
            .with(PayPalOrderDetailsData::setBillingAddress, billingAddress)
            .build();
    }

    private PayPalAddressDetailsData convertPayPalAddress(final AddressPortable addressPortable, final String email,
        final String firstName, final String lastName) {
        return GenericBuilder.of(PayPalAddressDetailsData::new)
            .with(PayPalAddressDetailsData::setCity, addressPortable.adminArea2())
            .with(PayPalAddressDetailsData::setRegion, addressPortable.adminArea1())
            .with(PayPalAddressDetailsData::setCountryCode, addressPortable.countryCode())
            .with(PayPalAddressDetailsData::setPostalCode, addressPortable.postalCode())
            .with(PayPalAddressDetailsData::setLine1, addressPortable.addressLine1())
            .with(PayPalAddressDetailsData::setLine2, addressPortable.addressLine2())
            .with(PayPalAddressDetailsData::setEmail, email)
            .with(PayPalAddressDetailsData::setFirstName, firstName)
            .with(PayPalAddressDetailsData::setLastName, lastName)
            .build();
    }
}
