package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.payments.CapturesRefundRequest;
import com.paypal.payments.Money;
import com.paypal.payments.Refund;
import com.paypal.payments.RefundRequest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.FollowOnRefundCommand;
import de.hybris.platform.payment.commands.request.FollowOnRefundRequest;
import de.hybris.platform.payment.commands.result.RefundResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Optional;


public class DefaultPayPalFollowOnRefundCommand extends PayPalAbstractCommand implements
    FollowOnRefundCommand<FollowOnRefundRequest> {

    private static final String TRANSACTION_REFUND_FAILURE_MESSAGE = "Transaction refund failed: ";
    private static final String TRANSACTION_REFUND_PREFER_HEADER = "return=representation";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalFollowOnRefundCommand.class);

    @Override
    public RefundResult perform(FollowOnRefundRequest request) {

        CapturesRefundRequest capturesRefundRequest = Optional.ofNullable(request)
            .map(refundRequest -> new CapturesRefundRequest(request.getRequestId())).orElseThrow(()
                -> new IllegalArgumentException(
                "Request ID is undefined, actual Request ID is: " + request.getRequestId()));

        capturesRefundRequest.requestBody(prepareRequestBody(request));
        capturesRefundRequest.prefer(TRANSACTION_REFUND_PREFER_HEADER);

        HttpResponse<Refund> response = null;
        try {
            response = createClient().execute(capturesRefundRequest);
        } catch (IOException e) {
            LOG.error(TRANSACTION_REFUND_FAILURE_MESSAGE, e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
        return translateResponse(response, request);
    }

    private RefundResult translateResponse(HttpResponse<Refund> response, FollowOnRefundRequest request) {
        String resultStatus = response.result().status();
        Refund refund = response.result();
        RefundResult result = getRefundResult(request, resultStatus, refund);
        return result;
    }

    private RefundResult getRefundResult(FollowOnRefundRequest request, String resultStatus, Refund refund) {
        final TransactionStatus transactionStatus = getTransactionStatusMap()
            .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
            getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);
        BigDecimal refundedAmount = request.getTotalAmount();

        if (refund.amount() != null){
            refundedAmount = new BigDecimal(refund.amount().value());
        }


        return GenericBuilder.of(RefundResult::new)
            .with(RefundResult::setRequestId, refund.id())
            .with(RefundResult::setCurrency, request.getCurrency())
            .with(RefundResult::setTotalAmount, refundedAmount)
            .with(RefundResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
            .with(RefundResult::setTransactionStatus, transactionStatus)
            .with(RefundResult::setTransactionStatusDetails, transactionStatusDetails)
            .build();
    }

    private RefundRequest prepareRequestBody(FollowOnRefundRequest request) {
        RefundRequest refundRequest = new RefundRequest();
        Money amount = new Money();
        amount.value(String.valueOf(request.getTotalAmount().setScale(2, RoundingMode.HALF_UP)));
        amount.currencyCode(request.getCurrency().getCurrencyCode());
        refundRequest.amount(amount);
        return refundRequest;
    }

}
