package com.paypal.hybris.core.commands.impl;

import com.paypal.base.rest.JSONFormatter;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalBillingAgreementRequestData;
import com.paypal.hybris.data.PayPalBillingAgreementTokenData;
import com.paypal.hybris.data.PayPalMerchantPreferencesData;
import com.paypal.hybris.data.PayPalPayerData;
import com.paypal.hybris.data.PayPalPlanData;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.Command;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;


public class DefaultPayPalCreateBillingAgreementTokenCommand extends PayPalAbstractCommand
    implements Command<Boolean, PayPalBillingAgreementTokenData> {

    private static final String PAYPAL_PAYMENT_METHOD = "PAYPAL";
    private static final String ACCEPTED_PYMT_TYPE_INSTANT = "INSTANT";
    private static final String RETURN_URL = "return";
    private static final String CANCEL_URL = "cancel";
    private static final String NOTIFY_URL = "notify";
    private static final String PLAN_DATA_TYPE = "MERCHANT_INITIATED_BILLING_SINGLE_AGREEMENT";
    private static final String AGREEMENT_TOKENS = "billing-agreements/agreement-tokens";
    private static final String CREATE_BILLING_AGREEMENT_TOKEN_FAILURE_MESSAGE = "Billing agreement token creation failed";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalCreateBillingAgreementTokenCommand.class);

    public PayPalBillingAgreementTokenData perform(Boolean skipShippingAddress) {
        PayPalBillingAgreementRequestData data = createPayPalBillingAgreementRequest(skipShippingAddress);
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders headers = new HttpHeaders();
        String agreementTokenUrl = getPayPalApiEndpoint().concat(AGREEMENT_TOKENS);
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, createAPIContext().getAccessToken());
        HttpEntity<String> httpEntity = new HttpEntity<>(JSONFormatter.toJSON(data), headers);
        ResponseEntity<PayPalBillingAgreementTokenData> response;
        try {
            response = restTemplate
                .postForEntity(agreementTokenUrl, httpEntity, PayPalBillingAgreementTokenData.class);
        } catch (Exception e) {
            LOG.error(CREATE_BILLING_AGREEMENT_TOKEN_FAILURE_MESSAGE, e);
            throw new AdapterException(e.getMessage());
        }

        return response.getBody();
    }

    private PayPalBillingAgreementRequestData createPayPalBillingAgreementRequest(boolean skipShippingAddress) {
        PayPalBillingAgreementRequestData requestData = new PayPalBillingAgreementRequestData();
        String customDescription = getDefaultPayPalConfigurationService().getCustomBillingAgreementDescription();
        requestData.setDescription(customDescription);
        PayPalPayerData payerData = new PayPalPayerData();
        payerData.setPaymentMethod(PAYPAL_PAYMENT_METHOD);
        PayPalMerchantPreferencesData merchantPreferencesData = GenericBuilder.of(PayPalMerchantPreferencesData::new)
            .with(PayPalMerchantPreferencesData::setAcceptedPymtType, ACCEPTED_PYMT_TYPE_INSTANT)
            .with(PayPalMerchantPreferencesData::setReturnUrl, RETURN_URL)
            .with(PayPalMerchantPreferencesData::setCancelUrl, CANCEL_URL)
            .with(PayPalMerchantPreferencesData::setNotifyUrl, NOTIFY_URL)
            .with(PayPalMerchantPreferencesData::setSkipShippingAddress, String.valueOf(skipShippingAddress))
            .build();
        PayPalPlanData planData = new PayPalPlanData();
        planData.setType(PLAN_DATA_TYPE);
        planData.setMerchantPreferences(merchantPreferencesData);
        requestData.setPlan(planData);
        requestData.setPayer(payerData);
        return requestData;
    }
}
