package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.orders.Capture;
import com.paypal.orders.Order;
import com.paypal.orders.OrderCaptureRequest;
import com.paypal.orders.OrdersCaptureRequest;
import com.paypal.orders.PaymentSource;
import com.paypal.orders.Token;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.CaptureCommand;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.NoSuchElementException;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.BILLING_AGREEMENT;


public class DefaultPayPalCaptureIntentCommand extends PayPalAbstractCommand implements CaptureCommand {

    private static final String TRANSACTION_CAPTURE_FAILURE_MESSAGE = "Transaction capture failed: ";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalCaptureIntentCommand.class);

    @Override
    public CaptureResult perform(CaptureRequest request) {
        OrdersCaptureRequest ordersCaptureRequest = new OrdersCaptureRequest(
            (request.getRequestId()));
        HttpResponse<Order> response = null;
        if (request.getRequestToken() != null) {
            ordersCaptureRequest.requestBody(prepareRequestBody(request));
        }
        try {
            response = createClient().execute(ordersCaptureRequest);
        } catch (IOException e) {
            LOG.error(TRANSACTION_CAPTURE_FAILURE_MESSAGE, e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
        return translateResponse(response, request);
    }


    private CaptureResult translateResponse(HttpResponse<Order> response, CaptureRequest request) {
        String resultStatus = response.result().status();
        Order capture = response.result();
        CaptureResult result = getCaptureResult(request, resultStatus, capture);
        return result;
    }

    private CaptureResult getCaptureResult(CaptureRequest request, String resultStatus, Order order) {
        final TransactionStatus transactionStatus = getTransactionStatusMap()
            .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
            getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

        final String requestId = order.purchaseUnits().stream()
            .flatMap(unit -> unit.payments().captures().stream())
            .map(Capture::id)
            .findFirst().orElseThrow(() -> new NoSuchElementException("Capture id not found"));

        return GenericBuilder.of(CaptureResult::new)
            .with(CaptureResult::setRequestId, requestId)
            .with(CaptureResult::setCurrency, request.getCurrency())
            .with(CaptureResult::setTotalAmount, request.getTotalAmount())
            .with(CaptureResult::setReconciliationId, request.getSubscriptionID())
            .with(CaptureResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
            .with(CaptureResult::setTransactionStatus, transactionStatus)
            .with(CaptureResult::setTransactionStatusDetails, transactionStatusDetails)
            .build();
    }

    private OrderCaptureRequest prepareRequestBody(CaptureRequest request) {
        OrderCaptureRequest orderCaptureRequest = new OrderCaptureRequest();
        PaymentSource paymentSource = new PaymentSource();
        Token token = new Token();
        token.id(request.getRequestToken());
        token.type(BILLING_AGREEMENT);
        paymentSource.token(token);
        orderCaptureRequest.paymentSource(paymentSource);
        return orderCaptureRequest;
    }
}
