package com.paypal.hybris.backoffice.widgets.order.reauthorize;

import com.google.common.base.Preconditions;
import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.core.service.PayPalPaymentService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.util.Locales;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Currency;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_AMOUNT_NOT_MATCH_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_ZERO_AMOUNT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;


public class PayPalReauthorizeController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalReauthorizeController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE = "orderModel can not be null";

    private OrderModel order;

    @Wire
    private Textbox orderCode;
    @Wire
    private Textbox customer;
    @Wire
    private Textbox amount;
    @Wire
    private Combobox transactions;

    @WireVariable
    private transient PayPalPaymentService paymentService;

    @WireVariable
    private transient ModelService modelService;

    @WireVariable
    private BusinessProcessService businessProcessService;

    @WireVariable
    private CommandFactoryRegistry commandFactoryRegistry;

    @SocketEvent(socketId = IN_SOCKET)
    public void initReauthorizeRequestForm(OrderModel inputOrder) {
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager()
            .setTitle(this.getWidgetInstanceManager().getLabel(REAUTHORIZE_TITLE)
                + " " + this.getOrder().getCode());
        this.orderCode.setValue(this.getOrder().getCode());
        this.customer.setValue(this.getOrder().getUser().getDisplayName());
        this.amount.setValue(formatAmount(getAmountAvailableForReauthorize()));
        configureTransactionsCombobox(inputOrder);
    }

    private void configureTransactionsCombobox(final OrderModel inputOrder) {
        List<PaymentTransactionEntryModel> trans = getAuthorizeTransactions(inputOrder);
        LOG.info("Authorization transactions: " + trans);

        for (PaymentTransactionEntryModel v : trans) {
            Comboitem ci = new Comboitem();
            ci.setValue(v);
            ci.setLabel(v.getRequestId());

            transactions.appendChild(ci);
        }

        if (!transactions.getItems().isEmpty()) {
            transactions.setSelectedItem(transactions.getItems().stream().findFirst().get());
        }
    }

    @ViewEvent(componentID = "transactions", eventName = "onSelect")
    public void selectTransactionEntry() {
        Comboitem ci = transactions.getSelectedItem();
        amount.setValue(formatAmount(((PaymentTransactionEntryModel) ci.getValue()).getAmount()));
    }

    @ViewEvent(componentID = "reauthorizerequest", eventName = "onClick")
    public void confirm() {
        PaymentTransactionEntryModel selectedEntry = new PaymentTransactionEntryModel();
        if (transactions.getSelectedIndex() != -1) {
            selectedEntry = transactions.getSelectedItem().getValue();
        }
        validateAmount();
        getTransactionInfo(selectedEntry);
    }

    private void getTransactionInfo(PaymentTransactionEntryModel selectedEntry) {
        final BigDecimal transactionAmount = new BigDecimal(this.amount.getValue());

        try {
            Boolean isComplete = reauthorize(selectedEntry, transactionAmount);
            if (isComplete) {
                Messagebox.show(
                        getLabel(REAUTHORIZE_SUCCESS),
                        getLabel(REAUTHORIZE_TITLE) + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.INFORMATION);
                sendOutput(OUT_MODIFIED_ITEM, order);
            } else {
                Messagebox
                    .show(
                            getLabel(REAUTHORIZE_AMOUNT_NOT_MATCH_ERROR) + " " +
                                    formatAmount(getAmountAvailableForReauthorize()),
                            getLabel(REAUTHORIZE_TITLE) + " " + this.getOrder().getCode(),
                            Messagebox.OK,
                            Messagebox.ERROR);
            }
        } catch (Exception e) {
            String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            Messagebox.show(e.getMessage(), getLabel(REAUTHORIZE_ERROR), Messagebox.OK, Messagebox.ERROR);
        }
    }

    private Boolean reauthorize(final PaymentTransactionEntryModel selectedEntry, final BigDecimal amount) {
        if (order != null) {
            AuthorizationResult authorizationResult = paymentService
                .reauthorize(prepareReauthorizeRequest(amount, selectedEntry));

            PaymentTransactionModel paymentTransactionModel = selectedEntry.getPaymentTransaction();

            selectedEntry.setAmount(amount);
            selectedEntry.setRequestId(authorizationResult.getAuthorizationCode());
            modelService.save(selectedEntry);
            modelService.refresh(paymentTransactionModel);

            paymentTransactionModel.setPlannedAmount(getAuthorizedAmount());
            paymentTransactionModel.setRequestId(authorizationResult.getAuthorizationCode());

            modelService.save(paymentTransactionModel);
            modelService.saveAll();
        }
        return Boolean.TRUE;
    }

    private SubscriptionAuthorizationRequest prepareReauthorizeRequest(final BigDecimal amount,
        final PaymentTransactionEntryModel selectedEntry) {
        final String merchantTransactionCode = selectedEntry.getPaymentTransaction().getCode();
        final Currency currency = Currency.getInstance(selectedEntry.getCurrency().getIsocode());
        final String paymentProvider = selectedEntry.getPaymentTransaction().getPaymentProvider();
        return new SubscriptionAuthorizationRequest(merchantTransactionCode,
            selectedEntry.getRequestId(), currency, amount, null, paymentProvider);
    }

    private String formatAmount(final BigDecimal amount) {
        final DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(Locales.getCurrent());
        decimalFormat.applyPattern("#0.00");
        return decimalFormat.format(amount);
    }

    private BigDecimal getAmountAvailableForReauthorize() {
        if (order != null) {
            return getPossibleAmountForReauthorize(order);
        }
        return BigDecimal.ZERO;
    }

    private void validateAmount() {
        String value = amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel(REAUTHORIZE_EMPTY_AMOUNT));
        }
        try {
            if (BigDecimal.ZERO.equals(new BigDecimal(value))) {
                throw new WrongValueException(amount, getLabel(REAUTHORIZE_ZERO_AMOUNT));
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel(REAUTHORIZE_INVALID_FORMAT_AMOUNT));
        }
    }

    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }

    private List<PaymentTransactionEntryModel> getAuthorizeTransactions(final OrderModel order) {
        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(paymentEntry -> (PaymentTransactionType.AUTHORIZATION.equals(paymentEntry.getType())
                && TransactionStatus.ACCEPTED.name().equals(paymentEntry.getTransactionStatus())))
            .collect(Collectors.toList());

    }

    private BigDecimal getPossibleAmountForReauthorize(OrderModel orderModel) {
        validateParameterNotNull(orderModel, ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE);
        return new BigDecimal(orderModel.getTotalPrice());
    }

    private void validateParameterNotNull(Object parameter, String nullMessage) {
        Preconditions.checkArgument(parameter != null, nullMessage);
    }

    private BigDecimal getAuthorizedAmount() {
        modelService.refresh(getOrder());
        Predicate<PaymentTransactionEntryModel> authorizationEntries = (entry -> (PaymentTransactionType.AUTHORIZATION
            .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()));

        return getOrder().getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(authorizationEntries).map(PaymentTransactionEntryModel::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
    }
}
