package com.paypal.hybris.backoffice.widgets.order.authorize;

import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.core.service.PayPalManualAuthorizationService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_ZERO_AMOUNT;


public class PayPalAuthorizationController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalAuthorizationController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String CONFIRM_TITLE = "paypalbackoffice.authorizeorder.confirm.title";

    private OrderModel order;
    @Wire
    private Textbox orderCode;
    @Wire
    private Textbox customer;
    @Wire
    private Textbox amount;


    @WireVariable
    PayPalManualAuthorizationService payPalManualAuthorizationService;

    @SocketEvent(socketId = IN_SOCKET)
    public void initCreateAuthorizeRequestForm(OrderModel inputOrder) {
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager().setTitle(
            this.getWidgetInstanceManager().getLabel(CONFIRM_TITLE) + " " + this
                .getOrder()
                .getCode());
        this.orderCode.setValue(this.getOrder().getCode());
        this.customer.setValue(this.getOrder().getUser().getDisplayName());
        this.amount.setValue(this.getOrder().getTotalPrice().toString());

    }

    @ViewEvent(componentID = "authorizeorderrequest", eventName = "onClick")
    public void confirm() {
        validateAmount();
        final BigDecimal confirmedAmount = new BigDecimal(this.amount.getValue());
        processAuthorization(confirmedAmount);
        sendOutput(OUT_MODIFIED_ITEM, order);
    }

    private void processAuthorization(BigDecimal amount) {
        try {
            final PaymentTransactionEntryModel paymentTransactionEntryModel = payPalManualAuthorizationService
                .doAuthorization(getOrder(), amount);

            if (TransactionStatus.ACCEPTED.toString().equals(paymentTransactionEntryModel.getTransactionStatus())) {

                Messagebox
                    .show(
                            getLabel(MANUAL_AUTHORIZATION_SUCCESS),
                            getLabel(MANUAL_AUTHORIZATION_TITLE) + " " + this.getOrder().getCode(),
                            Messagebox.OK,
                            Messagebox.INFORMATION);
            } else {
                Messagebox.show(
                        getLabel(MANUAL_AUTHORIZATION_ERROR) + " : " + paymentTransactionEntryModel
                            .getTransactionStatusDetails(),
                        getLabel(MANUAL_AUTHORIZATION_TITLE) + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.ERROR);
            }

        } catch (final AdapterException e) {
            final String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            Messagebox.show(
                    "Error message: " + e.getMessage(),
                    getLabel(MANUAL_AUTHORIZATION_TITLE)+ " " + this.getOrder().getCode(),
                    Messagebox.OK,
                    Messagebox.ERROR);
        }
    }

    private void validateAmount() {
        final String value = amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_EMPTY_AMOUNT));
        }
        try {
            if (BigDecimal.ZERO.equals(new BigDecimal(value))) {
                throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_ZERO_AMOUNT));
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT));
        }
    }

    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }

}
