/*

 */
package com.paypal.hybris.paypalb2bregistration2205addon.b2bfacade;

import com.paypal.hybris.b2bfacade.PayPalB2BRegistrationUserFacade;
import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.data.PayPalConnectB2BRegisterData;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.data.PayPalUserExistB2BData;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import de.hybris.platform.b2b.model.B2BCustomerModel;
import de.hybris.platform.cms2.servicelayer.services.CMSSiteService;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.b2b.model.B2BRegistrationModel;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.store.services.BaseStoreService;
import de.hybris.platform.tx.Transaction;
import de.hybris.platform.workflow.WorkflowProcessingService;
import de.hybris.platform.workflow.WorkflowService;
import de.hybris.platform.workflow.WorkflowTemplateService;
import de.hybris.platform.workflow.model.WorkflowModel;
import de.hybris.platform.workflow.model.WorkflowTemplateModel;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.BeanUtils;

/**
 * This class is a default implementation of the PayPalB2BRegistrationUserFacade interface
 */
public class DefaultPayPalB2BRegistrationUserFacade implements PayPalB2BRegistrationUserFacade {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalB2BRegistrationUserFacade.class);

    private static final String REGISTRATION_WORKFLOW = "B2BUserRegistration";
    private static final String USER_ALREADY_EXIST_MESSAGE = "User with uid '%s' already exists!";
    private static final String CREATE_WORKFLOW_TEMPLATE_MESSAGE = "User with uid '%s' already exists!";
    private static final String STARTING_WORKFLOW_MESSAGE = "Starting workflow for customer with email '%s' and organization (company name) '%s'.";

    private PayPalRegistrationUserFacade registrationUserFacade;
    private PayPalConnectService payPalConnectService;
    private ModelService modelService;
    private UserService userService;
    private CommonI18NService commonI18NService;
    private WorkflowTemplateService workflowTemplateService;
    private WorkflowService workflowService;
    private WorkflowProcessingService workflowProcessingService;
    private BaseStoreService baseStoreService;
    private CMSSiteService cmsSiteService;
    private PayPalCustomerAccountDao payPalCustomerAccountDao;

    @Override
    public String registerB2BPayPalUser(PayPalConnectB2BRegisterData registerData) throws DuplicateUidException {
        final PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(registerData.getAccessToken());
        registerData.setEmail(userData.getEmails().stream().findFirst().get().getValue().toLowerCase());
        registerData.setName(userData.getName());

        final Transaction tx = Transaction.current();
        tx.begin();
        boolean success = false;
        String payerId = StringUtils.EMPTY;
        try {
            final boolean userExists = userService.isUserExisting(registerData.getEmail());
            if (userExists) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug(String.format(USER_ALREADY_EXIST_MESSAGE, registerData.getEmail()));
                }
                throw new DuplicateUidException(
                    String.format(USER_ALREADY_EXIST_MESSAGE, registerData.getEmail()));
            }

            final B2BRegistrationModel registration = toRegistrationModel(registerData);
            modelService.save(registration);

            final CustomerModel customer = toCustomerModel(registerData);
            modelService.save(customer);

            final WorkflowTemplateModel workflowTemplate = workflowTemplateService
                .getWorkflowTemplateForCode(REGISTRATION_WORKFLOW);

            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format(CREATE_WORKFLOW_TEMPLATE_MESSAGE,
                    REGISTRATION_WORKFLOW));
            }

            final WorkflowModel workflow = workflowService.createWorkflow(workflowTemplate, registration,
                userService.getAdminUser());
            modelService.save(workflow);

            if (LOG.isDebugEnabled()) {
                LOG.debug(String
                    .format(STARTING_WORKFLOW_MESSAGE,
                        registration.getEmail(), registration.getCompanyName()));
            }

            workflowProcessingService.startWorkflow(workflow);
            tx.commit();
            success = true;
        } catch(Exception e) {
            String message = "Can not register B2B PayPal user " + e.getMessage();
            LOG.error(message);
        } finally {
            if (!success) {
                tx.rollback();
            }
        }

        if (success) {
            registrationUserFacade
                .setPayerIdToUser(userData.getEmails().get(0).getValue().toLowerCase(), userData.getPayer_id());
            payerId = userData.getPayer_id();
        }
        return payerId;
    }

    protected B2BRegistrationModel toRegistrationModel(final PayPalConnectB2BRegisterData data) {
        final B2BRegistrationModel model = modelService.create(B2BRegistrationModel.class);
        BeanUtils.copyProperties(data, model, "titleCode", "companyAddressCountryIso", "companyAddressRegion",
            "baseStore", "cmsSite", "currency",
            "language");

        if (StringUtils.isNotBlank(data.getTitleCode())) {
            final TitleModel title = userService.getTitleForCode(data.getTitleCode());
            model.setTitle(title);
        }

        final CountryModel country = commonI18NService.getCountry(data.getCompanyAddressCountryIso());
        model.setCompanyAddressCountry(country);
        model.setBaseStore(baseStoreService.getCurrentBaseStore());
        model.setBaseSite(cmsSiteService.getCurrentSite());
        model.setCurrency(commonI18NService.getCurrentCurrency());
        model.setLanguage(commonI18NService.getCurrentLanguage());
        return model;
    }

    protected CustomerModel toCustomerModel(PayPalConnectB2BRegisterData registerData) {
        final CustomerModel model = modelService.create(CustomerModel.class);

        model.setName(WordUtils.capitalizeFully(registerData.getName()));
        model.setUid(registerData.getEmail());
        model.setLoginDisabled(true);
        model.setSessionLanguage(commonI18NService.getCurrentLanguage());
        model.setSessionCurrency(commonI18NService.getCurrentCurrency());

        if (StringUtils.isNotBlank(registerData.getTitleCode())) {
            final TitleModel title = userService.getTitleForCode(registerData.getTitleCode());
            model.setTitle(title);
        }

        return model;
    }

    @Override
    public PayPalUserExistB2BData isCustomerExistAndApprovedByPayerId(String payerId) {
        final CustomerModel customer = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        final PayPalUserExistB2BData userExistB2BData = new PayPalUserExistB2BData();
        userExistB2BData.setIsExist(customer != null);
        userExistB2BData.setIsApproved(false);
        if (userExistB2BData.getIsExist()) {
            userExistB2BData.setIsApproved(customer.getItemModelContext().getItemType().equals(B2BCustomerModel._TYPECODE));
        }
        return userExistB2BData;
    }

    public PayPalRegistrationUserFacade getRegistrationUserFacade() {
        return registrationUserFacade;
    }

    public void setRegistrationUserFacade(PayPalRegistrationUserFacade registrationUserFacade) {
        this.registrationUserFacade = registrationUserFacade;
    }

    public CommonI18NService getCommonI18NService() {
        return commonI18NService;
    }

    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }

    public WorkflowTemplateService getWorkflowTemplateService() {
        return workflowTemplateService;
    }

    public void setWorkflowTemplateService(WorkflowTemplateService workflowTemplateService) {
        this.workflowTemplateService = workflowTemplateService;
    }

    public WorkflowService getWorkflowService() {
        return workflowService;
    }

    public void setWorkflowService(WorkflowService workflowService) {
        this.workflowService = workflowService;
    }

    public WorkflowProcessingService getWorkflowProcessingService() {
        return workflowProcessingService;
    }

    public void setWorkflowProcessingService(WorkflowProcessingService workflowProcessingService) {
        this.workflowProcessingService = workflowProcessingService;
    }

    public BaseStoreService getBaseStoreService() {
        return baseStoreService;
    }

    public void setBaseStoreService(BaseStoreService baseStoreService) {
        this.baseStoreService = baseStoreService;
    }

    public CMSSiteService getCmsSiteService() {
        return cmsSiteService;
    }

    public void setCmsSiteService(CMSSiteService cmsSiteService) {
        this.cmsSiteService = cmsSiteService;
    }

    public PayPalConnectService getPayPalConnectService() {
        return payPalConnectService;
    }

    public void setPayPalConnectService(PayPalConnectService payPalConnectService) {
        this.payPalConnectService = payPalConnectService;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public UserService getUserService() {
        return userService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public PayPalCustomerAccountDao getPayPalCustomerAccountDao() {
        return payPalCustomerAccountDao;
    }

    public void setPayPalCustomerAccountDao(PayPalCustomerAccountDao payPalCustomerAccountDao) {
        this.payPalCustomerAccountDao = payPalCustomerAccountDao;
    }
}
