/**
 *
 */
package com.paypal.hybris.controllers;


import de.hybris.platform.b2bacceleratorfacades.api.cart.CartFacade;
import de.hybris.platform.b2bacceleratorfacades.checkout.data.PlaceOrderData;
import de.hybris.platform.b2bacceleratorfacades.order.impl.DefaultB2BAcceleratorCheckoutFacade;
import de.hybris.platform.b2bacceleratorservices.enums.CheckoutPaymentType;
import de.hybris.platform.b2bocc.v2.controllers.BaseController;
import de.hybris.platform.b2bwebservicescommons.dto.order.ReplenishmentOrderWsDTO;
import de.hybris.platform.b2bwebservicescommons.dto.order.ScheduleReplenishmentFormWsDTO;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commerceservices.request.mapping.annotation.ApiVersion;
import de.hybris.platform.commercewebservicescommons.errors.exceptions.PaymentAuthorizationException;
import de.hybris.platform.commercewebservicescommons.errors.exceptions.RequestParameterException;
import de.hybris.platform.commercewebservicescommons.strategies.CartLoaderStrategy;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.webservicescommons.errors.exceptions.WebserviceValidationException;
import de.hybris.platform.webservicescommons.mapping.FieldSetLevelHelper;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.annotation.Secured;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE;
import static de.hybris.platform.util.localization.Localization.getLocalizedString;


@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}/paypal/orders")
@ApiVersion("v2")
@Api(tags = "PayPal Orders")
public class PayPalB2BOrdersController extends BaseController {

    private static final String REPLENISHMENT_ORDER_FIELDS = "active,trigger(DEFAULT),firstDate,replenishmentOrderCode,"
        + "potentialProductPromotions(DEFAULT),potentialOrderPromotions(DEFAULT),totalUnitCount,store,appliedOrderPromotions(DEFAULT),"
        + "pickupItemsQuantity,net,calculated,appliedVouchers(DEFAULT),productDiscounts(DEFAULT),totalDiscounts(DEFAULT),"
        + "subTotal(DEFAULT),orderDiscounts(DEFAULT),entries(DEFAULT),appliedProductPromotions(DEFAULT),totalPrice(DEFAULT),"
        + "site,deliveryMode(DEFAULT),code,deliveryOrderGroups(DEFAULT),guid,paymentInfo(FULL),pickupOrderGroups(DEFAULT),"
        + "totalItems,totalPriceWithTax(DEFAULT),deliveryItemsQuantity,totalTax(DEFAULT),user(DEFAULT),deliveryAddress(DEFAULT),"
        + "deliveryCost(DEFAULT),name,description,savedBy,saveTime,expirationTime";

    protected static final String API_COMPATIBILITY_B2B_CHANNELS = "api.compatibility.b2b.channels";
    private static final String CART_CHECKOUT_TERM_UNCHECKED = "cart.term.unchecked";
    private static final String OBJECT_NAME_SCHEDULE_REPLENISHMENT_FORM = "ScheduleReplenishmentForm";

    @Resource(name = "userFacade")
    protected UserFacade userFacade;

    @Resource(name = "defaultB2BAcceleratorCheckoutFacade")
    private DefaultB2BAcceleratorCheckoutFacade b2bCheckoutFacade;

    @Resource(name = "b2bCartFacade")
    private CartFacade cartFacade;

    @Resource(name = "cartLoaderStrategy")
    private CartLoaderStrategy cartLoaderStrategy;

    @Resource(name = "b2BPlaceOrderCartValidator")
    private Validator placeOrderCartValidator;

    @Resource(name = "scheduleReplenishmentFormWsDTOValidator")
    private Validator scheduleReplenishmentFormWsDTOValidator;

    @Resource(name = "sessionService")
    private SessionService sessionService;

    @Secured({ "ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @PostMapping(value = "/replenishmentOrders")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiBaseSiteIdAndUserIdParam
    @ApiOperation(nickname = "createReplenishmentOrder", value = "Creates an Order and schedules Replenishment.", notes = "Creates an Order and schedules Replenishment. By default the payment type is ACCOUNT. Please set payment type to CARD if placing an order using credit card.", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ReplenishmentOrderWsDTO createReplenishmentOrder(
            @ApiParam(value = "Cart identifier: cart code for logged in user, cart guid for anonymous user, 'current' for the last modified cart", required = true) @RequestParam(required = true) final String cartId,
            @ApiParam(value = "Whether terms were accepted or not.", required = true) @RequestParam(required = true) final boolean termsChecked,
            @ApiParam(value = "Schedule replenishment form object.", required = true) @RequestBody(required = true) final ScheduleReplenishmentFormWsDTO scheduleReplenishmentForm,
            @ApiFieldsParam @RequestParam(required = false, defaultValue = FieldSetLevelHelper.DEFAULT_LEVEL) final String fields)
            throws InvalidCartException, PaymentAuthorizationException
    {

        validateTerms(termsChecked);

        validateUser();

        cartLoaderStrategy.loadCart(cartId);
        final CartData cartData = cartFacade.getCurrentCart();

        getSessionService().setAttribute(PAYPAL_REPLENISHMENT_FLOW_SESSION_ATTRIBUTE, Boolean.TRUE);

        validateCart(cartData);
        validateAndAuthorizePayment(cartData);

        validateScheduleReplenishmentForm(scheduleReplenishmentForm);
        final PlaceOrderData placeOrderData = createPlaceOrderData(scheduleReplenishmentForm);

        return getDataMapper().map(b2bCheckoutFacade.placeOrder(placeOrderData), ReplenishmentOrderWsDTO.class, REPLENISHMENT_ORDER_FIELDS);
    }


    protected void validateUser()
    {
        if (userFacade.isAnonymousUser())
        {
            throw new AccessDeniedException("Access is denied");
        }
    }

    protected void validateTerms(final boolean termsChecked)
    {
        if (!termsChecked)
        {
            throw new RequestParameterException(getLocalizedString(CART_CHECKOUT_TERM_UNCHECKED));
        }
    }

    protected void validateScheduleReplenishmentForm(ScheduleReplenishmentFormWsDTO scheduleReplenishmentForm)
    {
        validate(scheduleReplenishmentForm, OBJECT_NAME_SCHEDULE_REPLENISHMENT_FORM, scheduleReplenishmentFormWsDTOValidator);
    }

    protected void validateAndAuthorizePayment(final CartData cartData)
            throws PaymentAuthorizationException
    {
        if (CheckoutPaymentType.CARD.getCode().equals(cartData.getPaymentType().getCode()) && !b2bCheckoutFacade.authorizePayment(null))
        {
            throw new PaymentAuthorizationException();
        }
    }

    protected void validateCart(final CartData cartData)
    {
        final Errors errors = new BeanPropertyBindingResult(cartData, "sessionCart");
        placeOrderCartValidator.validate(cartData, errors);
        if (errors.hasErrors())
        {
            throw new WebserviceValidationException(errors);
        }
    }

    protected PlaceOrderData createPlaceOrderData(final ScheduleReplenishmentFormWsDTO scheduleReplenishmentForm)
    {
        final PlaceOrderData placeOrderData = new PlaceOrderData();
        getDataMapper().map(scheduleReplenishmentForm, placeOrderData, false);
        if (scheduleReplenishmentForm != null)
        {
            placeOrderData.setReplenishmentOrder(Boolean.TRUE);
        }
        placeOrderData.setTermsCheck(Boolean.TRUE);
        return placeOrderData;
    }

    public UserFacade getUserFacade() {
        return userFacade;
    }

    public void setUserFacade(UserFacade userFacade) {
        this.userFacade = userFacade;
    }

    public DefaultB2BAcceleratorCheckoutFacade getB2bCheckoutFacade() {
        return b2bCheckoutFacade;
    }

    public void setB2bCheckoutFacade(DefaultB2BAcceleratorCheckoutFacade b2bCheckoutFacade) {
        this.b2bCheckoutFacade = b2bCheckoutFacade;
    }

    public CartFacade getCartFacade() {
        return cartFacade;
    }

    public void setCartFacade(CartFacade cartFacade) {
        this.cartFacade = cartFacade;
    }

    public CartLoaderStrategy getCartLoaderStrategy() {
        return cartLoaderStrategy;
    }

    public void setCartLoaderStrategy(CartLoaderStrategy cartLoaderStrategy) {
        this.cartLoaderStrategy = cartLoaderStrategy;
    }

    public Validator getPlaceOrderCartValidator() {
        return placeOrderCartValidator;
    }

    public void setPlaceOrderCartValidator(Validator placeOrderCartValidator) {
        this.placeOrderCartValidator = placeOrderCartValidator;
    }

    public Validator getScheduleReplenishmentFormWsDTOValidator() {
        return scheduleReplenishmentFormWsDTOValidator;
    }

    public void setScheduleReplenishmentFormWsDTOValidator(Validator scheduleReplenishmentFormWsDTOValidator) {
        this.scheduleReplenishmentFormWsDTOValidator = scheduleReplenishmentFormWsDTOValidator;
    }

    public SessionService getSessionService() {
        return sessionService;
    }

    public void setSessionService(SessionService sessionService) {
        this.sessionService = sessionService;
    }
}
