/**
 *
 */
package com.paypal.hybris.controllers;

import com.paypal.hybris.data.ws.PayPalBillingAgreementPaymentRequestWsDTO;
import com.paypal.hybris.data.ws.PayPalBillingAgreementTokenWsDTO;
import com.paypal.hybris.data.ws.PayPalOrderWsDTO;
import com.paypal.hybris.data.ws.PayPalPaymentRequestWsDTO;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalBillingAgreementData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalCheckoutFacade;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.commercewebservicescommons.dto.order.PaymentDetailsWsDTO;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdUserIdAndCartIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import de.hybris.platform.ycommercewebservices.exceptions.InvalidPaymentInfoException;
import de.hybris.platform.ycommercewebservices.v2.controller.BaseCommerceController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.annotation.Secured;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;

@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}/carts/{cartId}")
@CacheControl(directive = CacheControlDirective.NO_CACHE)
@Api(tags = "Braintree Payment Details")
public class PayPalB2BCheckoutController extends BaseCommerceController {

    private static final String ANONYMOUS_USER_UID = "anonymous";
    private static final String PAYPAL_ORDER_ID_PLACEHOLDER = "ORDER_ID_PLACEHOLDER";
    private static final String EXPRESS_CHECKOUT = "EXPRESS_CHECKOUT";
    private static final String MARK_CHECKOUT = "MARK_CHECKOUT";
    private static final String ANONYMOUS_CHECKOUT = "anonymous_checkout";
    private static final String ANONYMOUS_CHECKOUT_GUID = "anonymous_checkout_guid";
    private static final String DEFAULT_PAYMENT_METHOD = "paypal";

    private static final Logger LOG = Logger.getLogger(PayPalB2BCheckoutController.class);

    @Resource(name = "payPalCheckoutFacade")
    private DefaultPayPalCheckoutFacade checkoutFacade;

    @Resource(name = "customerFacade")
    private CustomerFacade customerFacade;

    @Resource(name = "sessionService")
    private SessionService sessionService;

    @Resource(name = "payPalPaymentRequestValidator")
    private Validator payPalPaymentRequestValidator;

    @Resource(name = "payPalBillingAgreementPaymentRequestValidator")
    private Validator payPalBillingAgreementPaymentRequestValidator;

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP",
        "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/createOrder")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "createPayPalOrder", value = "Create order on PayPal side", notes = "Create and save new PayPal oder")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public PayPalOrderWsDTO createPayPalOrder(
        @RequestParam(required = true) final String fundingSource) {
        final String orderID = checkoutFacade.createPayPalOrder();
        CreditCardType paymentType = checkoutFacade.setCreditCardType(fundingSource);
        CCPaymentInfoData payPalPaymentSubscription = checkoutFacade
            .createPayPalPaymentSubscriptionForMarkCheckout(orderID, paymentType);
        checkoutFacade.setPaymentDetails(payPalPaymentSubscription.getId());
        return GenericBuilder.of(PayPalOrderWsDTO::new)
            .with(PayPalOrderWsDTO::setOrderId, orderID).build();
    }


    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP",
        "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/createPayPalPayment")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public PaymentDetailsWsDTO createPayPalPaymentMethod(
        @ApiParam(value = "Request body parameter that contains" +
            "payPalOrderId and payPAlFlow option ", required = true) @RequestBody final PayPalPaymentRequestWsDTO payPalRequest,
        @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
            allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
        @ApiParam(value = "Payment method type") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_PAYMENT_METHOD) final String paymentMethodType) {
        validate(payPalRequest, "payPalRequest", payPalPaymentRequestValidator);
        final String orderId = payPalRequest.getOrderId();
        final String checkoutType = payPalRequest.getPayPalFlow();
        final PayPalOrderDetailsData orderDetailsData = checkoutFacade.getPayPalOrderDetails(
            orderId);
        if (StringUtils.equals(EXPRESS_CHECKOUT, checkoutType)) {
            if (getUserFacade().isAnonymousUser()) {
                processCheckoutForAnonymous(orderDetailsData.getShippingAddress());
            }
            checkoutFacade.processExpressCheckout(orderDetailsData, paymentMethodType);
        } else if (StringUtils.equals(MARK_CHECKOUT, checkoutType)) {
            CCPaymentInfoData payPalPaymentSubscription = checkoutFacade
                .updatePayPalPaymentSubscription(orderDetailsData,
                    checkoutFacade.getCheckoutCart().getPaymentInfo());
            checkoutFacade.setPaymentDetails(payPalPaymentSubscription.getId());
        }

        return getDataMapper().map(checkoutFacade.getCheckoutCart().getPaymentInfo(),
            PaymentDetailsWsDTO.class, fields);
    }

    private void processCheckoutForAnonymous(final PayPalAddressDetailsData addressDetailsData) {
        try {
            if (ANONYMOUS_USER_UID.equals(checkoutFacade.getCurrentSessionUserUid())) {
                getCustomerFacade().createGuestUserForAnonymousCheckout(
                    addressDetailsData.getEmail(),
                    addressDetailsData.getFirstName());
            } else if (!addressDetailsData.getEmail()
                .equalsIgnoreCase(
                    StringUtils.substringAfter(checkoutFacade.getCurrentSessionUserUid(), "|"))) {
                getCustomerFacade().createGuestUserForAnonymousCheckout(
                    addressDetailsData.getEmail(),
                    addressDetailsData.getFirstName());
                getSessionService().setAttribute(ANONYMOUS_CHECKOUT_GUID,
                    StringUtils.substringBefore(checkoutFacade.getCurrentSessionUserUid(), "|"));
            }
            getSessionService().setAttribute(ANONYMOUS_CHECKOUT, true);
        } catch (final DuplicateUidException e) {
            LOG.warn("Guest registration failed, PayPal email: " + addressDetailsData.getEmail()
                + ".\n details" + e);
        }
    }


    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP",
        "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/createBillingAgreementToken")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "create billing agreement token", value = "Create new billing agreement tok", notes = "Create and save new billing agreement token")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public PayPalBillingAgreementTokenWsDTO createBillingAgreementToken() {
        boolean skipShipping = !checkoutFacade.hasNoDeliveryAddress();
        return GenericBuilder.of(PayPalBillingAgreementTokenWsDTO::new)
            .with(PayPalBillingAgreementTokenWsDTO::setTokenId,
                checkoutFacade.createBillingAgreementToken(skipShipping)).build();
    }


    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP",
        "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/createPayPalBillingAgreementPayment")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public PaymentDetailsWsDTO createPayPalBillingAgreementPayment(
        @ApiParam(value = "Request body parameter that contains" +
            " billingAgreementToken and isSavePaymentInfo", required = true) @RequestBody final PayPalBillingAgreementPaymentRequestWsDTO payPalBillingAgreementPaymentRequest,
        @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
            allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
        @ApiParam(value = "Payment method type") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_PAYMENT_METHOD) final String paymentMethodType) {
        validate(payPalBillingAgreementPaymentRequest, "payPalRequest",
            payPalBillingAgreementPaymentRequestValidator);
        final String billingAgreementToken = payPalBillingAgreementPaymentRequest.getBillingAgreementTokenId();
        final String isSavePaymentInfo = payPalBillingAgreementPaymentRequest.getIsSavePaymentInfo();
        final PayPalBillingAgreementData billingAgreement = checkoutFacade.createBillingAgreement(
            billingAgreementToken);
        final CCPaymentInfoData payPalPaymentSubscription = checkoutFacade
            .createPayPalPaymentSubscription(billingAgreement, PAYPAL_ORDER_ID_PLACEHOLDER,
                Boolean.valueOf(isSavePaymentInfo), paymentMethodType);
        checkoutFacade.setPaymentDetails(payPalPaymentSubscription.getId());
        checkoutFacade.updateCheckoutPaymentInfoOrderId(PAYPAL_ORDER_ID_PLACEHOLDER);
        return getDataMapper().map(payPalPaymentSubscription, PaymentDetailsWsDTO.class, fields);
    }


    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP",
        "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/payPalPayment")
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdUserIdAndCartIdParam
    @ResponseStatus(HttpStatus.OK)
    public PaymentDetailsWsDTO processExpressCheckout() {
        return getDataMapper().map(checkoutFacade.processBillingAgreementExpressCheckout(),
            PaymentDetailsWsDTO.class);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP",
        "ROLE_TRUSTED_CLIENT"})
    @RequestMapping(value = "/paypal/paymentdetails", method = RequestMethod.PUT)
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "replaceCartPaymentDetails", value = "Sets credit card payment details for the cart.", notes = "Sets credit card payment details for the cart.")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public void replaceCartPaymentDetails(
        @ApiParam(value = "Payment details identifier.", required = true) @RequestParam final String paymentDetailsId)
        throws InvalidPaymentInfoException {
        checkoutFacade.setPaymentDetails(paymentDetailsId);
        checkoutFacade.updateCheckoutPaymentInfoOrderId(PAYPAL_ORDER_ID_PLACEHOLDER);
    }

    @Override
    public DefaultPayPalCheckoutFacade getCheckoutFacade() {
        return checkoutFacade;
    }

    public void setCheckoutFacade(DefaultPayPalCheckoutFacade checkoutFacade) {
        this.checkoutFacade = checkoutFacade;
    }

    public CustomerFacade getCustomerFacade() {
        return customerFacade;
    }

    public void setCustomerFacade(CustomerFacade customerFacade) {
        this.customerFacade = customerFacade;
    }

    public SessionService getSessionService() {
        return sessionService;
    }

    public void setSessionService(SessionService sessionService) {
        this.sessionService = sessionService;
    }
}
