package com.paypal.hybris.addon.controllers.pages;

import de.hybris.platform.acceleratorstorefrontcommons.annotations.PreValidateQuoteCheckoutStep;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.RequireHardLogIn;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.acceleratorstorefrontcommons.forms.PlaceOrderForm;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commerceservices.order.CommerceCartModificationException;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.yacceleratorstorefront.controllers.pages.checkout.steps.SummaryCheckoutStepController;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;


@Controller
@RequestMapping(value = "/paypal/checkout/multi/summary/")
public class PayPalSummaryCheckoutStepController extends SummaryCheckoutStepController {

    private static final String CHECKOUT_AUTHORIZATION_FAILED = "checkout.error.authorization.failed";
    private static final String CHECKOUT_PLACE_ORDER_FAILED = "checkout.placeOrder.failed";
    private static final String CHECKOUT_DELIVERY_ADDRESS_NOT_SELECTED = "checkout.deliveryAddress.notSelected";
    private static final String CHECKOUT_DELIVERY_METHOD_NOT_SELECTED = "checkout.deliveryMethod.notSelected";
    private static final String CHECKOUT_PAYMENT_METHOD_NOT_SELECTED = "checkout.paymentMethod.notSelected";
    private static final String CHECKOUT_TAX_MISSING = "checkout.error.tax.missing";
    private static final String CHECKOUT_CART_NOT_CALCULATED = "checkout.error.tax.missing";

    private static final Logger LOG = Logger.getLogger(PayPalSummaryCheckoutStepController.class);

    @Override
    @RequestMapping(value = "/placeOrder")
    @PreValidateQuoteCheckoutStep
    @RequireHardLogIn
    public String placeOrder(@ModelAttribute("placeOrderForm") final PlaceOrderForm placeOrderForm, final Model model,
        final HttpServletRequest request, final RedirectAttributes redirectModel)
        throws CMSItemNotFoundException,
        CommerceCartModificationException {
        if (validateOrderForm(model)) {
            return enterStep(model, redirectModel);
        }
        if (validateCart(redirectModel)) {
            return REDIRECT_PREFIX + "/cart";
        }
        boolean isPaymentAuthorized = false;
        try {
            isPaymentAuthorized = getCheckoutFacade().authorizePayment(null);
        } catch (final AdapterException ae) {
            LOG.error(ae.getMessage(), ae);
        }
        if (!isPaymentAuthorized) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_AUTHORIZATION_FAILED);
            return enterStep(model, redirectModel);
        }
        final OrderData orderData;
        try {
            orderData = getCheckoutFacade().placeOrder();
        } catch (final Exception e) {
            LOG.error("Failed to place Order", e);
            GlobalMessages.addErrorMessage(model, CHECKOUT_PLACE_ORDER_FAILED);
            return enterStep(model, redirectModel);
        }
        return redirectToOrderConfirmationPage(orderData);
    }

    protected boolean validateOrderForm(final Model model) {
        boolean invalid = false;
        if (getCheckoutFlowFacade().hasNoDeliveryAddress()) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_DELIVERY_ADDRESS_NOT_SELECTED);
            invalid = true;
        }
        if (getCheckoutFlowFacade().hasNoDeliveryMode()) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_DELIVERY_METHOD_NOT_SELECTED);
            invalid = true;
        }
        if (getCheckoutFlowFacade().hasNoPaymentInfo()) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_PAYMENT_METHOD_NOT_SELECTED);
            invalid = true;
        }
        final CartData cartData = getCheckoutFacade().getCheckoutCart();
        if (!getCheckoutFacade().containsTaxValues()) {
            LOG.error(String.format(
                "Cart %s does not have any tax values, which means the tax cacluation was not properly done, placement of order can't continue",
                cartData.getCode()));
            GlobalMessages.addErrorMessage(model, CHECKOUT_TAX_MISSING);
            invalid = true;
        }
        if (!cartData.isCalculated()) {
            LOG.error(
                String.format("Cart %s has a calculated flag of FALSE, placement of order can't continue",
                    cartData.getCode()));
            GlobalMessages.addErrorMessage(model, CHECKOUT_CART_NOT_CALCULATED);
            invalid = true;
        }
        return invalid;
    }
}
