package com.paypal.hybris.apitestingservice.service.impl;

import com.paypal.hybris.apitestingservice.service.ApitestingPaymentService;
import com.paypal.hybris.core.service.PayPalManualAuthorizationService;
import com.paypal.hybris.core.service.PayPalManualMultiCaptureService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.returns.OrderReturnException;
import de.hybris.platform.returns.ReturnActionResponse;
import de.hybris.platform.returns.ReturnCallbackService;
import de.hybris.platform.returns.model.ReturnEntryModel;
import de.hybris.platform.returns.model.ReturnRequestModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Currency;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

@Component
public class ApitestingPaymentServiceImpl implements ApitestingPaymentService {

    private static final Logger LOG = LoggerFactory.getLogger(ApitestingPaymentServiceImpl.class);
    private static final String REVIEW_DECISION = "_ReviewDecision";

    @Autowired
    PayPalManualMultiCaptureService payPalManualMultiCaptureService;

    @Autowired
    PayPalManualAuthorizationService payPalManualAuthorizationService;

    @Autowired
    private transient ModelService modelService;

    @Autowired
    private BusinessProcessService businessProcessService;

    @Autowired
    private PayPalPaymentService paymentService;

    @Autowired
    private ReturnCallbackService returnCallbackService;

    @Override
    public boolean doCapture(OrderModel order, BigDecimal amount, String transactionId) {
        boolean captured = false;
        PaymentTransactionEntryModel entryToCapture = getTransactionById(order, transactionId);
        PaymentTransactionType transactionType = amount.doubleValue() >= getAuthorizedAmount(order).doubleValue() ?
            PaymentTransactionType.CAPTURE :
            PaymentTransactionType.PARTIAL_CAPTURE;
        try {
            payPalManualMultiCaptureService.doMultiCapture(amount, entryToCapture, transactionType);
            if (isOrderFullyCaptured(order)) {
                executeManualPaymentCaptureOperation(order);
            }
            captured = true;

        } catch (AdapterException e) {
            String message = "Exception raised while doing multi capture: " + e.getMessage();
            LOG.warn(message);
        }
        return captured;
    }

    private boolean isOrderFullyCaptured(OrderModel order) {
        modelService.refresh(order);
        Predicate<PaymentTransactionEntryModel> captureEntries = entry ->
            (PaymentTransactionType.PARTIAL_CAPTURE.equals(entry.getType()) || PaymentTransactionType.CAPTURE
                .equals(entry.getType()))
                && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());

        double result = order.getPaymentTransactions().stream()
            .flatMap(transaction -> transaction.getEntries().stream())
            .filter(captureEntries).mapToDouble(entry -> entry.getAmount().doubleValue()).sum();

        return result >= getAuthorizedAmount(order).doubleValue();
    }

    protected void executeManualPaymentCaptureOperation(OrderModel order) {
        order.getOrderProcess().stream().filter((process) -> {
            return process.getCode().startsWith(order.getStore().getSubmitOrderProcessCode());
        }).forEach(filteredProcess ->
            businessProcessService.triggerEvent(filteredProcess.getOrder().getCode() + REVIEW_DECISION));
        LOG.info(String.format("Payment Capture Manual Release completed. %s triggered.", "ReviewDecision"));
        order.setStatus(OrderStatus.PAYMENT_CAPTURED);
        modelService.save(order);
    }

    @Override
    public boolean doAuthorize(OrderModel order, BigDecimal amount) {
        PaymentTransactionEntryModel entryModel = payPalManualAuthorizationService.doAuthorization(order, amount);
        String transactionStatus = entryModel.getTransactionStatus();
        return TransactionStatus.ACCEPTED.equals(transactionStatus);
    }

    @Override
    public boolean doReauthorize(OrderModel order, BigDecimal amount, String transactionId) {
        PaymentTransactionEntryModel entryModel = getTransactionById(order, transactionId);
        try {
            AuthorizationResult authorizationResult = paymentService
                .reauthorize(createSubscriptionRequest(amount, entryModel));

            entryModel.setRequestId(authorizationResult.getAuthorizationCode());
            entryModel.setAmount(amount);

            modelService.save(entryModel);
            modelService.refresh(entryModel.getPaymentTransaction());

            entryModel.getPaymentTransaction().setPlannedAmount(getAuthorizedAmount(order));
            modelService.saveAll();
            return true;
        } catch (AdapterException exception) {
            return false;
        }
    }

    public boolean doReturn(OrderModel order, boolean returnDeliveryCost, BigDecimal amount) {
        boolean result;
        ReturnRequestModel returnRequest = populateReturnRequest(order, returnDeliveryCost, amount);
        ReturnActionResponse returnActionResponse = new ReturnActionResponse(returnRequest);
        try {
            returnCallbackService.onReturnApprovalResponse(returnActionResponse);
            result = true;
        } catch (OrderReturnException e) {
            e.printStackTrace();
            result = false;
        }

        return result;
    }

    private ReturnRequestModel populateReturnRequest(OrderModel order, boolean returnDeliveryCost, BigDecimal amount) {
        ReturnRequestModel returnRequest = new ReturnRequestModel();
        returnRequest.setOrder(order);
        returnRequest.setCode(order.getCode());
        returnRequest.setRefundDeliveryCost(returnDeliveryCost);
        returnRequest.setSubtotal(amount);
        List<ReturnEntryModel> returnEntries = new ArrayList<>();
        returnRequest.setReturnEntries(returnEntries);
        return returnRequest;
    }

    public boolean doCancel(OrderModel order) {
        try {
            paymentService.doCancel(order);
        } catch (AdapterException e) {
            LOG.error("Transaction cancel failed: ", e);
            return false;
        }
        return OrderStatus.CANCELLED.equals(order.getStatus());
    }

    private PaymentTransactionEntryModel getTransactionById(OrderModel order, String transactionId) {
        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(entry -> entry.getRequestId().equals(transactionId)).findFirst().get();
    }

    private SubscriptionAuthorizationRequest createSubscriptionRequest(BigDecimal amount,
        PaymentTransactionEntryModel entryModel) {

        String merchantTransactionCode = entryModel.getPaymentTransaction().getCode();
        String subscriptionId = entryModel.getSubscriptionID();
        Currency currency = Currency.getInstance(entryModel.getCurrency().getIsocode());
        String paymentProvider = entryModel.getPaymentTransaction().getPaymentProvider();

        return new SubscriptionAuthorizationRequest(merchantTransactionCode, subscriptionId, currency, amount,
            null, paymentProvider);

    }

    public boolean isValidTransactionId(OrderModel order, String transactionId) {
        return getAuthorizedEntries(order).stream()
            .anyMatch(entryModel -> entryModel.getRequestId().equals(transactionId));
    }

    private BigDecimal getAuthorizedAmount(OrderModel order) {
        modelService.refresh(order);
        return getAuthorizedEntries(order).stream().map(PaymentTransactionEntryModel::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private List<PaymentTransactionEntryModel> getAuthorizedEntries(OrderModel order) {
        Predicate<PaymentTransactionEntryModel> isAuthorized = entryModel ->
            PaymentTransactionType.AUTHORIZATION.equals(entryModel.getType())
                && TransactionStatus.ACCEPTED.toString().equals(entryModel.getTransactionStatus());

        return order.getPaymentTransactions().stream()
            .flatMap(transaction -> transaction.getEntries().stream()).
                filter(isAuthorized).collect(Collectors.toList());
    }
}
