package com.paypal.hybris.apitestingservice.facades;

import com.paypal.hybris.apitestingservice.AddressDTO;
import com.paypal.hybris.apitestingservice.beans.PaymentTransactionBean;
import com.paypal.hybris.apitestingservice.service.ApitestingPaymentService;
import de.hybris.platform.commerceservices.customer.CustomerEmailResolutionService;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.security.PrincipalModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.returns.model.ReturnRequestModel;
import de.hybris.platform.servicelayer.search.FlexibleSearchQuery;
import de.hybris.platform.servicelayer.search.FlexibleSearchService;
import de.hybris.platform.servicelayer.search.SearchResult;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.CheckForNull;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;


@Component
public class OrderUiAutomationFacadeIml {

    private static final Logger LOG = Logger.getLogger(OrderUiAutomationFacadeIml.class);

    private static final String MANUAL_AUTHORIZATION_EMPTY_AMOUNT = "Enter amount";
    private static final String MANUAL_AUTHORIZATION_ZERO_AMOUNT = "Amount must be bigger then 0";
    private static final String MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT = "Invalid value";
    private static final String SELECT_PK_FROM = "SELECT {pk} FROM {";
    private static final String ORDER_ID = "orderID";
    private static final String SUCCESFULL = "SUCCESFULL";
    private static final String AMOUNT = "amount";
    private static final String AUTHORIZATION_ID = "authorizationID";
    private static final String USER_PK = "userPK";
    private static final String NOT_STATUS = "not_status";
    private static final String CANCELED = "CANCELED";
    private static final String RETURN_DELIVERY_COST = "returnDeliveryCost";

    @Autowired
    private FlexibleSearchService flexibleSearchService;
    @Autowired
    private CustomerEmailResolutionService customerEmailResolutionService;
    @Autowired
    private ApitestingPaymentService apitestingPaymentService;


    private static final String GET_ORDER_ADDRESS_BY_ID = SELECT_PK_FROM + OrderModel._TYPECODE
        + "} WHERE {" + AbstractOrderModel.CODE + "} = ?orderID";

    private static final String GET_USER_PK_BY_UID = SELECT_PK_FROM + UserModel._TYPECODE
        + "} WHERE {" + PrincipalModel.UID + "} LIKE '%s'";

    private static final String GET_ORDER_BY_USER_PK = SELECT_PK_FROM + OrderModel._TYPECODE
        + "} WHERE { " + AbstractOrderModel.USER + "} = ?userPK";

    private static final String GET_RETURN_RMA_BY_ID = "SELECT {pk} FROM {ReturnRequest JOIN " +
        "Order on {Order.pk}={ReturnRequest.order}} WHERE {Order." + AbstractOrderModel.CODE + "} = ?orderID";


    public AddressDTO getAddress(String orderID) {
        AddressDTO addressDTO = new AddressDTO();
        OrderModel order = getOrderById(orderID);

        if (order == null) {
            return addressDTO;
        }
        AddressModel address = order.getDeliveryAddress();
        if (address == null) {
            return addressDTO;
        }

        addressDTO.setStreet(address.getStreetname());
        addressDTO.setTown(address.getTown());
        addressDTO.setPostalcode(address.getPostalcode());
        addressDTO.setCountry(address.getCountry() != null ? address.getCountry().getName() : null);

        if (address.getRegion() != null && address.getRegion().getName() != null) {
            addressDTO.setRegion(address.getRegion().getName());
        }

        return addressDTO;
    }

    public Double getTotalPrice(String orderID) {
        OrderModel order = getOrderById(orderID);
        if (order == null) {
            return null;
        }
        return order.getTotalPrice();
    }

    public String getRMA(String orderID) {
        String orderRMA = getRMAReturnOrderById(orderID);
        if (orderRMA == null) {
            return null;
        }
        return orderRMA;
    }

    public Double getDeliveryCost(String orderID) {
        OrderModel order = getOrderById(orderID);
        if (order == null) {
            return null;
        }
        return order.getDeliveryCost();
    }

    public Double getTotalDiscount(String orderID) {
        OrderModel order = getOrderById(orderID);
        if (order == null) {
            return null;
        }
        return order.getTotalDiscounts();
    }

    private OrderModel getOrderById(String orderID) {
        final FlexibleSearchQuery flexibleSearchQuery = new FlexibleSearchQuery(GET_ORDER_ADDRESS_BY_ID);
        flexibleSearchQuery.addQueryParameter(ORDER_ID, orderID);
        SearchResult<OrderModel> resultSet = flexibleSearchService.search(flexibleSearchQuery);

        if (resultSet.getResult().isEmpty()) {
            LOG.info("Order " + orderID + "not found");
            return null;
        }
        return resultSet.getResult().get(0);
    }

    private String getRMAReturnOrderById(String orderID) {
        final FlexibleSearchQuery flexibleSearchQuery = new FlexibleSearchQuery(GET_RETURN_RMA_BY_ID);
        flexibleSearchQuery.addQueryParameter(ORDER_ID, orderID);
        SearchResult<ReturnRequestModel> resultSet = flexibleSearchService.search(flexibleSearchQuery);

        if (resultSet.getResult().isEmpty()) {
            LOG.info("Return RMA for Order " + orderID + "not found");
            return null;
        }
        return resultSet.getResult().get(0).getRMA();
    }

    private UserModel getUserPkByUid(String partOfUid) {

        final FlexibleSearchQuery flexibleSearchQuery = new FlexibleSearchQuery(
            String.format(GET_USER_PK_BY_UID, "%" + partOfUid + "%"));
        //TODO figure out why not works with query parameters.
        SearchResult<UserModel> resultSet = flexibleSearchService.search(flexibleSearchQuery);

        if (resultSet.getResult().isEmpty()) {
            LOG.warn("User where uid contains '" + partOfUid + "' was not found");
            return null;
        }
        return resultSet.getResult().get(0);
    }

    private List<OrderModel> getOrdersByUserPk(String userPk) {
        final FlexibleSearchQuery flexibleSearchQuery = new FlexibleSearchQuery(GET_ORDER_BY_USER_PK);
        flexibleSearchQuery.addQueryParameter(USER_PK, userPk);
        SearchResult<OrderModel> resultSet = flexibleSearchService.search(flexibleSearchQuery);

        if (resultSet.getResult().isEmpty()) {
            LOG.info("Order where 'user' = " + userPk + " was not found");
            return Collections.emptyList();
        }
        return resultSet.getResult();
    }

    public String getOrderIdByUID(String pardOfUid) {
        if (!StringUtils.isNotBlank(pardOfUid)) {
            LOG.error("Provided UID can`t be blank");
            return "";
        }

        UserModel user = getUserPkByUid(pardOfUid);
        if (user == null || user.getPk() == null) {
            return "";
        }

        List<OrderModel> orders = getOrdersByUserPk(user.getPk().toString());
        if (orders == null || orders.isEmpty()) {
            return "";
        }

        return orders.get(0).getCode();
    }

    public String getAddressFromAddressBook(String orderID) {
        OrderModel order = getOrderById(orderID);
        if (order == null) {
            return null;
        }
        AddressModel address = order.getDeliveryAddress();
        return address.getStreetname() + "  " + address.getStreetnumber() + "\n" + address.getTown() + "  " +
            address.getRegion().getName() + "\n" + address.getCountry().getName() + "  " + address.getPostalcode();
    }

    public String getDeliveryMode(String orderID) {
        OrderModel order = getOrderById(orderID);
        if (order == null) {
            return null;
        }
        return order.getDeliveryMode().getCode();
    }

    public String getAuthorizationPaymentTransactions(String orderID) {
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.AUTHORIZATION.equals(paymentTransactionEntryModel.getType())) {
                return paymentTransactionEntryModel.getTransactionStatusDetails();
            }
        }
        return null;
    }

    @CheckForNull
    private List<PaymentTransactionEntryModel> getPaymentTransactions(String orderID) {
        OrderModel orderModel = getOrderById(orderID);
        for (PaymentTransactionModel paymentTransactionModel : orderModel.getPaymentTransactions()) {
            return paymentTransactionModel.getEntries();
        }
        return new ArrayList<>();
    }

    public String getCapturedAmount(String orderID) {
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.CAPTURE.equals(paymentTransactionEntryModel.getType())) {
                return paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString();
            }
        }
        return "";
    }

    @CheckForNull
    public String getOrderStatus(String orderID) {
        OrderModel orderModel = getOrderById(orderID);
        return orderModel.getStatus() == null ? NOT_STATUS : orderModel.getStatus().getCode();
    }

    public String getCapturePaymentTransactionsStatusDetails(String orderID) {
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.CAPTURE.equals(paymentTransactionEntryModel.getType())) {
                return paymentTransactionEntryModel.getTransactionStatusDetails();
            }
        }
        return null;
    }

    public List<String> getAuthorizedAmountsList(String orderID) {
        List<String> list = new ArrayList<>();
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.AUTHORIZATION.equals(paymentTransactionEntryModel.getType()) &&
                SUCCESFULL.equals(paymentTransactionEntryModel.getTransactionStatusDetails()) &&
                TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())) {
                list.add(paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString());
            }
        }
        return list;
    }

    public List<PaymentTransactionBean> getAuthorizationTransactions(final String orderID) {
        List<PaymentTransactionBean> list = new ArrayList<>();
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.AUTHORIZATION.equals(paymentTransactionEntryModel.getType())) {
                PaymentTransactionBean bean = new PaymentTransactionBean();
                bean.setAmount(paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString());
                bean.setStatus(paymentTransactionEntryModel.getTransactionStatus());
                bean.setStatusDetails(paymentTransactionEntryModel.getTransactionStatusDetails());

                list.add(bean);
            }
        }
        return list;
    }


    public List<PaymentTransactionBean> getCaptureTransactions(final String orderID) {
        List<PaymentTransactionBean> list = new ArrayList<>();
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.CAPTURE.equals(paymentTransactionEntryModel.getType())) {
                PaymentTransactionBean bean = new PaymentTransactionBean();
                bean.setAmount(paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString());
                bean.setStatus(paymentTransactionEntryModel.getTransactionStatus());
                bean.setStatusDetails(paymentTransactionEntryModel.getTransactionStatusDetails());

                list.add(bean);
            }
        }
        return list;
    }

    public List<String> getVoidedAmountsList(String orderID) {
        List<String> list = new ArrayList<>();
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.AUTHORIZATION.equals(paymentTransactionEntryModel.getType()) &&
                SUCCESFULL.equals(paymentTransactionEntryModel.getTransactionStatusDetails()) &&
                CANCELED.equalsIgnoreCase(paymentTransactionEntryModel.getTransactionStatus())) {
                list.add(paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString());
            }
        }
        return list;
    }

    public List<String> getCapturedAmountsList(String orderID) {
        List<String> list = new ArrayList<>();
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if ((PaymentTransactionType.CAPTURE.equals(paymentTransactionEntryModel.getType())
                || PaymentTransactionType.PARTIAL_CAPTURE.equals(paymentTransactionEntryModel.getType())) &&
                TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())) {
                list.add(paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString());
            }
        }
        return list;
    }

    public List<String> getRefundedAmountsList(String orderID) {
        List<String> list = new ArrayList<>();
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if ((PaymentTransactionType.REFUND_PARTIAL.equals(paymentTransactionEntryModel.getType())) &&
                TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())) {
                list.add(paymentTransactionEntryModel.getAmount().setScale(2, RoundingMode.HALF_UP).toString());
            }
        }
        return list;
    }

    public boolean doCapture(Map<String, String> keyValueMap) {
        String orderID = keyValueMap.get(ORDER_ID);
        String amount = keyValueMap.get(AMOUNT);
        String transactionId;
        boolean result = false;

        if (keyValueMap.containsKey(AUTHORIZATION_ID)) {
            transactionId = keyValueMap.get(AUTHORIZATION_ID);
        } else {
            transactionId = getLastAuthorizationId(orderID);
        }

        validateAmount(amount);
        OrderModel order = getOrderById(orderID);
        BigDecimal amountToCapture = new BigDecimal(amount);
        if (apitestingPaymentService.isValidTransactionId(order, transactionId)) {
            result = apitestingPaymentService.doCapture(order, amountToCapture, transactionId);

        }

        return result;
    }

    public boolean doAuthorize(Map<String, String> keyValueMap) {
        String orderID = keyValueMap.get(ORDER_ID);
        String amount = keyValueMap.get(AMOUNT);

        validateAmount(amount);
        OrderModel order = getOrderById(orderID);
        BigDecimal totalAmount = new BigDecimal(amount);
        if (order != null) {
            return apitestingPaymentService.doAuthorize(order, totalAmount);
        }
        return false;
    }

    public boolean doReauthorize(Map<String, String> keyValueMap) {
        String orderID = keyValueMap.get(ORDER_ID);
        String amount = keyValueMap.get(AMOUNT);
        String transactionId;
        boolean result = false;

        if (keyValueMap.containsKey(AUTHORIZATION_ID)) {
            transactionId = keyValueMap.get(AUTHORIZATION_ID);
        } else {
            transactionId = getLastAuthorizationId(orderID);
        }

        validateAmount(amount);
        BigDecimal totalAmount = new BigDecimal(amount);
        OrderModel order = getOrderById(orderID);
        if (apitestingPaymentService.isValidTransactionId(order, transactionId)) {
            result = apitestingPaymentService.doReauthorize(order, totalAmount, transactionId);
        }
        return result;
    }

    public boolean doReturn(Map<String, String> keyValueMap) {
        String orderID = keyValueMap.get(ORDER_ID);
        String amount = keyValueMap.get(AMOUNT);
        boolean returnDeliveryCost;
        boolean result = false;

        if (keyValueMap.containsKey(RETURN_DELIVERY_COST)) {
            returnDeliveryCost = new Boolean(keyValueMap.get(RETURN_DELIVERY_COST));
        } else {
            returnDeliveryCost = false;
        }

        validateAmount(amount);
        BigDecimal totalAmount = new BigDecimal(amount);
        OrderModel order = getOrderById(orderID);
        if (order != null) {
            result = apitestingPaymentService.doReturn(order, returnDeliveryCost, totalAmount);
        }
        return result;
    }

    public boolean doCancel(Map<String, String> keyValueMap) {
        String orderID = keyValueMap.get(ORDER_ID);

        OrderModel order = getOrderById(orderID);
        if (order != null) {
            return apitestingPaymentService.doCancel(order);
        }
        return false;
    }

    public String getLastAuthorizationId(String orderID) {
        String transactionId = null;
        for (PaymentTransactionEntryModel paymentTransactionEntryModel : getPaymentTransactions(orderID)) {
            if (PaymentTransactionType.AUTHORIZATION.equals(paymentTransactionEntryModel.getType()) &&
                SUCCESFULL.equals(paymentTransactionEntryModel.getTransactionStatusDetails())) {
                transactionId = paymentTransactionEntryModel.getRequestId();
            }
        }

        return transactionId;
    }

    private void validateAmount(String amount) {
        if (StringUtils.isBlank(amount)) {
            throw new ArithmeticException(MANUAL_AUTHORIZATION_EMPTY_AMOUNT);
        }
        try {
            if (BigDecimal.ZERO.equals(new BigDecimal(amount))) {
                throw new ArithmeticException(MANUAL_AUTHORIZATION_ZERO_AMOUNT);
            }
        } catch (NumberFormatException e) {
            throw new ArithmeticException(MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT);
        }
    }

    @CheckForNull
    public String getUserEmail(String orderID) {
        CustomerModel customer = (CustomerModel) getOrderById(orderID).getUser();
        return customerEmailResolutionService.getEmailForCustomer(customer);
    }
}
