import {ChangeDetectionStrategy, Component, Input, OnInit} from '@angular/core';
import {
  PageType,
  PaypalCheckoutData,
  PayPalCreditMessageComponent,
  PaypalPaymentDetailsService,
  PaypalUtilsService
} from 'paypal-spartacus-core';
import {combineLatest, Observable} from 'rxjs';
import {CmsComponentData} from '@spartacus/storefront';
import {ActiveCartService, CmsService} from '@spartacus/core';
import {map} from 'rxjs/operators';

@Component({
  selector: 'paypal-credit-message',
  templateUrl: './paypal-credit-message.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class PaypalCreditMessageComponent implements OnInit {
  @Input()
    private pageType: PageType;

  loadedCheckoutData$: Observable<PaypalCheckoutData>;
  amount$: Observable<number>;
  componentData$: Observable<PayPalCreditMessageComponent>;

  constructor(
    protected component: CmsComponentData<PayPalCreditMessageComponent>,
    protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
    protected paypalUtilsService: PaypalUtilsService,
    protected activeCartService: ActiveCartService,
    protected cmsService: CmsService,
  ) {
  }

  ngOnInit(): void {
    if (this.pageType === PageType.CART) {
      this.componentData$ = this.cmsService.getComponentData('PayPalCartPageCreditMessageComponentSPA');
    }else{
      this.componentData$ = this.component.data$;
    }
    this.loadedCheckoutData$ = combineLatest(
      [this.componentData$,
        this.paypalPaymentDetailsService.loadPaymentDetails('mark')])
    .pipe(
      map(
        ([componentData, checkoutData]) => {
          if (!(componentData.pageType === 'CartPage' && checkoutData.payPalPaymentMethod.payPalExpressEnabled) && !checkoutData.configurationData.isBillingAgreementEnabled) {
            this.initializePayPalCreditMessage(checkoutData);
          }
          return checkoutData;
        }));


    this.amount$ = this.activeCartService
    .getActive()
    .pipe(
      map((cart) => {
        if (cart.totalPrice !== undefined) {
          return cart.totalPrice.value;
        } else {
          return 0.0;
        }
      })
    );
  }

  initializePayPalCreditMessage(checkoutData: PaypalCheckoutData): void {
    this.paypalUtilsService.loadPayPalSDK(checkoutData.configurationData, undefined);
  }


}
