import {Component, ElementRef, EventEmitter, Input, OnInit, Output, ViewChild} from '@angular/core';
import {FormBuilder, FormGroup, Validators} from '@angular/forms';
import {BehaviorSubject, Observable} from 'rxjs';
import {Address, CardType, CheckoutDeliveryService, CheckoutPaymentService, Country, UserPaymentService} from '@spartacus/core';
import {Card, ICON_TYPE} from '@spartacus/storefront';
import {map, tap} from 'rxjs/operators';

class CheckoutData {
}

@Component({
  selector: 'paypal-hosted-fields',
  templateUrl: './paypal-hosted-fields.component.html',
  styleUrls: [ './paypal-hosted-fields.component.scss' ]
})
export class PaypalHostedFieldsComponent implements OnInit {

  @Input() isGuestCheckout: boolean;
  @Input() checkoutData: CheckoutData;
  @Input() pageType = "BILLING";
  @Input() showSavePaymentInfo : boolean;

  @Output()
  setPaymentDetails = new EventEmitter<any>();

  iconTypes = ICON_TYPE;
  months: string[] = [];
  years: number[] = [];
  showDefaultMethod : boolean;

  paymentForm: FormGroup = this.fb.group({
    cardType: this.fb.group({
      code: [null, Validators.required],
    }),
    accountHolderName: ['', Validators.required],
    cardNumber: ['', Validators.required],
    expiryMonth: [null, Validators.required],
    expiryYear: [null, Validators.required],
    cvn: ['', Validators.required],
    defaultPayment: [false],
    saved: false,
  });

  billingAddressForm: FormGroup = this.fb.group({
    firstName: ['', Validators.required],
    lastName: ['', Validators.required],
    line1: ['', Validators.required],
    line2: [''],
    town: ['', Validators.required],
    country: this.fb.group({
      isocode: [null, Validators.required],
    }),
    postalCode: ['', Validators.required],
  });

  shippingAddress$: Observable<Address>;
  sameAsShippingAddress = true;
  countries$: Observable<Country[]>;
  cardTypes$: Observable<CardType[]>;
  selectedCountry$: BehaviorSubject<string> = new BehaviorSubject<string>('');

  constructor(
    protected fb: FormBuilder,
    protected userPaymentService: UserPaymentService,
    protected checkoutDeliveryService: CheckoutDeliveryService,
    protected checkoutPaymentService: CheckoutPaymentService
  ) {}

  ngOnInit(): void {
    this.showDefaultMethod = false;
    this.expMonthAndYear();
    this.shippingAddress$ = this.checkoutDeliveryService.getDeliveryAddress();

    this.cardTypes$ = this.checkoutPaymentService.getCardTypes().pipe(
      map(value => {
        return value.filter(value1 => value1.name);
      }),
      tap((cardTypes) => {
        if (Object.keys(cardTypes).length === 0) {
          this.checkoutPaymentService.loadSupportedCardTypes();
        }
      })
    );

    this.countries$ = this.userPaymentService.getAllBillingCountries().pipe(
      tap((countries) => {
        // If the store is empty fetch countries. This is also used when changing language.
        if (Object.keys(countries).length === 0) {
          this.userPaymentService.loadBillingCountries();
        }
      })
    );
  }

  expMonthAndYear(): void {
    const year = new Date().getFullYear();

    for (let i = 0; i < 10; i++) {
      this.years.push(year + i);
    }

    for (let j = 1; j <= 12; j++) {
      if (j < 10) {
        this.months.push(`0${j}`);
      } else {
        this.months.push(j.toString());
      }
    }
  }

  isBillingPage(): boolean{
    return "BILLING" === this.pageType;
  }

  toggleSameAsShippingAddress(): void {
    this.sameAsShippingAddress = !this.sameAsShippingAddress;
  }

  toggleShowDefaultMethod(): void {
    this.showDefaultMethod = !this.showDefaultMethod;
  }

  getAddressCardContent(address: Address): Card {
    let region = '';
    if (address.region && address.region.isocode) {
      region = address.region.isocode + ', ';
    }

    return {
      textBold: address.firstName + ' ' + address.lastName,
      text: [
        address.line1,
        address.line2,
        address.town + ', ' + region + address.country.isocode,
        address.postalCode,
        address.phone,
      ],
    };
  }

  countrySelected(country: Country): void {
    this.billingAddressForm.get('country.isocode').setValue(country.isocode);
    this.selectedCountry$.next(country.isocode);
  }

  onSubmit() {
    if (this.paymentForm.valid) {
      if (this.sameAsShippingAddress) {
        this.setPaymentDetails.emit({
          paymentDetails: this.paymentForm.value,
          billingAddress: null,
        });
      } else {
        if (this.billingAddressForm.valid) {
          this.setPaymentDetails.emit({
            paymentDetails: this.paymentForm.value,
            billingAddress: this.billingAddressForm.value,
          });
        } else {
          this.billingAddressForm.markAllAsTouched();
        }
      }
    } else {
      this.paymentForm.markAllAsTouched();

      if (!this.sameAsShippingAddress) {
        this.billingAddressForm.markAllAsTouched();
      }
    }
  }

}
