import {ChangeDetectorRef, Component, ElementRef, OnInit, ViewChild} from '@angular/core';
import {
  AddingPaymentMethodAndAddressStatusData,
  PayPalConnectData,
  PaypalConnectService,
  PaypalLoginService,
  PaypalPaymentDetailsService
} from 'paypal-spartacus-core';
import {ActivatedRoute, Router} from '@angular/router';
import {combineLatest, of} from 'rxjs';
import {ActiveCartService, AuthRedirectStorageService, GlobalMessageService, GlobalMessageType} from '@spartacus/core';
import {first} from 'rxjs/operators';

@Component({
  selector: 'paypal-connect-waiting',
  templateUrl: './paypal-connect-waiting.component.html'
})
export class PaypalConnectWaitingComponent implements OnInit {

  @ViewChild('paypalButton') paypalButton: ElementRef;

  keySessionAccessToken = 'accessToken';
  keySessionPayerId = 'payerId';
  accessToken: string;
  isCustomerRegistered = false;
  isPayPalLoaded = false;

  protected connectData: PayPalConnectData;

  constructor(
    protected paypalConnectService: PaypalConnectService,
    protected router: Router,
    protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
    protected paypalLoginService: PaypalLoginService,
    protected activatedRoute: ActivatedRoute,
    protected activeCartService: ActiveCartService,
    protected cdr: ChangeDetectorRef,
    protected globalMessageService: GlobalMessageService,
    protected authRedirectStorageService: AuthRedirectStorageService
  ) {
  }

  ngOnInit(): void {
    this.authRedirectStorageService.getRedirectUrl()
    .pipe(first())
    .subscribe(url => {
      if(url.startsWith('/connect')) {
        this.authRedirectStorageService.setRedirectUrl('/')
      }
    });

    this.activatedRoute.queryParams
    .pipe(first())
    .subscribe(params => {
      const authorizationCode = params['code'];
      if (authorizationCode) {
        const storageAccessToken = localStorage.getItem(this.keySessionAccessToken);
        const storagePayerId = localStorage.getItem(this.keySessionPayerId);
        if (storageAccessToken && storagePayerId) {
          this.accessToken = storageAccessToken;
          this.isCustomerRegistered = true;
          this.cdr.detectChanges();
          this.onSuccessRegistration(storagePayerId);
        } else {
          this.paypalConnectService.exchangeAuthorizationCode(authorizationCode)
            .subscribe(connectData => {
              this.accessToken = connectData.accessToken;
              localStorage.setItem(this.keySessionAccessToken, this.accessToken);
              if (connectData.isRegistered) {
                this.removeLocalStorage();
                this.paypalConnectService.login(this.accessToken).subscribe(flag => {
                });
              } else {
                this.paypalConnectService.register(this.accessToken).subscribe(payerId => {
                  if (payerId) {
                    this.isCustomerRegistered = true;
                    this.cdr.detectChanges();
                    localStorage.setItem(this.keySessionPayerId, payerId);
                    this.onSuccessRegistration(payerId);
                  } else {
                    this.addErrorMessage('connectRegister.error.somethingWentWrong');
                    this.router.navigate(['/login']);
                  }
                }, err => {
                  this.addErrorMessage('connectRegister.error.somethingWentWrong');
                  this.router.navigate(['/login']);
                });
              }

            });
        }
      } else {
        const forced = localStorage.getItem('forced');
        localStorage.removeItem('forced');
        if (forced === 'true') {
          this.router.navigateByUrl('/login?forced=true');
        } else {
          this.router.navigate(['/login']);
        }
      }
    });
  }

  loginUser() {
    this.removeLocalStorage();
    this.paypalConnectService.login(this.accessToken).subscribe(flag => {
    });
  }

  private addErrorMessage(key: string) {
    this.globalMessageService.add(
      {key: key},
      GlobalMessageType.MSG_TYPE_WARNING,
      10000
    );
    this.removeLocalStorage();
  }

  private addInfoMessage(key: string) {
    this.globalMessageService.add(
      {key: key},
      GlobalMessageType.MSG_TYPE_INFO,
      10000
    );
  }

  private addWarningMessage(key: string) {
    this.globalMessageService.add(
      {key: key},
      GlobalMessageType.MSG_TYPE_WARNING,
      10000
    );
  }

  onSuccessRegistration(payerId: string) {
    combineLatest(
      [this.paypalPaymentDetailsService.loadPaypalButtonStyles('mark'),
        this.paypalPaymentDetailsService.loadPaymentDetails('login')])
      .subscribe(([buttonStyle, checkoutData]) => {
        this.paypalLoginService.initializePayPalButton(checkoutData.configurationData, this.accessToken, this.paypalButton.nativeElement,
          buttonStyle, (status: AddingPaymentMethodAndAddressStatusData) => {
            if (status.isAddressAdded) {
              this.addInfoMessage('connectRegister.addressWasAddedSuccess');
            } else {
              this.addWarningMessage('connectRegister.error.addressNotAdded');
            }
            if (status.isPaymentMethodAdded) {
              this.addInfoMessage('connectRegister.paymentMethodWasAddedSuccess');
            } else {
              this.addWarningMessage('connectRegister.error.paymentMethodNotAdded');
            }
            this.paypalConnectService.login(this.accessToken).subscribe(flag => {
            });
            this.removeLocalStorage();
          });
        this.isPayPalLoaded = true;
        this.cdr.detectChanges();
      });
  }

  removeLocalStorage() {
    localStorage.removeItem(this.keySessionPayerId);
    localStorage.removeItem(this.keySessionAccessToken);
  }
}
