import {ChangeDetectorRef, Component, ElementRef, OnInit, ViewChild} from '@angular/core';
import {FormBuilder, FormGroup, Validators} from '@angular/forms';
import {
  AddingPaymentMethodAndAddressStatusData,
  PayPalConnectB2BData,
  PayPalConnectB2BRegisterData,
  PaypalConnectService,
  PaypalLoginService,
  PaypalPaymentDetailsService
} from 'paypal-spartacus-core';
import {ActivatedRoute, Router} from '@angular/router';
import {combineLatest, Observable, of} from 'rxjs';
import {first, map, switchMap, tap} from 'rxjs/operators';
import {
  AuthRedirectStorageService,
  Country,
  GlobalMessageService,
  GlobalMessageType,
  Title,
  UserAddressService,
  UserService
} from '@spartacus/core';
import {sortTitles} from '@spartacus/storefront';

@Component({
  selector: 'paypal-connect-additional',
  templateUrl: './paypal-connect-additional.component.html'
})
export class PaypalConnectAdditionalComponent implements OnInit {

  @ViewChild('paypalButton') paypalButton: ElementRef;

  payPalConnectData$: Observable<PayPalConnectB2BData>;
  countries$: Observable<Country[]>;
  titles$: Observable<Title[]>;
  accessToken: string;

  keySessionAccessToken = 'accessToken';
  keySessionPayerId = 'payerId';
  isCustomerRegistered = false;
  showRegistrationSpinner = false;

  registerForm: FormGroup = this.fb.group(
    {
      country: this.fb.group({
        isocode: [null, Validators.required],
      }),
      titleCode: [''],
      company: ['', Validators.required],
      addressLine1: ['', Validators.required],
      addressLine2: [''],
      city: ['', Validators.required],
      postcode: ['', Validators.required],
      yourPosition: ['', Validators.required],
      telephone: ['', Validators.required],
    }
  );

  constructor(
    protected globalMessageService: GlobalMessageService,
    protected fb: FormBuilder,
    protected paypalConnectService: PaypalConnectService,
    protected activatedRoute: ActivatedRoute,
    protected userAddressService: UserAddressService,
    protected userRegister: UserService,
    protected paypalLoginService: PaypalLoginService,
    protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
    protected authRedirectStorageService: AuthRedirectStorageService,
    protected cdr: ChangeDetectorRef,
    protected router: Router
  ) {
  }

  ngOnInit() {
    this.authRedirectStorageService.getRedirectUrl()
    .pipe(first())
    .subscribe(url => {
      if(url.startsWith('/connect/additional')) {
        this.authRedirectStorageService.setRedirectUrl('/')
      }
    });

    const storageAccessToken = localStorage.getItem(this.keySessionAccessToken);
    if (storageAccessToken) {
      this.accessToken = storageAccessToken;
      this.payPalConnectData$ = of({});
    } else {
      this.payPalConnectData$ = this.activatedRoute.queryParams.pipe(
        first(),
        switchMap(params => {
          const authorizationCode = params['code'];
          if (authorizationCode) {
            return this.paypalConnectService.exchangeAuthorizationCodeB2B(authorizationCode);
          } else {
            this.router.navigate(['/login']);
          }
          return of(null);
        }),
        map((connectData: PayPalConnectB2BData) => {
          if (connectData.isRegistered && !connectData.isApproved) {
            this.router.navigate(['/login']);
            this.globalMessageService.add(
              {key: 'connectRegister.accountNotApproved'},
              GlobalMessageType.MSG_TYPE_WARNING,
              10000
            );
            return null;
          } else if (connectData.isRegistered) {
            this.paypalConnectService.login(connectData.accessToken).subscribe(flag => {
            });
            return null;
          }
          this.accessToken = connectData.accessToken;
          localStorage.setItem(this.keySessionAccessToken, connectData.accessToken);
          return connectData;
        })
      );
    }

    this.countries$ = this.userAddressService.getDeliveryCountries().pipe(
      tap((countries: Country[]) => {
        if (Object.keys(countries).length === 0) {
          this.userAddressService.loadDeliveryCountries();
        }
      })
    );

    this.titles$ = this.userRegister.getTitles().pipe(
      map((titles: Title[]) => {
        return titles.sort(sortTitles);
      })
    );

    const storagePayerId = localStorage.getItem(this.keySessionPayerId);
    if (storageAccessToken && storagePayerId) {
      this.isCustomerRegistered = true;
      this.cdr.detectChanges();
      this.onSuccessRegistration(storagePayerId);
    }
  }

  countrySelected(country: Country): void {
    this.registerForm.get('country')?.get('isocode')?.setValue(country.isocode);
  }

  submitForm(): void {
    if (this.registerForm.valid) {
      this.showRegistrationSpinner = true;
      this.paypalConnectService.registerB2BCustomer(this.accessToken, this.collectDataFromForm(this.registerForm)).subscribe(payerId => {

        if (payerId) {
          this.isCustomerRegistered = true;
          this.cdr.detectChanges();
          this.addInfoMessage('connectRegister.successfulRegistration');
          localStorage.setItem(this.keySessionPayerId, payerId);
          this.onSuccessRegistration(payerId);
        } else {
          this.addErrorMessage('connectRegister.error.somethingWentWrong');
          this.router.navigate(['/login']);

        }
      }, err => {
        this.addErrorMessage('connectRegister.error.somethingWentWrong');
        this.router.navigate(['/login']);
      });
    } else {
      this.registerForm.markAllAsTouched();
    }
  }

  onSuccessRegistration(payerId: string) {
    combineLatest(
      [this.paypalPaymentDetailsService.loadPaypalButtonStyles('mark'),
        this.paypalPaymentDetailsService.loadPaymentDetails('login')])
    .subscribe(([buttonStyle, checkoutData]) => {
      this.paypalLoginService.initializePayPalButton(checkoutData.configurationData, this.accessToken, this.paypalButton.nativeElement,
        buttonStyle, (status: AddingPaymentMethodAndAddressStatusData) => {
          if (status.isAddressAdded) {
            this.addInfoMessage('connectRegister.addressWasAddedSuccess');
          } else {
            this.addWarningMessage('connectRegister.error.addressNotAdded');
          }
          if (status.isPaymentMethodAdded) {
            this.addInfoMessage('connectRegister.paymentMethodWasAddedSuccess');
          } else {
            this.addWarningMessage('connectRegister.error.paymentMethodNotAdded');
          }
          this.removeLocalStorage();
          this.router.navigate(['/login']);
        });
    });
    this.showRegistrationSpinner = false;
    this.cdr.detectChanges();
  }

  collectDataFromForm(formData: FormGroup): PayPalConnectB2BRegisterData {
    return {
      companyAddressCountryIso: formData.value.country.isocode,
      titleCode: formData.value.titleCode,
      companyName: formData.value.company,
      companyAddressStreet: formData.value.addressLine1,
      companyAddressStreetLine2: formData.value.addressLine2,
      companyAddressCity: formData.value.city,
      companyAddressPostalCode: formData.value.postcode,
      position: formData.value.yourPosition,
      telephone: formData.value.telephone
    };
  }

  navigateToLogin() {
    this.removeLocalStorage();
    this.router.navigate(['/login']);
  }

  private addErrorMessage(key: string) {
    this.globalMessageService.add(
      {key: key},
      GlobalMessageType.MSG_TYPE_WARNING,
      10000
    );
    this.removeLocalStorage();
  }

  private addInfoMessage(key: string) {
    this.globalMessageService.add(
      {key: key},
      GlobalMessageType.MSG_TYPE_INFO, 10000
    );
  }

  private addWarningMessage(key: string) {
    this.globalMessageService.add(
      {key: key},
      GlobalMessageType.MSG_TYPE_WARNING
    );
  }

  removeLocalStorage() {
    localStorage.removeItem(this.keySessionPayerId);
    localStorage.removeItem(this.keySessionAccessToken);
  }
}
