import {Component, ComponentRef, ElementRef, EventEmitter, Input, OnInit, Output, ViewChild, ViewContainerRef} from '@angular/core';
import {
  PageType,
  PayPalButtonStyle,
  PaypalCheckoutData,
  PaypalMarkCheckoutService,
  PaypalPaymentDetailsService,
  PaypalUtilsService
} from 'paypal-spartacus-core';
import {combineLatest, Observable, Subscription} from 'rxjs';
import {map, take} from 'rxjs/operators';
import {ActiveCartService, Address, CheckoutDeliveryService, CheckoutService, RoutingService} from '@spartacus/core';
import {CheckoutStepService, LAUNCH_CALLER, LaunchDialogService} from '@spartacus/storefront';
import {ActivatedRoute} from '@angular/router';

@Component({
  selector: 'paypal-new-payment-method',
  templateUrl: './paypal-new-payment-method.component.html',
})
export class PaypalNewPaymentMethodComponent implements OnInit {

  loadedCheckoutData$: Observable<PaypalCheckoutData>;
  payPalButtonStyle: PayPalButtonStyle;
  radioButtonType: string;
  pageType = PageType.MARK;
  showSavePaymentInfo: boolean;
  deliveryAddress: Address;
  subscription: Subscription = new Subscription();
  placedOrder: void | Observable<ComponentRef<any>>;

  @ViewChild('paypalMarkButton') paypalButton: ElementRef;

  @Input()
  protected paymentMethodsCount: number;

  @Input()
  isGuestCheckout: boolean;

  @Output()
  protected backToPaymentMethods = new EventEmitter<any>();

  @Output()
  setPaymentDetails = new EventEmitter<any>();

  constructor(
    protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
    protected paypalMarkCheckoutService: PaypalMarkCheckoutService,
    protected paypalUtilsService: PaypalUtilsService,
    protected checkoutDeliveryService: CheckoutDeliveryService,
    protected checkoutStepService: CheckoutStepService,
    protected activatedRoute: ActivatedRoute,
    protected routingService: RoutingService,
    protected launchDialogService: LaunchDialogService,
    protected vcr: ViewContainerRef,
    protected checkoutService: CheckoutService,
    protected activeCartService: ActiveCartService
  ) {
  }

  ngOnInit(): void {
    this.loadedCheckoutData$ = combineLatest(
      [this.paypalPaymentDetailsService.loadPaypalButtonStyles('mark'),
        this.paypalPaymentDetailsService.loadPaymentDetails('mark')])
    .pipe(
      map(
        ([buttonStyle, checkoutData]) => {
          this.payPalButtonStyle = buttonStyle;
          this.onChangePayPal(checkoutData, buttonStyle);
          return checkoutData;
        }));

    this.checkoutDeliveryService
    .getDeliveryAddress()
    .pipe(take(1))
    .subscribe((address: Address) => {
      this.deliveryAddress = address;
    });

    this.paypalUtilsService.savePaymentInfo = false;
    this.onLPMOrderPlacement();
  }

  back(): void {
    this.backToPaymentMethods.emit();
  }

  next(): void {
    this.checkoutStepService.next(this.activatedRoute);
  }

  isBackButtonAvailable(): boolean {
    return this.paymentMethodsCount > 0;
  }

  onChangePayPal(checkoutData: PaypalCheckoutData, buttonStyle: PayPalButtonStyle): void {
    this.radioButtonType = 'paypal';
    this.showSavePaymentInfo = checkoutData.configurationData.isBillingAgreementFlowEnabled && !this.activeCartService.isGuestCart();
    if (!this.paypalButton.nativeElement.hasChildNodes()) {
      this.paypalMarkCheckoutService.initializePayPalCheckout(checkoutData.configurationData, this.pageType, this.paypalButton.nativeElement,
        buttonStyle);
    }
  }

  toggleSavePaymentInfo(event): void {
    this.paypalUtilsService.toggleSavePaymentInfo(event.target.checked);
  }

  creditCardPayment(event): void {
    this.setPaymentDetails.emit(event);
  }

  onLPMOrderPlacement(): void {
    this.subscription.add(
      combineLatest([
        this.checkoutService.getPlaceOrderLoading(),
        this.checkoutService.getPlaceOrderSuccess(),
        this.checkoutService.getPlaceOrderError(),
      ]).subscribe(([orderLoading, orderSuccess, orderError]) => {
        if (orderLoading) {
          this.placedOrder = this.launchDialogService.launch(
            LAUNCH_CALLER.PLACE_ORDER_SPINNER,
            this.vcr
          );
        }

        if (orderError) {
          if (this.placedOrder) {
            this.placedOrder
            .subscribe((component) => {
              this.launchDialogService.clear(
                LAUNCH_CALLER.PLACE_ORDER_SPINNER
              );
              component.destroy();
            })
            .unsubscribe();
            this.checkoutService.clearPlaceOrderState();
          }
        }

        if (orderSuccess) {
          this.routingService.go({cxRoute: 'orderConfirmation'});
        }
      })
    );
  }

  ngOnDestroy(): void {
    this.subscription.unsubscribe();
    this.launchDialogService.clear(LAUNCH_CALLER.PLACE_ORDER_SPINNER);
    this.checkoutService.clearPlaceOrderState();
  }

}
