import {Injectable} from '@angular/core';
import {Observable} from 'rxjs';
import {HttpClient, HttpHeaders, HttpParams} from '@angular/common/http';
import {ConverterService, OccEndpointsService} from '@spartacus/core';
import {ConnectAdapter} from '../../paypal-checkout/connectors/connect.adapter';
import {
  PAYPAL_ADDING_PAYMENT_METHOD_AND_ADDRESS_STATUS_NORMALIZER,
  PAYPAL_BILLING_AGREEMENT_PAYMENT_FOR_NEW_USER_SERIALIZER,
  PAYPAL_BILLING_AGREEMENT_TOKEN_NORMALIZER,
  PAYPAL_BOOLEAN_NORMALIZER,
  PAYPAL_CONNECT_B2B_DATA_NORMALIZER,
  PAYPAL_CONNECT_B2B_REGISTER_DATA_SERIALIZER,
  PAYPAL_CONNECT_DATA_NORMALIZER,
  PAYPAL_CONNECT_LOGIN_DATA_NORMALIZER,
  PAYPAL_STRING_NORMALIZER
} from '../../paypal-checkout/converters';
import {
  AddingPaymentMethodAndAddressStatusData,
  Fields,
  PayPalBillingAgreementPaymentForNewUserRequest,
  PayPalBillingAgreementPaymentRequest,
  PayPalBillingAgreementToken,
  PayPalConnectB2BData,
  PayPalConnectB2BRegisterData,
  PayPalConnectData,
  PayPalConnectLoginData
} from '../../paypal-checkout/models';

@Injectable()
export class PaypalConnectAdapter implements ConnectAdapter {

  constructor(
    protected http: HttpClient,
    protected occEndpoints: OccEndpointsService,
    protected converter: ConverterService
  ) {
  }

  protected getEndpoint(userId): string {
    return this.occEndpoints.getEndpoint('/users/' + userId + '/connect');
  }

  exchangeAuthorizationCode(
    authorizationToken: string,
    userId: string): Observable<PayPalConnectData> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/exchange`;

    return this.http.post(url, {'authorizationCode': authorizationToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_CONNECT_DATA_NORMALIZER));
  }

  afterLogin(
    accessToken: string,
    userId: string
  ): Observable<boolean> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/afterLogin`;

    return this.http.post(url, {'accessToken': accessToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_BOOLEAN_NORMALIZER));
  }

  login(
    accessToken: string,
    userId: string
  ): Observable<PayPalConnectLoginData> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/login`;

    return this.http.post(url, {'accessToken': accessToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_CONNECT_LOGIN_DATA_NORMALIZER));
  }

  register(
    accessToken: string,
    userId: string
  ): Observable<string> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/register`;

    return this.http.post(url, {'accessToken': accessToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_STRING_NORMALIZER));
  }

  exchangeAuthorizationCodeB2B(
    authorizationToken: string,
    userId: string
  ): Observable<PayPalConnectB2BData> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/exchange`;

    return this.http.post(url, {'authorizationCode': authorizationToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_CONNECT_B2B_DATA_NORMALIZER));
  }

  registerB2BCustomer(
    accessToken: string,
    registerData: PayPalConnectB2BRegisterData,
    userId: string
  ): Observable<string> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/register`;

    registerData = this.converter.convert(registerData, PAYPAL_CONNECT_B2B_REGISTER_DATA_SERIALIZER);

    return this.http.post(url, registerData, {headers})
    .pipe(this.converter.pipeable(PAYPAL_STRING_NORMALIZER));
  }

  savePaymentDetailsAndAddressForNewUser(
    payPalBillingAgreementPaymentRequest: PayPalBillingAgreementPaymentRequest,
    accessToken: string,
    userId: string,
    fields?: Fields,
    paymentMethodType?: string
  ): Observable<AddingPaymentMethodAndAddressStatusData> {

    const params: HttpParams = new HttpParams()
    .set('fields', fields)
    .append('paymentMethodType', paymentMethodType);

    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    let request: PayPalBillingAgreementPaymentForNewUserRequest = {
      billingAgreementRequest: payPalBillingAgreementPaymentRequest,
      accessToken: accessToken
    };

    request = this.converter
    .convert(request, PAYPAL_BILLING_AGREEMENT_PAYMENT_FOR_NEW_USER_SERIALIZER);

    const url = `${this.getEndpoint(userId)}/createPaymentDetails`;

    return this.http.post(url, request, {headers, params})
    .pipe(this.converter.pipeable(PAYPAL_ADDING_PAYMENT_METHOD_AND_ADDRESS_STATUS_NORMALIZER));
  }

  createBillingAgreementTokenWithoutCart(userId: string): Observable<PayPalBillingAgreementToken> {
    const headers = new HttpHeaders({
      'Content-Type': 'application/json'
    });

    const url = `${this.getEndpoint(userId)}/createBillingAgreementToken`;

    return this.http.post(url, {}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_BILLING_AGREEMENT_TOKEN_NORMALIZER));
  }



}
