import {Injectable} from '@angular/core';
import {Observable} from 'rxjs';
import {HttpClient, HttpHeaders, HttpParams} from '@angular/common/http';
import {
  PAYPAL_BILLING_AGREEMENT_PAYMENT_SERIALIZER,
  PAYPAL_BILLING_AGREEMENT_TOKEN_NORMALIZER,
  PAYPAL_ORDER_NORMALIZER,
  PAYPAL_PAYMENT_DETAILS_SERIALIZER,
} from '../../paypal-checkout/converters/converters';
import {
  PayPalBillingAgreementPaymentRequest,
  PayPalBillingAgreementToken,
  PaypalOrder,
  PayPalPaymentRequest
} from '../../paypal-checkout/models/paypa-checkout-data.model';
import {ConverterService, OccEndpointsService, PAYMENT_DETAILS_NORMALIZER, PaymentDetails} from '@spartacus/core';
import {CheckoutAdapter} from '../../paypal-checkout/connectors/checkout.adapter';
import {Fields} from '../../paypal-checkout/models/paypal-payment-data.model';

@Injectable()
export class PaypalCheckoutAdapter implements CheckoutAdapter {

  constructor(
    protected http: HttpClient,
    protected occEndpoints: OccEndpointsService,
    protected converter: ConverterService
  ) {
  }

  protected getEndpoint(userId, cartId): string {
    return this.occEndpoints.getEndpoint('/users/' + userId + '/carts/' + cartId + '/paypal/paymentInfo');
  }

  public createOrder(
    cartId: string,
    userId: string,
    fundingSource: string
  ): Observable<PaypalOrder> {

    const params = new HttpParams()
    .set('fundingSource', fundingSource);

    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId, cartId)}/createOrder`;

    console.log(url);

    return this.http.post(url, {}, {headers, params})
    .pipe(this.converter.pipeable(PAYPAL_ORDER_NORMALIZER));
  }

  public createBillingAgreementToken(
    cartId: string,
    userId: string
  ): Observable<PayPalBillingAgreementToken> {

    const headers = new HttpHeaders({
      'Content-Type': 'application/json'
    });

    const url = `${this.getEndpoint(userId, cartId)}/createBillingAgreementToken`;

    return this.http.post(url, {}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_BILLING_AGREEMENT_TOKEN_NORMALIZER));
  }

  public createPayPalPaymentMethod(
    cartId: string,
    userId: string,
    payPalPaymentRequest: PayPalPaymentRequest,
    fields?: Fields,
    paymentMethodType?: string
  ): Observable<PaymentDetails> {

    payPalPaymentRequest = this.converter.convert(payPalPaymentRequest, PAYPAL_PAYMENT_DETAILS_SERIALIZER);

    const params: HttpParams = new HttpParams()
    .set('fields', fields)
    .append('paymentMethodType', paymentMethodType);

    const headers = new HttpHeaders({
      'Content-Type': 'application/json'
    });

    const url = `${this.getEndpoint(userId, cartId)}/createPayPalPayment`;

    return this.http.post(url, payPalPaymentRequest, {headers, params})
    .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));

  }

  public createPayPalBillingAgreementPayment(
    cartId: string,
    userId: string,
    payPalBillingAgreementPaymentRequest: PayPalBillingAgreementPaymentRequest,
    fields?: Fields,
    paymentMethodType?: string
  ): Observable<PaymentDetails> {

    console.log(paymentMethodType);
    const params: HttpParams = new HttpParams()
    .set('fields', fields)
    .append('paymentMethodType', paymentMethodType);
    console.log(params);

    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    payPalBillingAgreementPaymentRequest = this.converter
    .convert(payPalBillingAgreementPaymentRequest, PAYPAL_BILLING_AGREEMENT_PAYMENT_SERIALIZER);

    const url = `${this.getEndpoint(userId, cartId)}/createPayPalBillingAgreementPayment`;

    return this.http.post(url, payPalBillingAgreementPaymentRequest, {headers, params})
    .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
  }

  public processBillingAgreementExpressCheckout(
    userId: string,
    cartId: string
  ): Observable<PaymentDetails> {

    const url = `${this.getEndpoint(userId, cartId)}/payPalPayment`;

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, {}, {headers});
  }

  public setPayPalPaymentMethod(
    userId: string,
    cartId: string,
    paymentDetailsId: string
  ): Observable<any> {

    const params: HttpParams = new HttpParams().set('paymentDetailsId', paymentDetailsId);

    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = this.occEndpoints
    .getEndpoint('/users/' + userId + '/carts/' + cartId + '/paypal/paymentdetails');


    return this.http.put(url, {}, {headers, params});
  }
}
