import {Injectable} from '@angular/core';
import {
  CheckoutPaymentAdapter,
  ConverterService,
  OccCheckoutPaymentAdapter,
  OccEndpointsService,
  PAYMENT_DETAILS_NORMALIZER,
  PAYMENT_DETAILS_SERIALIZER,
  PaymentDetails
} from '@spartacus/core';
import {Observable} from 'rxjs';
import {map, mergeMap} from 'rxjs/operators';
import {HttpClient} from '@angular/common/http';

@Injectable()
export class PaypalCheckoutPaymentAdapter extends OccCheckoutPaymentAdapter implements CheckoutPaymentAdapter {

  private myDomParser: DOMParser;

  constructor(protected http: HttpClient,
              protected occEndpoints: OccEndpointsService,
              protected converter: ConverterService) {
    super(http, occEndpoints, converter);
    if (typeof DOMParser !== 'undefined') {
      this.myDomParser = new DOMParser();
    }
  }

  private myConvertToMap(
    paramList: { key: string; value: string }[]
  ): { [key: string]: string | undefined } {
    return paramList.reduce(function (
      result: { [key: string]: string | undefined },
      item
      ) {
        const key = item.key;
        result[key] = item.value;
        return result;
      },
      {});
  }

  public create(
    userId: string,
    cartId: string,
    paymentDetails: PaymentDetails
  ): Observable<PaymentDetails> {
    paymentDetails = this.converter.convert(
      paymentDetails,
      PAYMENT_DETAILS_SERIALIZER
    );
    return this.getProviderSubInfo(userId, cartId).pipe(
      map((data) => {
        const labelsMap = this.myConvertToMap(data.mappingLabels.entry) as {
          [key: string]: string;
        };
        return {
          url: data.postUrl,
          parameters: this.myGetParamsForPaymentProvider(
            paymentDetails,
            data.parameters.entry,
            labelsMap
          ),
          mappingLabels: labelsMap,
        };
      }),
      mergeMap((sub) => {
        // create a subscription directly with payment provider
        return this.createSubWithProvider(sub.url, sub.parameters).pipe(
          map((response) => this.myExtractPaymentDetailsFromHtml(response)),
          mergeMap((fromPaymentProvider) => {
            fromPaymentProvider['defaultPayment'] =
              paymentDetails.defaultPayment ?? false;
            fromPaymentProvider['savePaymentInfo'] =
              paymentDetails.saved ?? false;
            return this.createDetailsWithParameters(
              userId,
              cartId,
              fromPaymentProvider
            ).pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
          })
        );
      })
    );
  }

  private myExtractPaymentDetailsFromHtml(
    html: string
  ): { [key: string]: string | boolean } {
    const domdoc = this.myDomParser.parseFromString(html, 'text/xml');
    const responseForm = domdoc.getElementsByTagName('form')[0];
    const inputs = responseForm.getElementsByTagName('input');

    const values: { [key: string]: string | boolean } = {};
    for (let i = 0; inputs[i]; i++) {
      const input = inputs[i];
      const name = input.getAttribute('name');
      const value = input.getAttribute('value');
      if (name && name !== '{}' && value && value !== '') {
        values[name] = value;
      }
    }

    return values;
  }

  private myGetParamsForPaymentProvider(
    paymentDetails: PaymentDetails,
    parameters: { key: string; value: string }[],
    mappingLabels: { [key: string]: string }
  ) {
    const params = this.myConvertToMap(parameters);
    params[mappingLabels['hybris_account_holder_name']] =
      paymentDetails.accountHolderName;
    params[mappingLabels['hybris_card_type']] = paymentDetails.cardType?.code;
    params[mappingLabels['hybris_card_number']] = paymentDetails.cardNumber;
    if (mappingLabels['hybris_combined_expiry_date'] === 'true') {
      params[mappingLabels['hybris_card_expiry_date']] =
        paymentDetails.expiryMonth +
        mappingLabels['hybris_separator_expiry_date'] +
        paymentDetails.expiryYear;
    } else {
      params[mappingLabels['hybris_card_expiration_month']] =
        paymentDetails.expiryMonth;
      params[mappingLabels['hybris_card_expiration_year']] =
        paymentDetails.expiryYear;
    }
    params[mappingLabels['hybris_card_cvn']] = paymentDetails.cvn;

    // billing address
    params[mappingLabels['hybris_billTo_country']] =
      paymentDetails.billingAddress?.country?.isocode;
    params[mappingLabels['hybris_billTo_firstname']] =
      paymentDetails.billingAddress?.firstName;
    params[mappingLabels['hybris_billTo_lastname']] =
      paymentDetails.billingAddress?.lastName;
    params[mappingLabels['hybris_billTo_street1']] =
      paymentDetails.billingAddress?.line1 +
      ' ' +
      paymentDetails.billingAddress?.line2;
    params[mappingLabels['hybris_billTo_city']] =
      paymentDetails.billingAddress?.town;
    if (paymentDetails.billingAddress?.region) {
      params[mappingLabels['hybris_billTo_region']] =
        paymentDetails.billingAddress.region.isocodeShort;
    } else {
      params[mappingLabels['hybris_billTo_region']] = '';
    }
    params[mappingLabels['hybris_billTo_postalcode']] =
      paymentDetails.billingAddress?.postalCode;
    return params;
  }

}
