import {Injectable} from '@angular/core';
import {PaymentDetails} from '@spartacus/core';
import {Card} from '@spartacus/storefront';
import {PaymentType, PaypalCheckoutData} from '../../models';


@Injectable({
  providedIn: 'root'
})
export class PaypalPaymentMethodPresentationService {

  constructor() {
  }

  preparePaymentMethod(content: Card, paymentMethod: PaymentDetails, checkoutData?: PaypalCheckoutData): Card {
    if (paymentMethod.cardType.code === PaymentType.PAYPAL) {
      return this.preparePaypalMethod(content, paymentMethod);
    }
    if (paymentMethod.cardType.code === PaymentType.LPM) {
      return this.prepareLPMMethod(content, paymentMethod);
    }
    if (paymentMethod.cardType.code === PaymentType.VENMO) {
      return this.prepareVenmo(content, paymentMethod, checkoutData);
    }
    if (paymentMethod.cardType.code === PaymentType.CARD) {
      return this.prepareCard(content, paymentMethod);
    }

    this.prepareDefault(content, paymentMethod);
    return content;
  }

  private preparePaypalMethod(content: Card, paymentMethod: PaymentDetails): Card {
    content.text = [paymentMethod.billingAddress?.email];
    this.addBillingAddress(content, paymentMethod);
    content.img = 'https://www.paypalobjects.com/webstatic/en_US/i/buttons/pp-acceptance-small.png';
    return content;
  }

  private prepareVenmo(content: Card, paymentMethod: PaymentDetails, checkoutData: PaypalCheckoutData): Card {
    content = this.preparePaypalMethod(content, paymentMethod);
    content.img = checkoutData.configurationData.venmoIconUrl;
    return content;
  }

  private prepareCard(content: Card, paymentMethod: PaymentDetails): Card {
    content.textBold = 'PayPal Debit/Credit Card';
    content.text = [paymentMethod.accountHolderName,
      paymentMethod.billingAddress?.email];
    this.addBillingAddress(content, paymentMethod);
    content.img = 'https://www.paypalobjects.com/webstatic/en_US/i/buttons/pp-acceptance-small.png';
    return content;
  }

  private addBillingAddress(content: Card, paymentMethod: PaymentDetails): Card {
    content.paragraphs = [
      {
        text: [
          paymentMethod.billingAddress?.line1,
          paymentMethod.billingAddress?.town + ', ' +
          paymentMethod.billingAddress?.region?.isocode + ', ' +
          paymentMethod.billingAddress?.country?.isocode,
          paymentMethod.billingAddress?.postalCode,
        ],
      },
    ];
    return content
  }

  private prepareLPMMethod(content: Card, paymentMethod: PaymentDetails): Card {
    content.textBold = '';
    content.text = ['LocalPayment'];
    return content;
  }

  private prepareDefault(content: Card, paymentMethod: PaymentDetails): Card {
    content.text = [paymentMethod.cardType.name].concat(content.text);
    this.addBillingAddress(content, paymentMethod);
    return content;
  }

}
