import {Injectable} from '@angular/core';
import {HttpErrorResponse, HttpRequest} from '@angular/common/http';
import {
  ActiveCartService,
  BadRequestHandler, CheckoutService, ErrorModel,
  GlobalMessageService,
  GlobalMessageType,
  HttpResponseStatus,
  Priority
} from '@spartacus/core';
import {cart} from '@spartacus/assets/translations/en/cart';
import {map} from 'rxjs/operators';
import {Observable} from 'rxjs';


@Injectable({
  providedIn: 'root'
})
export class PaypalErrorHandlerService extends BadRequestHandler {

  responseStatus = HttpResponseStatus.BAD_REQUEST;
  isCartEmpty$: Observable<boolean>;


  constructor(
    protected globalMessageService: GlobalMessageService,
    protected checkoutService: CheckoutService,
    protected cartService: ActiveCartService
  ) {
    super(globalMessageService);
  }

  handleError(
    request: HttpRequest<any>,
    errorResponse: HttpErrorResponse
  ): void {
    super.handleError(request, errorResponse);
    if (errorResponse.error?.errors?.length > 0) {
      errorResponse.error?.errors.forEach((error) => {
        if (error.type === 'AdapterError') {
          this.globalMessageService.add(
            {key: 'error.adapterError'},
            GlobalMessageType.MSG_TYPE_ERROR
          );
        }
        if (error.type === 'PayPalConnectError') {
          this.globalMessageService.add(
            {key: 'error.payPalConnectError'},
            GlobalMessageType.MSG_TYPE_ERROR
          );
        }
        if (error.type === 'ModelSavingError') {
          const message = ' email already exist. Please checkout with a different email to register using a guest account';
          let email;
          this.checkoutService.getOrderDetails()
          .pipe().subscribe(x => email = (x.user.uid.split('|').pop()));
          this.globalMessageService.add(email + message,
            GlobalMessageType.MSG_TYPE_ERROR)
        }
      });
    }
  }

  protected handleBadCartRequest(
    _request: HttpRequest<any>,
    response: HttpErrorResponse
  ): void {
    this.getErrors(response)
      .filter((e) => this.isCartNotFoundError(e))
      .forEach(() => {
        if(this.isCartEmpty()) {
          this.globalMessageService.add(
            { key: 'httpHandlers.cartNotFound' },
            GlobalMessageType.MSG_TYPE_ERROR
          );
        }
      });
  }

  getPriority(): Priority {
    return Priority.NORMAL;
  }

  isCartNotFoundError(error: ErrorModel): boolean {
    return (
      error.reason === 'notFound' &&
      error.subjectType === 'cart' &&
      !this.isSelectiveCart(error.subject)
    );
  }

  isSelectiveCart(cartId = ''): boolean {
    return cartId.startsWith('selectivecart');
  }

  isCartEmpty(): boolean{
    this.isCartEmpty$ = this.cartService
      .getActive()
      .pipe(map((cart) => !(cart && cart.totalItems && cart.totalItems > 0)));
    return this.isCartEmpty$._isScalar;
}
}
