import {Injectable} from '@angular/core';
import {PaypalPaymentDetailsService} from '../../facade/paypal-payment-details.service';
import {Fields, PageType, PayPalBillingAgreementPaymentRequest, PayPalPaymentRequest} from '../../models';
import {first, map, takeWhile} from 'rxjs/operators';
import {Router} from '@angular/router';
import {PaypalUtilsService} from '../utils/paypal-utils.service';
import {
  ActiveCartService,
  CheckoutActions,
  CheckoutConnector,
  CheckoutDetails,
  CheckoutService,
  MultiCartService,
  StateWithCheckout,
  UserIdService
} from '@spartacus/core';
import {Observable} from 'rxjs';
import {Store} from '@ngrx/store';

@Injectable({
  providedIn: 'root'
})
export class PaypalProcessPaymentCheckoutService {

  constructor(
    protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
    protected paypalUtilsService: PaypalUtilsService,
    protected router: Router,
    protected checkoutService: CheckoutService,
    protected activeCartService: ActiveCartService,
    protected multiCartService: MultiCartService,
    protected userIdService: UserIdService,
    protected checkoutConnector: CheckoutConnector,
    protected checkoutStore: Store<StateWithCheckout>
  ) { }

  public processPaypalResponse(data, pageType: PageType, paymentMethodType?: string): void {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();

    let cartId;
    this.activeCartService
    .getActiveCartId()
    .subscribe((activeCartId) => (cartId = activeCartId))
    .unsubscribe();
    let payPalPaymentRequest: PayPalPaymentRequest;

    if (pageType === PageType.MARK) {
      payPalPaymentRequest = {
        orderId: data.orderID,
        payPalFlow: 'MARK_CHECKOUT'
      };
    } else if (pageType === PageType.CART)
      payPalPaymentRequest = {
        orderId: data.orderID,
        payPalFlow: 'EXPRESS_CHECKOUT'
      };

    this.paypalPaymentDetailsService.createPayPalPaymentMethod(payPalPaymentRequest, Fields.Full, paymentMethodType)
      .pipe(takeWhile(paymentDetails => paymentDetails === undefined || paymentDetails.id === undefined, true))
      .subscribe(res => {
        this.loadCheckoutDetails(userId, cartId).pipe(first()).subscribe(value => {
          this.multiCartService.reloadCart(cartId);
          if (res.cardType.code === 'local_payment'){
            this.checkoutService.placeOrder(true);
          }else {
            this.router.navigate(['/checkout/review-order']);
          }
        });
      });
  }

  public savePaymentDetailsForNewUser(data, accessToken: string, onSuccessCallback, paymentMethodType?: string): void {
    let payPalBillingAgreementPaymentRequest: PayPalBillingAgreementPaymentRequest;

    payPalBillingAgreementPaymentRequest = {
      billingAgreementTokenId: data.billingToken,
      flow: 'BILLING_AGREEMENT_FLOW',
      isSavePaymentInfo: true
    };

    this.paypalPaymentDetailsService.createPayPalPaymentAndAddAddressForNewUser(payPalBillingAgreementPaymentRequest, accessToken,
      Fields.Full, paymentMethodType)
    .pipe(takeWhile(status => status === undefined, true))
    .subscribe(res => {
      onSuccessCallback(res);
    });


  }

  public processBillingAgreementPaypalResponse(data, pageType: PageType, paymentMethodType?: string): void {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();

    let cartId;
    this.activeCartService
    .getActiveCartId()
    .subscribe((activeCartId) => (cartId = activeCartId))
    .unsubscribe();

    let payPalBillingAgreementPaymentRequest: PayPalBillingAgreementPaymentRequest;

    if (pageType === PageType.MARK) {
      payPalBillingAgreementPaymentRequest = {
        billingAgreementTokenId: data.billingToken,
        flow: 'BILLING_AGREEMENT_FLOW',
        isSavePaymentInfo: this.paypalUtilsService.savePaymentInfo
      };
    } else if (pageType === PageType.CART) {
      payPalBillingAgreementPaymentRequest = {
        billingAgreementTokenId: data.billingToken,
        flow: 'EXPRESS_BILLING_AGREEMENT_FLOW',
        isSavePaymentInfo: true
      };
    }

    this.paypalPaymentDetailsService.createPayPalBillingAgreementPayment(payPalBillingAgreementPaymentRequest, Fields.Full, paymentMethodType)
      .pipe(takeWhile(paymentDetails => paymentDetails === undefined || paymentDetails.id === undefined, true))
      .subscribe(res => {
        this.loadCheckoutDetails(userId, cartId).pipe(first()).subscribe(value => {
          this.multiCartService.reloadCart(cartId);
          this.router.navigate(['/checkout/review-order']);
        });
      });
  }

  private loadCheckoutDetails(userId: string, cartId: string): Observable<CheckoutDetails> {
    return this.checkoutConnector.loadCheckoutDetails(userId, cartId)
    .pipe(
      map(
        (data: CheckoutDetails) => {
          this.checkoutStore.dispatch(new CheckoutActions.LoadCheckoutDetailsSuccess(data));
          return data;
        }
      )
    );
  }


}
