import { Injectable } from '@angular/core';
import {PaypalUtilsService} from '../utils';
import {PageType, PayPalButtonStyle, PaypalConfiguration, PayPalPaymentRequest} from '../../models';
import { PaypalPaymentDetailsService } from '../../facade/paypal-payment-details.service';
import {PaypalProcessPaymentCheckoutService} from './paypal-process-payment-checkout.service';
import { ActiveCartService, Cart, OrderEntry } from '@spartacus/core';

@Injectable({
  providedIn: 'root'
})
export class PaypalCartCheckoutService {

  constructor(
    protected paypalUtilsService: PaypalUtilsService,
    protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
    protected paypalProcessPaymentCheckoutService: PaypalProcessPaymentCheckoutService,
    protected activeCartService: ActiveCartService
  ) { }

  initializePayPalCheckout(configurationData: PaypalConfiguration, pageType: PageType, payPalButtonContainer, buttonStyle: PayPalButtonStyle): void{

    if (configurationData.isBillingAgreementFlowEnabled){
      this.paypalUtilsService.loadPayPalSDK(configurationData, pageType, () => {
        this.renderBillingAgreementButton(payPalButtonContainer, buttonStyle, pageType);
      });
    }else {
      this.paypalUtilsService.loadPayPalSDK(configurationData, pageType, () => {
        this.activeCartService.getActive().subscribe(value => {
          this.renderButton(value, payPalButtonContainer, buttonStyle, pageType);
        }).unsubscribe();
      });
    }

  }

  renderBillingAgreementButton(payPalButtonContainer, buttonStyle: PayPalButtonStyle, pageType: PageType): void{
    try {
      if (typeof (window as any).paypalSdk !== undefined) {
        let fundingSource;

        (window as any).paypalSdk.Buttons({
          style: buttonStyle,

          onClick: (button) => {
            fundingSource = button.fundingSource;
          },

          createBillingAgreement: () => {
            return this.paypalPaymentDetailsService.createBillingAgreementToken().toPromise()
              .then((res) => {
                return res.tokenId;
              });
          },
          onApprove: (data) => {

            return this.paypalProcessPaymentCheckoutService.processBillingAgreementPaypalResponse(data, pageType, fundingSource);
          },
          onError: (err) => {
            console.error('Error: ' + err, err);
          }
        }).render(payPalButtonContainer);
      }
    } catch (err) {
      console.log(err.message);
    }
  }

  renderButton(cart: Cart, payPalButtonContainer, buttonStyle: PayPalButtonStyle, pageType: PageType): void{
    let fundingSource;
    try {
      if (typeof (window as any).paypalSdk !== undefined) {

        (window as any).paypalSdk.Buttons(
          {
            style: buttonStyle,

            onClick: (button) => {
              fundingSource = button.fundingSource;
            },

            createOrder: (data, actions) => {
              return actions.order.create({
                purchase_units: [{
                  amount: {
                    value: cart.totalPrice.value,
                    currency_code: cart.totalPrice.currencyIso
                  }
                }]
              });
            },
            onApprove: (data) => {

              return this.paypalProcessPaymentCheckoutService.processPaypalResponse(data, pageType, fundingSource);

            },
            onError: (err) => {
              console.error('Error: ' + err, err);
            }
          }).render(payPalButtonContainer);
      }
    } catch (err) {
      console.log(err.message);
    }
  }
}
