import {Injectable} from '@angular/core';
import {
  PaypalConfiguration,
  PayPalConnectB2BData,
  PayPalConnectB2BRegisterData,
  PayPalConnectComponent,
  PayPalConnectData
} from '../models';
import {PaypalUtilsService} from '../services/utils';
import {from, Observable, of} from 'rxjs';
import {PaypalConnectConnector} from '../connectors/paypal-connect.connector';
import {AuthService, UserIdService} from '@spartacus/core';
import {Router} from '@angular/router';

@Injectable({
  providedIn: 'root'
})
export class PaypalConnectService {

  constructor(
    protected paypalUtilsService: PaypalUtilsService,
    protected userIdService: UserIdService,
    protected paypalConnectConnector: PaypalConnectConnector,
    protected router: Router,
    protected auth: AuthService
  ) { }


  public exchangeAuthorizationCode(authorizationCode: string): Observable<PayPalConnectData> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    if (userId) {
      return this.paypalConnectConnector.exchangeAuthorizationCode(authorizationCode, userId);
    }
    return of(null);
  }

  public login(accessToken: string): Observable<boolean> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    if(userId) {
      this.paypalConnectConnector.login(accessToken, userId).subscribe(loginData => {
        from(this.auth.loginWithCredentials(loginData.login, loginData.password)).subscribe(te => {
          this.userIdService
          .getUserId()
          .subscribe((occUserId) => (userId = occUserId))
          .unsubscribe();
          this.paypalConnectConnector.afterLogin(accessToken, userId).subscribe(flag => {})
        });
      });
      return of(true);
    }
    return of(true);
  }

  public register(accessToken: string): Observable<string> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    if(userId) {
      return this.paypalConnectConnector.register(accessToken, userId);
    }
  }

  public exchangeAuthorizationCodeB2B(authorizationCode: string): Observable<PayPalConnectB2BData> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    if (userId) {
      return this.paypalConnectConnector.exchangeAuthorizationCodeB2B(authorizationCode, userId);
    }
    return of(null);
  }

  public registerB2BCustomer(
    accessToken: string,
    registerData: PayPalConnectB2BRegisterData
  ): Observable<string> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    if(userId) {
      registerData.accessToken = accessToken;
      return this.paypalConnectConnector.registerB2BCustomer(accessToken, registerData, userId);
    }
    return of(undefined);
  }

  public initializeConnectWithPayPal(componentData: PayPalConnectComponent, configurationData: PaypalConfiguration): void {
      this.paypalUtilsService.loadConnectWithPayPalSdk(configurationData,
        () => this.renderConnectWithPayPalButton(componentData, configurationData));
  }

  protected renderConnectWithPayPalButton(componentData: PayPalConnectComponent, configurationData: PaypalConfiguration): void {
    try {
      if (typeof (window as any).paypal !== undefined) {
        let authend = '';
        if (configurationData.environmentType === 'sandbox'){
            authend = 'sandbox';
        }

        (window as any).paypal.use( ['login'], function (login) {
          login.render ({
            "appid": configurationData.client_id,
            "authend": authend,
            "scopes": configurationData.payPalConnectScopes,
            "containerid": componentData.buttonDiv,
            "responseType": configurationData.payPalConnectResponseType,
            "theme": componentData.buttonTheme,
            "buttonType": componentData.buttonType,
            "buttonShape": componentData.buttonShape.toLocaleLowerCase(),
            "buttonSize": componentData.buttonSize.toLocaleLowerCase(),
            "fullPage": componentData.fullPage,
            "returnurl": configurationData.payPalConnectReturnUrl,
            "nonce": '1111111'
          });
        });
      }
    } catch (err) {
      console.log(err.message);
    }
  }

}
