/**
 *
 */
package com.braintree.controllers;


import com.braintree.email.emailServices.BraintreeLocalPaymentReversedEmailEventService;
import com.braintree.enums.BTFraudToolReviewStatus;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreePaymentFacade;
import com.braintree.service.impl.DefaultBrainTreeFraudCheckToolService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.LocalPaymentReversed;
import com.braintreegateway.TransactionReview;
import com.braintreegateway.WebhookNotification;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.site.BaseSiteService;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;

@UnitTest
@RunWith(MockitoJUnitRunner.class)
public class BraintreeWebhookListenerTest {

    private static final String PAYMENT_ID = "testId";
    private static final String TRANSACTION_ID = "transactionId";
    private static final String APPROVE_DECISION = "Approve";

    @Mock
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;
    @Mock
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;
    @Mock
    private HttpServletRequest request;
    @Mock
    private HttpServletResponse response;
    @Mock
    private WebhookNotification webhookNotification;
    @Mock
    private DefaultBrainTreeFraudCheckToolService btFraudCheckToolService;
    @Mock
    private TransactionReview transactionReview;
    @Mock
    private OrderModel orderModel;
    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;
    @Mock
    private PaymentTransactionModel paymentTransactionModel;
    @Mock
    private CartModel cartModel;
    @Mock
    private LocalPaymentReversed localPaymentReversed;
    @Mock
    private BaseSiteService baseSiteService;
    @Mock
    private BraintreeLocalPaymentReversedEmailEventService localPaymentReversedEmailEventService;
    @Mock
    private CustomerModel customerModel;

    @InjectMocks
    private BraintreeWebhookListener braintreeWebhookListener;

    @Before
    public void initSetup() {
        when(brainTreePaymentFacade.getWebhookNotification(request)).thenReturn(webhookNotification);
    }

    @Test
    public void handleLocalPaymentRefund() throws InvalidCartException {
        when(localPaymentReversed.getPaymentId()).thenReturn(PAYMENT_ID);
        when(webhookNotification.getLocalPaymentReversed()).thenReturn(localPaymentReversed);
        when(webhookNotification.getKind()).thenReturn(WebhookNotification.Kind.LOCAL_PAYMENT_REVERSED);
        when(brainTreePaymentFacade.getWebhookNotification(any())).thenReturn(webhookNotification);
        when(brainTreePaymentFacade.getCartByPaymentId(PAYMENT_ID)).thenReturn(cartModel);
        when(cartModel.getUser()).thenReturn(customerModel);
        when(brainTreeCheckoutFacade.placeOrderForLocalPaymentReversed(cartModel)).thenReturn(orderModel);

        ResponseEntity<String> handleLocalPaymentRefundResult = braintreeWebhookListener
                .handleLocalPaymentRefund(request, response);

        verify(brainTreeCheckoutFacade).processLocalPaymentReversedWebhook(cartModel);
        verify(localPaymentReversedEmailEventService).sendBraintreeLocalPaymentReversedEmailEvent(customerModel, orderModel);

        assertEquals(HttpStatus.OK, handleLocalPaymentRefundResult.getStatusCode());

    }

    @Test
    public void shouldNotProcessOtherNotificationThenTransactionReviewNotification() {
        when(brainTreePaymentFacade.getWebhookNotification(request)).thenReturn(webhookNotification);
        when(webhookNotification.getKind()).thenReturn(WebhookNotification.Kind.LOCAL_PAYMENT_REVERSED);

        ResponseEntity<String> actual = braintreeWebhookListener.doHandleBtFraudCheckWebhook(request);

        assertEquals(HttpStatus.OK, actual.getStatusCode());

        verifyZeroInteractions(btFraudCheckToolService);
    }

    @Test
    public void shouldPopulateTransactionReviewDecision() {
        when(brainTreePaymentFacade.getWebhookNotification(request)).thenReturn(webhookNotification);
        when(webhookNotification.getKind()).thenReturn(WebhookNotification.Kind.TRANSACTION_REVIEWED);
        when(webhookNotification.getTransactionReview()).thenReturn(transactionReview);
        when(transactionReview.getTransactionId()).thenReturn(TRANSACTION_ID);
        when(transactionReview.getDecision()).thenReturn(APPROVE_DECISION);
        when(brainTreeTransactionService.findTransactionByTransactionId(TRANSACTION_ID)).thenReturn(paymentTransactionModel);
        when(btFraudCheckToolService.translateStatusFromBtToHybris(APPROVE_DECISION)).thenReturn(BTFraudToolReviewStatus.ACCEPTED);

        ResponseEntity<String> actual = braintreeWebhookListener.doHandleBtFraudCheckWebhook(request);

        assertEquals(HttpStatus.OK, actual.getStatusCode());

        verify(btFraudCheckToolService).updateTransactionStatus(paymentTransactionModel, BTFraudToolReviewStatus.ACCEPTED);
        verify(btFraudCheckToolService).saveBtFraudToolReviewStatus(paymentTransactionModel, BTFraudToolReviewStatus.ACCEPTED);
    }


}
