package braintreehac.services;

import de.hybris.platform.catalog.CatalogVersionService;
import de.hybris.platform.catalog.model.CatalogModel;
import de.hybris.platform.catalog.model.CatalogVersionModel;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.site.CMSSiteModel;
import de.hybris.platform.cms2.servicelayer.services.CMSComponentService;
import de.hybris.platform.cms2.servicelayer.services.CMSSiteService;
import de.hybris.platform.cms2.servicelayer.services.admin.CMSAdminSiteService;
import de.hybris.platform.cms2.model.contents.components.SimpleCMSComponentModel;
import de.hybris.platform.cmsfacades.cmsitems.CMSItemFacade;
import de.hybris.platform.core.model.ItemModel;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Component;
import javax.annotation.Resource;
import java.util.*;

@Component
public class BraintreeConnectWithPayPalButtonConfigurationsService {

    private static final Logger LOG = Logger.getLogger(BraintreeConnectWithPayPalButtonConfigurationsService.class);

    private static final String STAGED = "Staged";
    private static final String ONLINE = "Online";
    private static final String B2B_PREFIX = "B2B";
    private static final String SPARTACUS_PREFIX = "SPA";
    private static final String CHECKOUT_PREFIX = "Checkout";

    @Resource(name = "cmsAdminSiteService")
    private CMSAdminSiteService cmsAdminSiteService;

    @Resource(name = "cmsSiteService")
    private CMSSiteService cmsSiteService;

    @Resource(name = "catalogVersionService")
    private CatalogVersionService catalogVersionService;

    @Resource(name = "cmsItemFacade")
    private CMSItemFacade cmsItemFacade;

    @Resource(name = "cmsComponentService")
    private CMSComponentService cmsComponentService;

    @Resource(name = "brainTreeCatalogSynchronizationService")
    private BrainTreeCatalogSynchronizationService brainTreeCatalogSynchronizationService;

    /**
     * This method is used to get PayPal connect item by uId
     * @param componentByUid componentByUid
     * @return payPalConnectComponent
     * @throws CMSItemNotFoundException if item is not found
     */
    public Map<String, Object> getPayPalConnectItemByUid(final SimpleCMSComponentModel componentByUid)
            throws CMSItemNotFoundException {
        setActiveCatalogVersion(componentByUid.getCatalogVersion());
        Map<String, Object> payPalConnectComponent = cmsItemFacade.getCMSItemByUuid(componentByUid.getUid());
        return payPalConnectComponent;
    }

    /**
     * This method is used to get component by uId
     * @param uid uid
     * @return cms component model
     */
    public SimpleCMSComponentModel getComponentByUid(final String uid) {
        try {
            return cmsComponentService.getSimpleCMSComponent(uid);
        } catch (CMSItemNotFoundException e) {
            LOG.error("Wasn't found CMS item with uid - " + uid);
        }
        return null;
    }

    /**
     * This method is used to process component updating
     * @param payPalConnectComponentUid payPalConnectComponentUid
     * @param refreshedParameters refreshedParameters
     */
    public void processComponentUpdating(final String payPalConnectComponentUid,
                                         Map<String, String[]> refreshedParameters) {
        final SimpleCMSComponentModel payPalConnectComponent = getComponentByUid(payPalConnectComponentUid);
        final SimpleCMSComponentModel payPalConnectComponentB2B = getComponentByUid(payPalConnectComponentUid + B2B_PREFIX);
        final SimpleCMSComponentModel payPalConnectComponentSPA = getComponentByUid(payPalConnectComponentUid + SPARTACUS_PREFIX);
        final SimpleCMSComponentModel payPalConnectComponentCheckout = getComponentByUid(payPalConnectComponentUid + CHECKOUT_PREFIX);
        final SimpleCMSComponentModel payPalConnectComponentB2BCheckout = getComponentByUid(payPalConnectComponentUid
                + B2B_PREFIX + CHECKOUT_PREFIX);
        List<SimpleCMSComponentModel> itemsToSynchronize = new ArrayList<>();

        if (payPalConnectComponent != null) {
            itemsToSynchronize.add(payPalConnectComponent);
        }
        if (payPalConnectComponentB2B != null) {
            itemsToSynchronize.add(payPalConnectComponentB2B);
        }
        if (payPalConnectComponentSPA != null) {
            itemsToSynchronize.add(payPalConnectComponentSPA);
        }
        if (payPalConnectComponentCheckout != null) {
            itemsToSynchronize.add(payPalConnectComponentCheckout);
        }
        if (payPalConnectComponentB2BCheckout != null) {
            itemsToSynchronize.add(payPalConnectComponentB2BCheckout);
        }

        updatePayPalConnectComponents(itemsToSynchronize, refreshedParameters);
        itemsToSynchronize.forEach(this::synchronizePayPalConnectComponent);
    }

    /**
     * This method is used to update PayPal connect component
     * @param componentsToUpdate componentsToUpdate
     * @param refreshedParameters refreshedParameters
     */
    private void updatePayPalConnectComponents(List<SimpleCMSComponentModel> componentsToUpdate,
                                               Map<String, String[]> refreshedParameters) {
        componentsToUpdate.forEach(component -> {
            try {
                updateComponent(component, refreshedParameters);
            } catch (Exception e) {
                String message = "Can not update component with uid: " + component.getUid() +
                        " Error message: " + e.getMessage();
                LOG.error(message, e);
            }
        });
    }

    private void synchronizePayPalConnectComponent(final SimpleCMSComponentModel payPalConnectComponent) {
        List<ItemModel> itemsToSynchronize = new ArrayList<>();
        itemsToSynchronize.add(payPalConnectComponent);
        synchronizeComponent(itemsToSynchronize);
    }

    private void synchronizeComponent(List<ItemModel> itemsToSynchronize) {
        final CatalogVersionModel source = getActiveCatalogVersion();
        final CatalogVersionModel target = getTargetCatalogVersion(source);
        getBrainTreeCatalogSynchronizationService().synchronizeItems(itemsToSynchronize, source, target);
    }

    /**
     * This method is used to update PayPal Component Values
     * @param payPalConnectItem payPalConnectItem
     * @param refreshedParameters refreshedParameters
     * @return updated values
     */
    public Map<String, Object> updatePayPalConnectComponentValues(Map<String, Object> payPalConnectItem,
                                                                  Map<String, String[]> refreshedParameters) {
        LOG.warn("Start component update process");
        Set<String> refreshedParametersKeys = refreshedParameters.keySet();
        Iterator<String> iterator = refreshedParametersKeys.iterator();

        while (iterator.hasNext()) {
            String refreshedParameterKey = iterator.next();
            String refreshedParameterValue = Arrays.stream(refreshedParameters.get(refreshedParameterKey)).findFirst()
                    .get();
            if (payPalConnectItem.containsKey(refreshedParameterKey)) {
                payPalConnectItem.replace(refreshedParameterKey, refreshedParameterValue);
            } else {
                payPalConnectItem.put(refreshedParameterKey, refreshedParameterValue);
            }
        }
        return payPalConnectItem;
    }

    private void updatePayPalConnectComponentItem(final String uuid, final Map<String,
            Object> updatedPayPalConnectComponent) throws CMSItemNotFoundException {
        cmsItemFacade.updateItem(uuid, updatedPayPalConnectComponent);
        LOG.warn("Updated component");
    }

    private void updateComponent(SimpleCMSComponentModel payPalConnectComponent,
                                 Map<String, String[]> refreshedParameters) throws CMSItemNotFoundException {
        Map<String, Object> updatedPayPalConnectValues = updatePayPalConnectComponentValues(
                getPayPalConnectItemByUid(payPalConnectComponent), refreshedParameters);
        updatePayPalConnectComponentItem(updatedPayPalConnectValues.get("uuid").toString(),
                updatedPayPalConnectValues);
    }

    private void setActiveCatalogVersion(final CatalogVersionModel catalogVersion) {
        String catalogID = catalogVersion.getCatalog().getId();
        String versionName = STAGED;
        CatalogVersionModel version = getCatalogVersionService().getCatalogVersion(catalogID, versionName);
        cmsAdminSiteService.setActiveCatalogVersion(version);
    }

    public CatalogVersionModel getActiveCatalogVersion() {
        return cmsAdminSiteService.getActiveCatalogVersion();
    }

    /**
     * This method is used to get target catalog version
     * @param sourceCatalogVersion source catalog version
     * @return catalog version model
     */
    public CatalogVersionModel getTargetCatalogVersion(final CatalogVersionModel sourceCatalogVersion) {
        String catalogId = sourceCatalogVersion.getCatalog().getId();
        String versionName = ONLINE;
        if (ONLINE.equals(sourceCatalogVersion.getVersion())) {
            versionName = STAGED;
        }
        return getCatalogVersionService().getCatalogVersion(catalogId, versionName);
    }

    /**
     * This method is used to process request
     * @return result of the operation
     */
    public boolean processNormalRequest() {
        Collection<CatalogModel> allCatalogs = new ArrayList<>();
        Collection<CatalogVersionModel> catalogVersions = new ArrayList<>();
        for (CMSSiteModel site : getCmsSiteService().getSites()) {
            allCatalogs.addAll(getCmsSiteService().getAllCatalogs(site));
            for (CatalogModel catalog : getCmsSiteService().getAllCatalogs(site)) {
                catalogVersions.add(catalog.getActiveCatalogVersion());
            }
        }
        getCatalogVersionService().setSessionCatalogVersions(catalogVersions);
        return true;
    }

    public CMSSiteService getCmsSiteService() {
        return cmsSiteService;
    }

    public void setCmsSiteService(CMSSiteService cmsSiteService) {
        this.cmsSiteService = cmsSiteService;
    }

    public CMSAdminSiteService getCmsAdminSiteService() {
        return cmsAdminSiteService;
    }

    public void setCmsAdminSiteService(CMSAdminSiteService cmsAdminSiteService) {
        this.cmsAdminSiteService = cmsAdminSiteService;
    }

    public CatalogVersionService getCatalogVersionService() {
        return catalogVersionService;
    }

    public void setCatalogVersionService(CatalogVersionService catalogVersionService) {
        this.catalogVersionService = catalogVersionService;
    }

    public BrainTreeCatalogSynchronizationService getBrainTreeCatalogSynchronizationService() {
        return brainTreeCatalogSynchronizationService;
    }
}