package braintreehac.services;

import com.braintree.enums.BraintreePageType;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import de.hybris.platform.braintreehac.data.GooglePayButtonStyleData;
import de.hybris.platform.braintreehac.data.PayPalButtonStyleData;
import de.hybris.platform.braintreehac.data.ApplePayButtonStyleData;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.OutputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

@Component
public class BrainTreeButtonConfigurationService {

    private static final Logger LOG = Logger.getLogger(BrainTreeButtonConfigurationService.class);

    private static final String CART_BUTTON_CONFIG_FILE =
        System.getProperty("HYBRIS_CONFIG_DIR") + "/braintreeButtonConfig.properties";

    private static final String PAYPAL_CART_BUTTON_CONFIG = "payPal.cart.button.config";
    private static final String PAYPAL_MINICART_BUTTON_CONFIG = "payPal.mini.cart.button.config";
    private static final String PAYPAL_MARK_BUTTON_CONFIG = "payPal.mark.button.config";
    private static final String GOOGLE_PAY_CART_BUTTON_CONFIG = "google.pay.cart.button.config";
    private static final String GOOGLE_PAY_MINI_CART_BUTTON_CONFIG = "google.pay.mini.cart.button.config";
    private static final String GOOGLE_PAY_MARK_BUTTON_CONFIG = "google.pay.mark.button.config";
    private static final String APPLE_PAY_CART_BUTTON_CONFIG = "apple.pay.cart.button.config";
    private static final String APPLE_PAY_MINI_CART_BUTTON_CONFIG = "apple.pay.mini.cart.button.config";
    private static final String APPLE_PAY_MARK_BUTTON_CONFIG = "apple.pay.mark.button.config";
    private static final String PAYPAL = "paypal";
    private static final String VERTICAL = "vertical";
    private Properties prop;
    private File propFile;
    private ObjectMapper mapper;
    private List<Integer> smartButtonHeight;
    private List<String> smartButtonLocation = new ArrayList<>();
    private List<String> smartButtonColor = new ArrayList<>();
    private List<String> smartButtonShape = new ArrayList<>();
    private List<String> smartButtonLabel = new ArrayList<>();
    private List<String> smartButtonLayout = new ArrayList<>();
    private List<String> smartGooglePayButtonSize = new ArrayList<>();
    private List<String> smartGooglePayButtonType = new ArrayList<>();
    private List<String> smartApplePayButtonColor = new ArrayList<>();
    private List<String> smartApplePayButtonType = new ArrayList<>();
    private List<String> smartApplePayButtonShape = new ArrayList<>();

    @PostConstruct
    public void init() {
        prop = new Properties();
        propFile = new File(CART_BUTTON_CONFIG_FILE);
        mapper = new ObjectMapper();
        mapper.configure(JsonGenerator.Feature.QUOTE_FIELD_NAMES, false);
        mapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        if (!propFile.isFile()) {
            LOG.info("Created new " + CART_BUTTON_CONFIG_FILE);

            setDefaultPayPalButtonConfigs(PAYPAL_MINICART_BUTTON_CONFIG);
            setDefaultPayPalButtonConfigs(PAYPAL_MARK_BUTTON_CONFIG);
            setDefaultPayPalButtonConfigs(PAYPAL_CART_BUTTON_CONFIG);

            setDefaultGooglePayButtonConfigs(GOOGLE_PAY_MINI_CART_BUTTON_CONFIG);
            setDefaultGooglePayButtonConfigs(GOOGLE_PAY_MARK_BUTTON_CONFIG);
            setDefaultGooglePayButtonConfigs(GOOGLE_PAY_CART_BUTTON_CONFIG);

            setDefaultApplePayButtonConfigs(APPLE_PAY_MINI_CART_BUTTON_CONFIG);
            setDefaultApplePayButtonConfigs(APPLE_PAY_MARK_BUTTON_CONFIG);
            setDefaultApplePayButtonConfigs(APPLE_PAY_CART_BUTTON_CONFIG);

            storeInPropertiesFile();
        }
        LOG.info("Loaded " + CART_BUTTON_CONFIG_FILE);
        loadPropertiesFromFile();
        smartButtonHeight = initHeightDefaultValues();

        //PayPal data
        Collections.addAll(smartButtonColor, "gold", "blue", "silver", "black");
        Collections.addAll(smartButtonShape, "pill", "rect");
        Collections.addAll(smartButtonLabel, "checkout", "pay", "buynow", PAYPAL);
        Collections.addAll(smartButtonLayout, "horizontal", VERTICAL);

        //GooglePay data
        Collections.addAll(smartGooglePayButtonSize, "static", "fill");
        Collections.addAll(smartGooglePayButtonType, "book", "buy", "checkout", "donate", "order", "pay", "plain", "subscribe");

        //Apple Pay area
        Collections.addAll(smartApplePayButtonColor, "white", "black", "white-outline");
        Collections.addAll(smartApplePayButtonType, "add-money", "book", "buy", "check-out",
                "continue", "contribute", "donate", "order", "pay", "plain", "reload", "rent", "set-up", "subscribe", "support", "tip", "top-up");
        Collections.addAll(smartApplePayButtonShape, "squareCorners", "pill-shaped");

        //Shared data
        Collections.addAll(smartButtonLocation, "cart", "minicart", "billing");
    }

    public void setProperty(String key, String value) {
        prop.setProperty(key, value);
        storeInPropertiesFile();
    }

    public void setDefaultPayPalButtonConfigs(final String key) {
        PayPalButtonStyleData miniCartButtonStyleData = createPayPalButtonStyleData("minicart", "gold", "rect",
                PAYPAL, VERTICAL, Boolean.FALSE, 35);
        PayPalButtonStyleData cartButtonStyleData = createPayPalButtonStyleData("cart", "gold", "rect", PAYPAL,
                VERTICAL, Boolean.FALSE, 35);
        PayPalButtonStyleData markButtonStyleData = createPayPalButtonStyleData("mark", "gold", "rect", PAYPAL,
                "horizontal", Boolean.FALSE, 55);

        String miniCartStyle = parsePayPalButtonConfigDataIntoJsonString(miniCartButtonStyleData);
        String cartButtonStyle = parsePayPalButtonConfigDataIntoJsonString(cartButtonStyleData);
        String markButtonStyle = parsePayPalButtonConfigDataIntoJsonString(markButtonStyleData);

        switch (key) {
            case PAYPAL_MINICART_BUTTON_CONFIG:
                setProperty(key, miniCartStyle);
                break;
            case PAYPAL_MARK_BUTTON_CONFIG:
                setProperty(key, markButtonStyle);
                break;
            case PAYPAL_CART_BUTTON_CONFIG:
                setProperty(key, cartButtonStyle);
        }
    }

    public void setDefaultGooglePayButtonConfigs(final String key) {
        GooglePayButtonStyleData miniCartButtonStyleData = createGooglePayButtonStyleData("cart", "white", "fill", "buy");
        GooglePayButtonStyleData cartButtonStyleData = createGooglePayButtonStyleData("minicart", "white", "fill", "buy");
        GooglePayButtonStyleData markButtonStyleData = createGooglePayButtonStyleData("billing", "white", "fill", "buy");

        String miniCartStyle = parseGooglePayButtonConfigDataIntoJsonString(miniCartButtonStyleData);
        String cartButtonStyle = parseGooglePayButtonConfigDataIntoJsonString(cartButtonStyleData);
        String markButtonStyle = parseGooglePayButtonConfigDataIntoJsonString(markButtonStyleData);

        switch (key) {
            case GOOGLE_PAY_MINI_CART_BUTTON_CONFIG:
                setProperty(key, miniCartStyle);
                break;
            case GOOGLE_PAY_MARK_BUTTON_CONFIG:
                setProperty(key, markButtonStyle);
                break;
            case GOOGLE_PAY_CART_BUTTON_CONFIG:
                setProperty(key, cartButtonStyle);
        }
    }

    public void setDefaultApplePayButtonConfigs(final String key) {
        ApplePayButtonStyleData miniCartButtonStyleData = createApplePayButtonStyleData("minicart", "black", "buy", "squareCorners");
        ApplePayButtonStyleData cartButtonStyleData = createApplePayButtonStyleData("cart", "black", "buy", "squareCorners");
        ApplePayButtonStyleData markButtonStyleData = createApplePayButtonStyleData("billing", "black", "buy", "squareCorners");

        String miniCartStyle = parseApplePayButtonConfigDataIntoJsonString(miniCartButtonStyleData);
        String cartButtonStyle = parseApplePayButtonConfigDataIntoJsonString(cartButtonStyleData);
        String markButtonStyle = parseApplePayButtonConfigDataIntoJsonString(markButtonStyleData);

        switch (key) {
            case APPLE_PAY_MINI_CART_BUTTON_CONFIG:
                setProperty(key, miniCartStyle);
                break;
            case APPLE_PAY_MARK_BUTTON_CONFIG:
                setProperty(key, markButtonStyle);
                break;
            case APPLE_PAY_CART_BUTTON_CONFIG:
                setProperty(key, cartButtonStyle);
        }
    }

    private PayPalButtonStyleData createPayPalButtonStyleData(String location, String color, String shape, String label, String layout,
                                                              boolean tagline, int height) {
        PayPalButtonStyleData buttonStyleData = new PayPalButtonStyleData();
        buttonStyleData.setLocation(location);
        buttonStyleData.setColor(color);
        buttonStyleData.setLayout(layout);
        buttonStyleData.setTagline(tagline);
        buttonStyleData.setHeight(height);
        buttonStyleData.setLabel(label);
        buttonStyleData.setShape(shape);
        return buttonStyleData;
    }

    private GooglePayButtonStyleData createGooglePayButtonStyleData(String location, String color, String size, String type) {
        GooglePayButtonStyleData buttonStyleData = new GooglePayButtonStyleData();
        buttonStyleData.setLocation(location);
        buttonStyleData.setSize(size);
        buttonStyleData.setType(type);
        return buttonStyleData;
    }

    private ApplePayButtonStyleData createApplePayButtonStyleData(String location, String color, String type, String shape) {
        ApplePayButtonStyleData buttonStyleData = new ApplePayButtonStyleData();
        buttonStyleData.setLocation(location);
        buttonStyleData.setColor(color);
        buttonStyleData.setType(type);
        buttonStyleData.setShape(shape);
        return buttonStyleData;
    }

    public void handlePayPalStyleDataUpdate(HttpServletRequest request, final String buttonConfig) {
        PayPalButtonStyleData buttonStyleData = createPayPalButtonStyleData(request.getParameter("location"), request.getParameter("color"),
                request.getParameter("shape"), request.getParameter("label"), request.getParameter("layout"),
                Boolean.parseBoolean(request.getParameter("tagline")), Integer.parseInt(request.getParameter("height")));
        String styleDataJson = parsePayPalButtonConfigDataIntoJsonString(buttonStyleData);
        setProperty(buttonConfig, styleDataJson);
    }

    public void handleGooglePayStyleDataUpdate(HttpServletRequest request, final String buttonConfig) {
        GooglePayButtonStyleData buttonStyleData = createGooglePayButtonStyleData(request.getParameter("location"), request.getParameter("color"),
                request.getParameter("size"), request.getParameter("type"));
        String styleDataJson = parseGooglePayButtonConfigDataIntoJsonString(buttonStyleData);
        setProperty(buttonConfig, styleDataJson);
    }

    public void handleApplePayStyleDataUpdate(HttpServletRequest request, final String buttonConfig) {
        ApplePayButtonStyleData buttonStyleData = createApplePayButtonStyleData(request.getParameter("location"), request.getParameter("color"),
                request.getParameter("type"), request.getParameter("shape"));
        String styleDataJson = parseApplePayButtonConfigDataIntoJsonString(buttonStyleData);
        setProperty(buttonConfig, styleDataJson);
    }

    public String parsePayPalButtonConfigDataIntoJsonString(PayPalButtonStyleData buttonConfig) {
        String buttonStyleConfig = "";

        try {
            buttonStyleConfig = getMapper().writeValueAsString(buttonConfig);
        } catch (JsonProcessingException e) {
            LOG.error("Error during parsing PayPalButtonConfigData into JSON\n" + e.getMessage(), e);
            e.printStackTrace();
        }

        return buttonStyleConfig;
    }

    public String parseGooglePayButtonConfigDataIntoJsonString(GooglePayButtonStyleData buttonConfig) {
        String buttonStyleConfig = "";

        try {
            buttonStyleConfig = getMapper().writeValueAsString(buttonConfig);
        } catch (JsonProcessingException e) {
            LOG.error("Error during parsing GooglePayButtonConfigData into JSON\n" + e.getMessage(), e);
            e.printStackTrace();
        }

        return buttonStyleConfig;
    }

    public String parseApplePayButtonConfigDataIntoJsonString(ApplePayButtonStyleData buttonConfig) {
        String buttonStyleConfig = "";

        try {
            buttonStyleConfig = getMapper().writeValueAsString(buttonConfig);
        } catch (JsonProcessingException e) {
            LOG.error("Error during parsing ApplePayButtonConfigData into JSON\n" + e.getMessage(), e);
            e.printStackTrace();
        }

        return buttonStyleConfig;
    }

    public PayPalButtonStyleData parsingJsonStringIntoPayPalButtonConfigData(String buttonKey) {
        loadPropertiesFromFile();
        PayPalButtonStyleData buttonConfig = null;
        try {
            buttonConfig = getMapper().readValue(prop.getProperty(buttonKey), PayPalButtonStyleData.class);

        } catch (JsonProcessingException e) {
            buttonConfig = new PayPalButtonStyleData();
            LOG.error("Error during parsing JSON date into PayPalButtonConfigData\n" + e.getMessage(), e);
            e.printStackTrace();
        }

        return buttonConfig;
    }

    public PayPalButtonStyleData getPayPalButtonStyleByPageType(final BraintreePageType page) {
        final String buttonKey;
        switch (page) {
            case CART:
                buttonKey = PAYPAL_CART_BUTTON_CONFIG;
                break;
            case MARK:
                buttonKey = PAYPAL_MARK_BUTTON_CONFIG;
                break;
            case MINICART:
                buttonKey = PAYPAL_MINICART_BUTTON_CONFIG;
                break;
            default:
                buttonKey = "";
        }
        return parsingJsonStringIntoPayPalButtonConfigData(buttonKey);
    }

    public GooglePayButtonStyleData getGooglePayButtonStyleByPageType(final BraintreePageType page) {
        final String buttonKey;
        switch (page) {
            case CART:
                buttonKey = GOOGLE_PAY_CART_BUTTON_CONFIG;
                break;
            case MARK:
                buttonKey = GOOGLE_PAY_MARK_BUTTON_CONFIG;
                break;
            case MINICART:
                buttonKey = GOOGLE_PAY_MINI_CART_BUTTON_CONFIG;
                break;
            default:
                buttonKey = "";
        }
        return parsingJsonStringIntoGooglePayButtonConfigData(buttonKey);
    }

    public ApplePayButtonStyleData getApplePayButtonStyleByPageType(final BraintreePageType page) {
        final String buttonKey;
        switch (page) {
            case CART:
                buttonKey = APPLE_PAY_CART_BUTTON_CONFIG;
                break;
            case MARK:
                buttonKey = APPLE_PAY_MARK_BUTTON_CONFIG;
                break;
            case MINICART:
                buttonKey = APPLE_PAY_MINI_CART_BUTTON_CONFIG;
                break;
            default:
                buttonKey = "";
        }
        return parsingJsonStringIntoApplePayButtonConfigData(buttonKey);
    }

    public GooglePayButtonStyleData parsingJsonStringIntoGooglePayButtonConfigData(String buttonKey) {
        loadPropertiesFromFile();
        GooglePayButtonStyleData buttonConfig = null;
        try {
            buttonConfig = getMapper().readValue(prop.getProperty(buttonKey), GooglePayButtonStyleData.class);

        } catch (JsonProcessingException e) {
            buttonConfig = new GooglePayButtonStyleData();
            LOG.error("Error during parsing JSON date into GooglePayButtonConfigData\n" + e.getMessage(), e);
            e.printStackTrace();
        }

        return buttonConfig;
    }

    public ApplePayButtonStyleData parsingJsonStringIntoApplePayButtonConfigData(String buttonKey) {
        loadPropertiesFromFile();
        ApplePayButtonStyleData buttonConfig = null;
        try {
            buttonConfig = getMapper().readValue(prop.getProperty(buttonKey), ApplePayButtonStyleData.class);

        } catch (JsonProcessingException e) {
            buttonConfig = new ApplePayButtonStyleData();
            LOG.error("Error during parsing JSON date into ApplePayButtonConfigData\n" + e.getMessage(), e);
            e.printStackTrace();
        }

        return buttonConfig;
    }

    public void storeInPropertiesFile() {
        try (OutputStream out = new FileOutputStream(propFile)) {
            prop.store(out, "Save properties in file");
        } catch (Exception e) {
            LOG.error(CART_BUTTON_CONFIG_FILE + " is not saved,  error: " + e.getMessage(), e);
        }
    }

    private void loadPropertiesFromFile() {
        try (InputStream input = new FileInputStream(propFile)) {
            prop.load(input);
        } catch (Exception e) {
            LOG.error("Error during reading " + CART_BUTTON_CONFIG_FILE + ": " + e.getMessage(), e);
        }
    }

    private static List<Integer> initHeightDefaultValues() {
        List<Integer> heightValues = new ArrayList<>();
        for (int i = 25; i <= 55; i++) {
            heightValues.add(i);
        }
        return heightValues;
    }

    public ObjectMapper getMapper() {
        return mapper;
    }

    public void setMapper(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    public List<String> getSmartButtonLocation() {
        return smartButtonLocation;
    }

    public List<String> getSmartButtonColor() {
        return smartButtonColor;
    }

    public List<String> getSmartButtonShape() {
        return smartButtonShape;
    }

    public List<Integer> getSmartButtonHeight() {
        return smartButtonHeight;
    }

    public List<String> getSmartButtonLabel() {
        return smartButtonLabel;
    }

    public List<String> getSmartButtonLayout() {
        return smartButtonLayout;
    }

    public List<String> getGooglePaySmartButtonSize() {
        return smartGooglePayButtonSize;
    }

    public List<String> getGooglePaySmartButtonType() {
        return smartGooglePayButtonType;
    }

    public List<String> getApplePaySmartButtonColor() {
        return smartApplePayButtonColor;
    }

    public List<String> getApplePaySmartButtonType() {
        return smartApplePayButtonType;
    }

    public List<String> getApplePaySmartButtonShape() {
        return smartApplePayButtonShape;
    }

}
