package com.braintree.actions.order;


import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.util.BrainTreeUtils;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.orderprocessing.model.OrderProcessModel;
import de.hybris.platform.processengine.action.AbstractSimpleDecisionAction;
import de.hybris.platform.task.RetryLaterException;
import org.apache.log4j.Logger;

public class BrainTreeGoToSubmitForSettlementAction extends AbstractSimpleDecisionAction<OrderProcessModel> {

    private static final Logger LOG = Logger.getLogger(BrainTreeGoToSubmitForSettlementAction.class);

    private BrainTreeConfigService brainTreeConfigService;

    @Override
    public Transition executeAction(OrderProcessModel orderProcessModel) throws RetryLaterException, Exception {
        final OrderModel order = orderProcessModel.getOrder();

        if (order.getPaymentInfo() != null && order.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            if (canCreateCaptureTransactionForOrder(order)) {
                LOG.info("[BRAINTREE ORDER PROCESS] Order process (orderProcess.code " + orderProcessModel.getCode()
                    + ") will be captured. order.code: " + order.getCode());
                return Transition.OK;
            }
            LOG.info("[BRAINTREE ORDER PROCESS] Order process (code " + orderProcessModel.getCode()
                + ") is waiting for submitForSettlement event.");
            return Transition.NOK;
        }

        LOG.warn(
            "[BRAINTREE ORDER PROCESS] PayPal intent not found for this order. Order process is stopped and waiting for submitForSettlement event, order.code: "
                + order.getCode());
        return Transition.OK;
    }

    private boolean canCreateCaptureTransactionForOrder(OrderModel order) {
        final String paymentProvider = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider();
        final String intent = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPayPalIntent();
        if (BrainTreeUtils.isApplePayPayment(paymentProvider)) {
            return getBrainTreeConfigService().isSubmitForSettlement();
        }
        if (BrainTreeUtils.isGooglePayment(paymentProvider)) {
            return getBrainTreeConfigService().isSubmitForSettlement();
        }
        if (BrainTreeUtils.isSrcPayment(paymentProvider)) {
            return getBrainTreeConfigService().isSubmitForSettlement();
        }
        if (BrainTreeUtils.isCreditCardPayment(paymentProvider)) {
            return getBrainTreeConfigService().isSubmitForSettlement();
        }
        if (BrainTreeUtils.isVenmoPayment(paymentProvider)) {
            return getBrainTreeConfigService().isSubmitForSettlement();
        }
        if (BrainTreeUtils.isPayPalPayment(paymentProvider)) {
            LOG.warn("intent: " + intent);
            return BrainTreeUtils.isIntentSale(intent) || BrainTreeUtils.isIntentOrder(intent);
        }
        if (BrainTreeUtils.isLPMPayment(paymentProvider)) {
            return BrainTreeUtils.isIntentSale(intent);
        }
        if (BrainTreeUtils.isUsBankAccountPayment(paymentProvider)) {
            return true;
        }
        LOG.warn("[BRAINTREE ORDER PROCESS] Unknown payment provider: " + paymentProvider);
        return false;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }
}

