/**
 *
 */
package com.braintree.facade.impl;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.times;

import com.braintree.command.result.BrainTreeCreatePaymentMethodResult;
import com.braintree.command.result.BrainTreePaymentMethodNonceResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants.PayPalIntent;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.ExpirationStatus;
import com.braintree.exceptions.BrainTreeCardVerifyException;
import com.braintree.exceptions.BrainTreeDeliveryAddressNotFoundException;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.hybris.data.BrainTreePaymentInfoData;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.hybris.data.BraintreePaymentMethodNonceData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.payment.dto.BraintreeInfo;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.payment.local.methods.service.BraintreeLocalPaymentMethodsService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.WebhookNotification;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.customer.CustomerEmailResolutionService;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.commerceservices.order.CommerceCartService;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import javax.servlet.http.HttpServletRequest;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Spy;
import org.mockito.MockitoAnnotations;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


@UnitTest
public class DefaultBrainTreePaymentFacadeTest {

    private static final String ORDER_CODE = "123456";

    private static final String PAYMENT_ID = "testId";

    private static final String BRAINTREE_CUSTOMER_ID = "testId";
    private static final String NONCE = "nonce";
    private static final String TOKEN = "token";
    private static final String DEVICE_DATA = "deviceData";
    private static final String UID = "12341234";
    private static final String ADDRESS_ID = "addressId";
    private static final String EMAIL = "test@email.com";

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private CheckoutCustomerStrategy checkoutCustomerStrategy;

    @Mock
    private CartService cartService;

    @Mock
    private BaseStoreService baseStoreService;

    @Mock
    private BrainTreeCustomerAccountService customerAccountService;

    @Mock
    private BraintreeLocalPaymentMethodsService braintreeLocalPaymentMethodsService;

    @Mock
    private Converter<BrainTreePaymentInfoModel, BrainTreePaymentInfoData> brainTreePaymentInfoDataConverter;

    @Mock
    private Converter<OrderModel, OrderData> orderConverter;

    @Mock
    private Converter<AddressData, AddressModel> addressReverseConverter;

    @Mock
    private Converter<AddressModel, AddressData> addressConverter;

    @Mock
    private Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter;

    @Mock
    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private ModelService modelService;

    @Mock
    private CustomerEmailResolutionService customerEmailResolutionService;

    @Mock
    private BrainTreeUserFacade brainTreeUserFacade;

    @Mock
    private UserService userService;

    @Mock
    private BraintreePaymentInfoService braintreePaymentInfoService;

    @Mock
    private CommerceCartService commerceCartService;

    @Spy
    @InjectMocks
    private DefaultBrainTreePaymentFacade unit;

    private BrainTreeSubscriptionInfoData subscriptionInfo;
    private CustomerModel customerModel;
    private CartModel cartModel;
    private AddressModel addressModel;
    private AddressData addressData;
    private BrainTreePaymentInfoModel paymentInfoModel;
    private BraintreeInfo braintreeInfo;
    private BrainTreeCreatePaymentMethodResult result;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void createLocalPaymentMethodSubscriptionTest() {
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final CartModel cart = mock(CartModel.class);
        final CustomerModel customer = mock(CustomerModel.class);

        when(cartService.getSessionCart()).thenReturn(cart);
        when(modelService.create(BrainTreePaymentInfoModel.class)).thenReturn(paymentInfo);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customer);
        when(customer.getUid()).thenReturn(BRAINTREE_CUSTOMER_ID);

        unit.createLocalPaymentMethodSubscription(PAYMENT_ID);

        verify(modelService).saveAll(paymentInfo, cart);
    }

    @Test
    public void updateLocalPaymentMethodSubscriptionForGuestTest() throws BrainTreeDeliveryAddressNotFoundException {
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final AddressModel address = mock(AddressModel.class);
        final CartModel cart = mock(CartModel.class);
        final UserModel user = mock(UserModel.class);
        final CCPaymentInfoData ccPaymentInfoData = mock(CCPaymentInfoData.class);

        when(cartService.getSessionCart()).thenReturn(cart);
        when(cart.getPaymentInfo()).thenReturn(paymentInfo);
        when(cart.getDeliveryAddress()).thenReturn(address);
        when(modelService.clone(address)).thenReturn(address);
        when(cart.getUser()).thenReturn(user);
        when(user.getUid()).thenReturn(UID);
        when(user.getName()).thenReturn(CustomerType.GUEST.getCode());
        when(brainTreePaymentInfoConverter.convert(paymentInfo)).thenReturn(ccPaymentInfoData);

        CCPaymentInfoData ccPaymentInfo = unit.updateLocalPaymentMethodSubscription(NONCE, DEVICE_DATA, null);

        assertEquals(ccPaymentInfoData, ccPaymentInfo);
        verify(modelService).saveAll(paymentInfo, address);
    }

    @Test
    public void updateLocalPaymentMethodSubscriptionForRegisteredTest() throws BrainTreeDeliveryAddressNotFoundException {
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final AddressModel address = mock(AddressModel.class);
        final CartModel cart = mock(CartModel.class);
        final UserModel user = mock(UserModel.class);
        final CCPaymentInfoData ccPaymentInfoData = mock(CCPaymentInfoData.class);

        when(cartService.getSessionCart()).thenReturn(cart);
        when(cart.getPaymentInfo()).thenReturn(paymentInfo);
        when(cart.getDeliveryAddress()).thenReturn(address);
        when(modelService.clone(address)).thenReturn(address);
        when(cart.getUser()).thenReturn(user);
        when(user.getUid()).thenReturn(UID);
        when(user.getName()).thenReturn(CustomerType.REGISTERED.getCode());
        when(brainTreePaymentInfoConverter.convert(paymentInfo)).thenReturn(ccPaymentInfoData);

        CCPaymentInfoData ccPaymentInfo = unit.updateLocalPaymentMethodSubscription(NONCE, DEVICE_DATA, null);

        assertEquals(ccPaymentInfoData, ccPaymentInfo);
        verify(modelService).saveAll(paymentInfo, address);
    }

    @Test(expected = BrainTreeDeliveryAddressNotFoundException.class)
    public void updateLocalPaymentMethodSubscriptionMustThrowExceptionTest() throws BrainTreeDeliveryAddressNotFoundException {
        final CartModel cart = mock(CartModel.class);

        when(cartService.getSessionCart()).thenReturn(cart);

        unit.updateLocalPaymentMethodSubscription(NONCE, DEVICE_DATA, null);
    }

    @Test
    public void shouldGetBrainTreePaymentInfoData() {
        final CartModel cartModel1 = mock(CartModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel1 = mock(BrainTreePaymentInfoModel.class);
        final BrainTreePaymentInfoData brainTreePaymentInfoDataExpected = mock(BrainTreePaymentInfoData.class);

        when(cartService.getSessionCart()).thenReturn(cartModel1);
        when(cartModel1.getPaymentInfo()).thenReturn(paymentInfoModel1);
        when(brainTreePaymentInfoDataConverter.convert(paymentInfoModel1)).thenReturn(brainTreePaymentInfoDataExpected);
        unit.setBrainTreePaymentInfoDataConverter(brainTreePaymentInfoDataConverter);

        final BrainTreePaymentInfoData brainTreePaymentInfoDataActual = unit
            .getBrainTreePaymentInfoData();

        assertEquals(brainTreePaymentInfoDataExpected, brainTreePaymentInfoDataActual);

    }

    @Test
    public void shouldGetExpirationStatusForCustomer() {
        BrainTreePaymentInfoModel fCard = mock(BrainTreePaymentInfoModel.class);
        BrainTreePaymentInfoModel sCard = mock(BrainTreePaymentInfoModel.class);
        List<BrainTreePaymentInfoModel> brainTreePaymentInfoModelList = List.of(fCard, sCard);
        customerModel = mock(CustomerModel.class);

        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getBraintreeCustomerId()).thenReturn(BRAINTREE_CUSTOMER_ID);
        when(braintreePaymentInfoService.getAllSavedCreditCardByCustomer(BRAINTREE_CUSTOMER_ID)).thenReturn(brainTreePaymentInfoModelList);

        // case when one card expired
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRED);
        when(sCard.getExpirationStatus()).thenReturn(ExpirationStatus.NOT_EXPIRED);

        assertEquals(ExpirationStatus.EXPIRED.getCode(), unit.getExpirationStatusForCustomer());

        // case when one card expired soon
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRE_SOON);
        when(sCard.getExpirationStatus()).thenReturn(ExpirationStatus.NOT_EXPIRED);
        when(customerModel.isCustomerHasNewExpireSoonCard()).thenReturn(true);

        assertEquals(ExpirationStatus.EXPIRE_SOON.getCode(), unit.getExpirationStatusForCustomer());
        verify(modelService).save(customerModel);

        // case when all cards are not expired
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.NOT_EXPIRED);
        when(sCard.getExpirationStatus()).thenReturn(ExpirationStatus.NOT_EXPIRED);

        assertEquals(ExpirationStatus.NOT_EXPIRED.getCode(), unit.getExpirationStatusForCustomer());

        // case when one card expired and one expired soon
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRED);
        when(sCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRE_SOON);

        assertEquals(ExpirationStatus.EXPIRED.getCode(), unit.getExpirationStatusForCustomer());
    }

    @Test
    public void shouldGetBrainTreePaymentInfoDataByOrderCode() {
        final BrainTreePaymentInfoData brainTreePaymentInfoDataExpected = mock(BrainTreePaymentInfoData.class);
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final OrderModel order = mock(OrderModel.class);
        final BaseStoreModel baseStoreModel = mock(BaseStoreModel.class);

        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(customerAccountService.getOrderForCode(ORDER_CODE, baseStoreModel)).thenReturn(order);
        when(order.getPaymentInfo()).thenReturn(paymentInfo);
        when(brainTreePaymentInfoDataConverter.convert(paymentInfo)).thenReturn(brainTreePaymentInfoDataExpected);
        unit.setBrainTreePaymentInfoDataConverter(brainTreePaymentInfoDataConverter);

        final BrainTreePaymentInfoData brainTreePaymentInfoDataActual = unit
            .getBrainTreePaymentInfoData(ORDER_CODE);

        assertEquals(brainTreePaymentInfoDataExpected, brainTreePaymentInfoDataActual);
    }

    @Test
    public void shouldGetWebhookNotification() {
        final WebhookNotification webhookNotification = mock(WebhookNotification.class);
        when(brainTreePaymentService.getWebhookNotification(any())).thenReturn(webhookNotification);

        unit.getWebhookNotification(mock(HttpServletRequest.class));

        verify(brainTreePaymentService).getWebhookNotification(any());
    }

    @Test
    public void shouldGetOrderByPaymentId() {
        final OrderModel orderModel = mock(OrderModel.class);
        when(braintreeLocalPaymentMethodsService.getOrderByPaymentId(PAYMENT_ID)).thenReturn(orderModel);

        unit.getOrderByPaymentId(PAYMENT_ID);

        verify(braintreeLocalPaymentMethodsService).getOrderByPaymentId(PAYMENT_ID);
    }

    private void defaultPreparingBeforeCompleteCreateSubscription() {
        subscriptionInfo = mock(BrainTreeSubscriptionInfoData.class);
        customerModel = mock(CustomerModel.class);
        cartModel = mock(CartModel.class);
        addressModel = mock(AddressModel.class);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);

        addressData = mock(AddressData.class);
        when(subscriptionInfo.getAddressData()).thenReturn(addressData);

        paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        when(brainTreeTransactionService.createPaymentMethodInfo(any(), any(), any())).thenReturn(paymentInfoModel);

        braintreeInfo = mock(BraintreeInfo.class);
        when(brainTreeSubscriptionInfoConverter.convert(subscriptionInfo)).thenReturn(braintreeInfo);

        result = mock(BrainTreeCreatePaymentMethodResult.class);
        when(result.isSuccess()).thenReturn(true);
        when(brainTreePaymentService.createPaymentMethodOnBraintreeOrGetExisting(customerModel, addressModel, subscriptionInfo))
            .thenReturn(result);


    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodIfOrderIntentAndCreditEnabledAndChangePaymentButtonEnabled() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.TRUE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.PAYPALACCOUNT.getCode())).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreeTransactionService).createPaymentMethodInfo(addressModel, customerModel, subscriptionInfo);
        verify(modelService).save(cartModel);
    }

    @Test
    public void completeCreateSubscriptionShouldNotCreatePaymentMethodIfOrderSaleAndCreditEnabledAndChangePaymentButtonEnabled() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.TRUE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.PAYPALACCOUNT.getCode())).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreePaymentService, never()).createPaymentMethodOnBraintreeOrGetExisting(customerModel, addressModel, subscriptionInfo);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodIfOrderSaleAndCreditNotEnabledAndChangePaymentButtonNotEnabled() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(false);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.TRUE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.PAYPALACCOUNT.getCode())).thenReturn(true);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreeTransactionService).createPaymentMethodInfo(addressModel, customerModel, subscriptionInfo);
        verify(modelService).save(cartModel);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodForGooglePayWhenVaultTrue() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.TRUE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.ANDROIDPAYCARD.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.ANDROIDPAYCARD.getCode())).thenReturn(true);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreeTransactionService).createPaymentMethodInfo(addressModel, customerModel, subscriptionInfo);
        verify(modelService).save(cartModel);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodForVenmoWhenVaultFalse() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.FALSE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.VENMOACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.VENMOACCOUNT.getCode())).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreeTransactionService).createPaymentMethodInfo(addressModel, customerModel, subscriptionInfo);
        verify(modelService).save(cartModel);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodForACHTest() {
        final BrainTreeCreatePaymentMethodResult createPaymentMethod = mock(BrainTreeCreatePaymentMethodResult.class);

        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.FALSE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.USBANKACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.USBANKACCOUNT.getCode())).thenReturn(false);
        when(brainTreePaymentService.createPaymentMethodOnBraintreeOrGetExisting(customerModel, addressModel, subscriptionInfo)).thenReturn(createPaymentMethod);
        when(createPaymentMethod.isSuccess()).thenReturn(true);
        when(createPaymentMethod.getVerified()).thenReturn(true);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreeTransactionService).createPaymentMethodInfo(addressModel, customerModel, subscriptionInfo);
        verify(modelService).save(cartModel);
    }

    @Test(expected = BrainTreeCardVerifyException.class)
    public void completeCreateSubscriptionForACHShouldThrowBrainTreeCardVerifyExceptionTest() {
        final BrainTreeCreatePaymentMethodResult createPaymentMethod = mock(BrainTreeCreatePaymentMethodResult.class);

        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.FALSE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.USBANKACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.USBANKACCOUNT.getCode())).thenReturn(false);

        when(brainTreePaymentService.createPaymentMethodOnBraintreeOrGetExisting(customerModel, addressModel, subscriptionInfo)).thenReturn(createPaymentMethod);
        when(createPaymentMethod.isSuccess()).thenReturn(true);
        when(createPaymentMethod.getPaymentMethodToken()).thenReturn(TOKEN);
        doNothing().when(brainTreeUserFacade).removeBTCCPaymentInfo(TOKEN);
        when(createPaymentMethod.getVerified()).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);
    }

    @Test(expected = AdapterException.class)
    public void completeCreateSubscriptionForACHShouldThrowAdapterExceptionTest() {
        final BrainTreeCreatePaymentMethodResult createPaymentMethod = mock(BrainTreeCreatePaymentMethodResult.class);

        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.isCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.FALSE);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.USBANKACCOUNT.getCode());
        when(subscriptionInfo.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.isVaultingForCurrentUser(BrainTreePaymentMethod.USBANKACCOUNT.getCode())).thenReturn(false);

        when(brainTreePaymentService.createPaymentMethodOnBraintreeOrGetExisting(customerModel, addressModel, subscriptionInfo)).thenReturn(createPaymentMethod);
        when(createPaymentMethod.isSuccess()).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);
    }

    @Test
    public void completeCreateSubscriptionForMyAccountTest() {
        final BrainTreeSubscriptionInfoData subscriptionInfo = mock(BrainTreeSubscriptionInfoData.class);
        final CustomerModel customer = mock(CustomerModel.class);
        final AddressData addressData1 = mock(AddressData.class);
        final AddressModel addressModel1 = mock(AddressModel.class);
        final BrainTreeCreatePaymentMethodResult createPaymentMethodResult = mock(BrainTreeCreatePaymentMethodResult.class);
        final BrainTreePaymentInfoModel brainTreePaymentInfo = mock(BrainTreePaymentInfoModel.class);

        when(subscriptionInfo.getAddressData()).thenReturn(addressData1);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel1);
        when(addressData1.getBrainTreeAddressId()).thenReturn(ADDRESS_ID);
        when(addressData1.getEmail()).thenReturn(EMAIL);
        when(customer.getType()).thenReturn(CustomerType.REGISTERED);
        when(brainTreePaymentService.createPaymentMethodOnBraintreeOrGetExisting(customer, addressModel1, subscriptionInfo)).thenReturn(createPaymentMethodResult);
        when(createPaymentMethodResult.isSuccess()).thenReturn(true);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(brainTreeTransactionService.createPaymentMethodInfo(addressModel1, customer, subscriptionInfo)).thenReturn(brainTreePaymentInfo);

        unit.completeCreateSubscriptionForMyAccount(subscriptionInfo, customer);

        verify(braintreePaymentInfoService).checkOnDuplicatePaymentMethod(brainTreePaymentInfo, customer, addressModel1);
    }

    @Test(expected = BrainTreeCardVerifyException.class)
    public void completeCreateSubscriptionForMyAccountForACHShouldThrowBrainTreeCardVerifyExceptionTest() {
        final BrainTreeSubscriptionInfoData subscriptionInfo = mock(BrainTreeSubscriptionInfoData.class);
        final CustomerModel customer = mock(CustomerModel.class);
        final AddressData addressData1 = mock(AddressData.class);
        final AddressModel addressModel1 = mock(AddressModel.class);
        final BrainTreeCreatePaymentMethodResult braintreePaymentMethod = mock(BrainTreeCreatePaymentMethodResult.class);

        when(subscriptionInfo.getAddressData()).thenReturn(addressData1);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel1);
        when(addressData1.getBrainTreeAddressId()).thenReturn(ADDRESS_ID);
        when(addressData1.getEmail()).thenReturn(EMAIL);
        when(customer.getType()).thenReturn(CustomerType.REGISTERED);
        when(brainTreePaymentService.createPaymentMethodOnBraintreeOrGetExisting(customer, addressModel1, subscriptionInfo)).thenReturn(braintreePaymentMethod);
        when(braintreePaymentMethod.isSuccess()).thenReturn(true);
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.USBANKACCOUNT.getCode());
        when(braintreePaymentMethod.getVerified()).thenReturn(false);

        unit.completeCreateSubscriptionForMyAccount(subscriptionInfo, customer);
    }

    @Test
    public void completeCreateSubscriptionForMyAccountWithOneParameterTest() {
        final BrainTreeSubscriptionInfoData subscriptionData = mock(BrainTreeSubscriptionInfoData.class);
        final CustomerModel customer = mock(CustomerModel.class);
        final CCPaymentInfoData ccPaymentInfoData = mock(CCPaymentInfoData.class);

        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customer);
        doReturn(ccPaymentInfoData).when(unit).completeCreateSubscriptionForMyAccount(subscriptionData, customer);

        unit.completeCreateSubscriptionForMyAccount(subscriptionData);

        verify(checkoutCustomerStrategy).getCurrentUserForCheckout();
        verify(unit).completeCreateSubscriptionForMyAccount(subscriptionData, customer);
    }

    @Test
    public void createPaymentInfoForLoginTest() {
        final BrainTreeSubscriptionInfoData subscriptionData = mock(BrainTreeSubscriptionInfoData.class);
        final CustomerModel customer = mock(CustomerModel.class);
        final CCPaymentInfoData ccPaymentInfoData = mock(CCPaymentInfoData.class);

        when(userService.getUserForUID(EMAIL)).thenReturn(customer);
        doReturn(ccPaymentInfoData).when(unit).completeCreateSubscriptionForMyAccount(subscriptionData, customer);

        unit.createPaymentInfoForLogin(subscriptionData, EMAIL);

        verify(userService).getUserForUID(EMAIL);
        verify(unit).completeCreateSubscriptionForMyAccount(subscriptionData, customer);
    }

    @Test
    public void setPaymentMethodNonceTest() {
        final CCPaymentInfoData ccPaymentInfoData = new CCPaymentInfoData();
        final BrainTreePaymentMethodNonceResult nonceResult = mock(BrainTreePaymentMethodNonceResult.class);

        ccPaymentInfoData.setPaymentMethodToken(TOKEN);
        when(brainTreePaymentService.createPaymentMethodNonce(TOKEN)).thenReturn(nonceResult);
        when(nonceResult.getNonce()).thenReturn(NONCE);

        unit.setPaymentMethodNonce(ccPaymentInfoData);

        assertEquals(NONCE, ccPaymentInfoData.getPaymentMethodNonce());
    }

    @Test
    public void createPaymentMethodNonceTest() {
        final BrainTreePaymentMethodNonceResult nonceResult = mock(BrainTreePaymentMethodNonceResult.class);

        when(brainTreePaymentService.createPaymentMethodNonce(TOKEN)).thenReturn(nonceResult);
        when(nonceResult.getNonce()).thenReturn(NONCE);
        when(nonceResult.getBin()).thenReturn(UID);
        when(nonceResult.isShouldPerform3dSecure()).thenReturn(true);

        BraintreePaymentMethodNonceData paymentMethodNonceData = unit.createPaymentMethodNonce(TOKEN);

        assertEquals(NONCE, paymentMethodNonceData.getNonce());
        assertEquals(UID, paymentMethodNonceData.getDetails().getBin());
        assertEquals(true, paymentMethodNonceData.isShouldPerform3dSecure());
    }

    @Test
    public void updateLocalPaymentMethodSubscriptionTest() {
        final CartModel cart = mock(CartModel.class);
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final AddressModel addressForClone = mock(AddressModel.class);
        final AddressModel billingAddress = mock(AddressModel.class);

        when(cart.getPaymentInfo()).thenReturn(paymentInfo);
        when(cart.getDeliveryAddress()).thenReturn(addressForClone);
        when(modelService.clone(addressForClone)).thenReturn(billingAddress);

        unit.updateLocalPaymentMethodSubscription(NONCE, cart);

        verify(modelService).saveAll(paymentInfo, cart, billingAddress);
    }

    @Test
    public void setDeviceDataTest() {
        final BrainTreePaymentInfoModel paymentInfo = new BrainTreePaymentInfoModel();

        unit.setDeviceData(DEVICE_DATA, paymentInfo);

        verify(modelService).save(paymentInfo);
        assertEquals(DEVICE_DATA, paymentInfo.getDeviceData());
    }

    @Test
    public void resolveBillingAddressTestShouldAddressBillingDataEqualsNull() throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        final AddressData addressBillingData = null;
        final CartModel cart = mock(CartModel.class);
        final CustomerModel customer = mock(CustomerModel.class);
        final BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData = mock(BrainTreeSubscriptionInfoData.class);
        final AddressModel deliveryAddress = mock(AddressModel.class);

        when(modelService.create(AddressModel.class)).thenReturn(new AddressModel());
        when(cart.getDeliveryAddress()).thenReturn(deliveryAddress);
        when(deliveryAddress.getBrainTreeAddressId()).thenReturn(ADDRESS_ID);
        when(brainTreeSubscriptionInfoData.getEmail()).thenReturn(EMAIL);

        Method resolveBillingAddressMethod = DefaultBrainTreePaymentFacade.class.getDeclaredMethod("resolveBillingAddress",
                AddressData.class, AbstractOrderModel.class, CustomerModel.class, BrainTreeSubscriptionInfoData.class);
        resolveBillingAddressMethod.setAccessible(true);
        AddressModel result = (AddressModel) resolveBillingAddressMethod.invoke(unit, addressBillingData, cart, customer, brainTreeSubscriptionInfoData);

        assertEquals(EMAIL, result.getEmail());
    }

    @Test
    public void getBrainTreePaymentInfoDataByCartTest() {
        final CustomerModel customer = mock(CustomerModel.class);
        final CartModel cart = mock(CartModel.class);
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final BrainTreePaymentInfoData brainTreePaymentInfoDataExpected = mock(BrainTreePaymentInfoData.class);

        when(userService.getCurrentUser()).thenReturn(customer);
        when(commerceCartService.getCartForCodeAndUser(UID, customer)).thenReturn(cart);
        when(cart.getPaymentInfo()).thenReturn(paymentInfo);
        when(brainTreePaymentInfoDataConverter.convert(paymentInfo)).thenReturn(brainTreePaymentInfoDataExpected);
        unit.setBrainTreePaymentInfoDataConverter(brainTreePaymentInfoDataConverter);

        BrainTreePaymentInfoData result = unit.getBrainTreePaymentInfoDataByCart(UID);

        assertEquals(brainTreePaymentInfoDataExpected, result);
    }

    @Test
    public void forceUnduplicateCartForReplenishmentTest() {
        final CustomerModel customer = mock(CustomerModel.class);
        final CartModel cart = mock(CartModel.class);
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);

        when(userService.getCurrentUser()).thenReturn(customer);
        when(commerceCartService.getCartForCodeAndUser(UID, customer)).thenReturn(cart);
        when(cart.getPaymentInfo()).thenReturn(paymentInfo);
        when(paymentInfo.getDuplicate()).thenReturn(true);
        when(paymentInfo.isSaved()).thenReturn(true);

        unit.forceUnduplicateCartForReplenishment(UID);

        verify(modelService).save(paymentInfo);
    }

    @Test
    public void getLocalPaymentMethodsTest() {
        when(braintreeLocalPaymentMethodsService.getAllLocalPaymentMethods()).thenReturn(new ArrayList<>());

        unit.getLocalPaymentMethods();

        verify(braintreeLocalPaymentMethodsService).getAllLocalPaymentMethods();
    }

    @Test
    public void getOrderByPaymentMethodNonceTest() {
        final OrderModel orderModel = mock(OrderModel.class);
        final OrderData orderData = mock(OrderData.class);

        when(braintreeLocalPaymentMethodsService.getOrderByPaymentMethodNonce(NONCE)).thenReturn(orderModel);
        when(orderConverter.convert(orderModel, orderData)).thenReturn(orderData);
        unit.setOrderConverter(orderConverter);

        unit.getOrderByPaymentMethodNonce(NONCE);

        verify(braintreeLocalPaymentMethodsService, times(2)).getOrderByPaymentMethodNonce(NONCE);
    }

    @Test
    public void getCartByPaymentIdTest() {
        final CartModel cart = mock(CartModel.class);

        when(braintreeLocalPaymentMethodsService.getCartByPaymentId(PAYMENT_ID)).thenReturn(cart);

        unit.getCartByPaymentId(PAYMENT_ID);

        verify(braintreeLocalPaymentMethodsService).getCartByPaymentId(PAYMENT_ID);
    }

    @Test
    public void getAvailablePaymentsTest() {
        final CCPaymentInfoData paymentInfo = mock(CCPaymentInfoData.class);
        List<CCPaymentInfoData> list = Arrays.asList(paymentInfo);

        when(paymentInfo.getPaymentMethodNonce()).thenReturn(NONCE);

        List<CCPaymentInfoData> result = unit.getAvailablePayments(list);

        assertEquals(paymentInfo, result.get(0));
    }
}
