package com.braintree.facade.impl;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.hybris.data.BrainTreeConnectWithPayPalAddressData;
import com.braintree.hybris.data.BrainTreeConnectWithPayPalUserData;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.service.BrainTreeConnectWithPayPalService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertFalse;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;

@UnitTest
public class DefaultBrainTreeConnectWithPayPalFacadeTest {

    private static final String ACCESS_TOKEN = "accessToken";
    private static final String AUTHORIZATION_TOKEN = "authorizationToken";
    private static final String PAYER_ID = "payerId";
    private static final String EMAIL = "email";
    private static final String US = "US";
    private static final String AL = "AL";
    private static final String ISO_CODE = "12345";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";

    @Mock
    private BrainTreeConnectWithPayPalService brainTreeConnectWithPayPalService;
    @Mock
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;
    @Mock
    private UserService userService;
    @Mock
    private ModelService modelService;
    @Mock
    private CommonI18NService commonI18NService;
    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Spy
    @InjectMocks
    private DefaultBrainTreeConnectWithPayPalFacade unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void getUserDataByAccessTokenTest() {
        final BrainTreeConnectWithPayPalUserData userData = mock(BrainTreeConnectWithPayPalUserData.class);

        when(brainTreeConnectWithPayPalService.getUserDataByAccessToken(ACCESS_TOKEN)).thenReturn(userData);

        unit.getUserDataByAccessToken(ACCESS_TOKEN);

        verify(brainTreeConnectWithPayPalService).getUserDataByAccessToken(ACCESS_TOKEN);
    }

    @Test
    public void exchangeAuthorizationCodeToAccessTokenTest() {
        when(brainTreeConnectWithPayPalService.exchangeAuthorizationCodeToAccessToken(AUTHORIZATION_TOKEN)).thenReturn(ACCESS_TOKEN);

        unit.exchangeAuthorizationCodeToAccessToken(AUTHORIZATION_TOKEN);

        verify(brainTreeConnectWithPayPalService).exchangeAuthorizationCodeToAccessToken(AUTHORIZATION_TOKEN);
    }

    @Test
    public void setAccessTokenForCustomerTest() {
        doNothing().when(brainTreeCustomerAccountService).setAccessTokenForCustomer(ACCESS_TOKEN, PAYER_ID);

        unit.setAccessTokenForCustomer(ACCESS_TOKEN, PAYER_ID);

        verify(unit).setAccessTokenForCustomer(ACCESS_TOKEN, PAYER_ID);
    }

    @Test
    public void setDefaultAccessTokenTest() {
        doNothing().when(brainTreeCustomerAccountService).setDefaultAccessToken(PAYER_ID);

        unit.setDefaultAccessToken(PAYER_ID);

        verify(unit).setDefaultAccessToken(PAYER_ID);
    }

    @Test
    public void isPayPalAddressPresentMustReturnTrueTest() {
        final BrainTreeConnectWithPayPalAddressData addressData = mock(BrainTreeConnectWithPayPalAddressData.class);
        final CustomerModel customerModel = mock(CustomerModel.class);
        final Collection<AddressModel> addresses = new ArrayList<>();
        final AddressModel addressModel = mock(AddressModel.class);
        addresses.add(addressModel);
        final CountryModel countryModel = mock(CountryModel.class);

        when(customerModel.getAddresses()).thenReturn(addresses);
        when(userService.getUserForUID(EMAIL)).thenReturn(customerModel);
        when(addressModel.getCountry()).thenReturn(countryModel);
        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(addressData.getCountry()).thenReturn(ISO_CODE);

        assertTrue(unit.isPayPalAddressPresent(EMAIL, addressData));
    }

    @Test
    public void isPayPalPaymentMethodPresentMustReturnTrueTest() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        List<BrainTreePaymentInfoModel> paymentInfoModels = new ArrayList<>();
        BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        paymentInfoModels.add(paymentInfoModel);

        when(userService.getUserForUID(EMAIL)).thenReturn(customerModel);
        when(brainTreeCustomerAccountService.getBrainTreePaymentInfos(customerModel, true)).thenReturn(paymentInfoModels);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(paymentInfoModel.getPayer()).thenReturn(EMAIL);

        assertTrue(unit.isPayPalPaymentMethodPresent(EMAIL));
    }

    @Test
    public void isPayPalPaymentMethodPresentMustReturnFalseTest() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        List<BrainTreePaymentInfoModel> paymentInfoModels = new ArrayList<>();
        BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        paymentInfoModels.add(paymentInfoModel);

        when(userService.getUserForUID(EMAIL)).thenReturn(customerModel);
        when(brainTreeCustomerAccountService.getBrainTreePaymentInfos(customerModel, true)).thenReturn(paymentInfoModels);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.APPLEPAYCARD.getCode());
        when(paymentInfoModel.getPayer()).thenReturn(EMAIL);

        assertFalse(unit.isPayPalPaymentMethodPresent(EMAIL));
    }

    @Test
    public void isPayPalPaymentMethodPresentOrIntentOrderMustReturnTrueTest() {
        when(brainTreeConfigService.isPayPalStandardEnabled()).thenReturn(true);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(true);
        doReturn(false).when(unit).isPayPalPaymentMethodPresent(EMAIL);
        when(brainTreeConfigService.getIntent()).thenReturn(BraintreeConstants.PayPalIntent.PAYPAL_INTENT_ORDER);

        assertTrue(unit.isPayPalPaymentMethodPresentOrIntentOrder(EMAIL));
    }

    @Test
    public void shouldSavePayPalPaymentInfoMustReturnTrueTest() {
        when(brainTreeConfigService.isPayPalStandardEnabled()).thenReturn(true);
        when(brainTreeConfigService.getIntent()).thenReturn(BraintreeConstants.PayPalIntent.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(true);

        assertTrue(unit.shouldSavePayPalPaymentInfo());
    }

    @Test
    public void addAddressForNewUserTest() {
        final BrainTreeConnectWithPayPalAddressData addressData = mock(BrainTreeConnectWithPayPalAddressData.class);
        final AddressModel newAddress = mock(AddressModel.class);
        final CountryModel countryModel = mock(CountryModel.class);
        final CustomerModel customer = mock(CustomerModel.class);
        final RegionModel regionModel = mock(RegionModel.class);

        when(modelService.create(AddressModel.class)).thenReturn(newAddress);
        when(addressData.getCountry()).thenReturn(US);
        when(commonI18NService.getCountry(US)).thenReturn(countryModel);
        when(userService.getUserForUID(EMAIL)).thenReturn(customer);
        when(addressData.getRegion()).thenReturn(AL);
        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(commonI18NService.getRegion(countryModel, ISO_CODE)).thenReturn(regionModel);
        when(customer.getName()).thenReturn(FIRST_NAME + " " + LAST_NAME);

        unit.addAddressForNewUser(addressData, EMAIL);

        verify(brainTreeCustomerAccountService).saveAddressEntry(customer, newAddress);
    }
}
