/**
 *
 */
package com.braintree.facade.impl;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.anyString;

import com.braintree.command.result.BrainTreeFindCustomerResult;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants.PayPalIntent;
import com.braintree.customfield.service.BraintreeCustomFieldsService;
import com.braintree.delivery.service.BraintreeDeliveryService;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.BraintreePageType;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.hybris.data.BraintreeLocalPaymentMethodData;
import com.braintree.hybris.data.PayPalAddressData;
import com.braintree.hybris.data.PayPalCheckoutData;
import com.braintree.hybris.data.PayPalConfigurationData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.model.BraintreeLocalPaymentMethodsModel;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.paypal.converters.impl.BrainTreePayPalAddressDataConverter;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.Environment;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorservices.uiexperience.UiExperienceService;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.catalog.CatalogVersionService;
import de.hybris.platform.catalog.model.CatalogVersionModel;
import de.hybris.platform.commercefacades.order.CartFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commerceservices.order.CommerceCheckoutService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.commerceservices.service.data.CommerceOrderResult;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.media.MediaService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.Session;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.servicelayer.user.UserService;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Currency;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Arrays;
import java.util.Optional;
import org.apache.commons.lang.StringUtils;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

@UnitTest
public class DefaultBrainTreeCheckoutFacadeTest {

    private static final String INTENT = "intent";
    private static final String SHIPS_FROM_POSTAL_CODE = "ships";
    private static final String CUSTOMER_FIELD_KEY = "customerFieldKey";
    private static final String CUSTOMER_FIELD_VALUE = "customFieldValue";
    private static final String SECURITY_CODE = "securityCode";
    private static final String PAYMENT_INFO_ID = "paymentInfoId";
    private static final String PAYMENT_METHOD_NONCE = "paymentMethodNonce";
    private static final String USD = "USD";
    private static final String GOOGLE_IMG_CODE = "googlePayMark";
    private static final String CURRENCY_ISO_CODE = "currencyIsoCode";
    private static final String SITE_UID = "siteUID";
    private static final String MERCHANT_ACCOUNT_ID = "MerchantAccountId";
    private static final String BRAINREE_CUSTOMER_ID = "braintreeCustomerId";
    private static final String COUNTRY_CODE = "countryCode";
    private static final String TOKEN = "token";
    private static final Double DOUBLE_TEN = 10d;
    private static final Double DOUBLE_ONE = 1d;

    @Mock
    private MediaService mediaService;

    @Mock
    private CheckoutCustomerStrategy checkoutCustomerStrategy;

    @Mock
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;

    @Mock
    private CommerceCheckoutService commerceCheckoutService;

    @Mock
    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private CatalogVersionService catalogVersionService;

    @Mock
    private CustomerAccountService customerAccountService;

    @Mock
    private CartService cartService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private ModelService modelService;

    @Mock
    private I18NService i18NService;

    @Mock
    private Converter<OrderModel, OrderData> orderConverter;

    @Mock
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementEntryConverter;

    @Mock
    private BrainTreePayPalAddressDataConverter payPalAddressDataConverter;

    @Mock
    private CartFacade cartFacade;

    @Mock
    private UserService userService;

    @Mock
    private BrainTreeUserFacade brainTreeUserFacade;

    @Mock
    private Converter<BraintreeLocalPaymentMethodsModel, BraintreeLocalPaymentMethodData> brainTreeLocalPaymentsConverter;

    @Mock
    private UiExperienceService uiExperienceService;

    @Mock
    private BraintreeCustomFieldsService customFieldsService;

    @Mock
    private SessionService sessionService;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private BraintreeDeliveryService braintreeDeliveryService;

    @Mock
    private BraintreePaymentInfoService braintreePaymentInfoService;

    @Spy
    @InjectMocks
    private DefaultBrainTreeCheckoutFacade unit;

    private CartModel sessionCardModel;

    private BrainTreePaymentInfoModel sessionPaymentInfoModel;

    private CCPaymentInfoData checkoutPaymentInfoModel;

    private Map<String, String> customFields;


    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        unit.setBrainTreePaymentInfoConverter(brainTreePaymentInfoConverter);
        customFields = new HashMap<>();
        customFields.put(CUSTOMER_FIELD_KEY, CUSTOMER_FIELD_VALUE);
        sessionCardModel = mock(CartModel.class);
        sessionPaymentInfoModel = mock(BrainTreePaymentInfoModel.class);

        when(sessionCardModel.getPaymentInfo()).thenReturn(sessionPaymentInfoModel);
        when(cartService.getSessionCart()).thenReturn(sessionCardModel);
        when(brainTreeConfigService.getIntent()).thenReturn(INTENT);
        when(i18NService.getCurrentJavaCurrency()).thenReturn(Currency.getInstance(USD));
        when(i18NService.getCurrentLocale()).thenReturn(Locale.US);

        CartData checkoutCartData = mock(CartData.class);
        checkoutPaymentInfoModel = mock(CCPaymentInfoData.class);

        when(checkoutCartData.getPaymentInfo()).thenReturn(checkoutPaymentInfoModel);
        when(cartFacade.getSessionCart()).thenReturn(checkoutCartData);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(customFieldsService.getDefaultCustomFieldsMap()).thenReturn(customFields);
    }

    @Test
    public void storeIntentToCartShouldSetIntentAndSavePaymentInfoModel() {
        unit.storeIntentToCart();
        verify(sessionPaymentInfoModel).setPayPalIntent(INTENT);
        verify(modelService).save(sessionPaymentInfoModel);
    }

    @Test
    public void storeShipsFromPostalCodeToCartShouldSetShipsAndSavePaymentInfoModel() {
        unit.storeShipsFromPostalCodeToCart(SHIPS_FROM_POSTAL_CODE);
        verify(sessionPaymentInfoModel).setShipsFromPostalCode(SHIPS_FROM_POSTAL_CODE);
        verify(modelService).save(sessionPaymentInfoModel);
    }

    @Test
    public void storeCustomFieldsToCartShouldSetCustomFieldsAndSavePaymentInfoModel() {
        unit.storeCustomFieldsToCart(customFields);
        verify(sessionPaymentInfoModel).setCustomFields(customFields);
        verify(modelService).save(sessionPaymentInfoModel);
    }

    @Test
    public void authorizePaymentShouldReturnTrueIfPayPalPaymentMethodOrderIntent() {
        when(brainTreeConfigService.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(checkoutPaymentInfoModel.getSubscriptionId()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        assertTrue(unit.authorizePayment(SECURITY_CODE));
    }

    @Test
    public void authorizePaymentMethodCartArgumentTest() {
        CartModel cartModel = mock(CartModel.class);
        PaymentTransactionEntryModel entryModel = mock(PaymentTransactionEntryModel.class);

        when(brainTreeTransactionService.createAuthorizationTransaction(cartModel)).thenReturn(entryModel);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.REVIEW.name());

        assertTrue(unit.authorizePayment(cartModel));
        verify(brainTreeTransactionService).createAuthorizationTransaction(cartModel);
    }

    @Test
    public void authorizePaymentMethodOrderAndAmountArgumentTest() {
        OrderModel orderModel = mock(OrderModel.class);
        BigDecimal amount = mock(BigDecimal.class);
        PaymentTransactionEntryModel entryModel = mock(PaymentTransactionEntryModel.class);
        PaymentTransactionEntryData entryData = mock(PaymentTransactionEntryData.class);

        when(brainTreeTransactionService.createAuthorizationTransaction(orderModel, amount)).thenReturn(entryModel);
        when(ordermanagementEntryConverter.convert(entryModel)).thenReturn(entryData);
        unit.setOrdermanagementPaymentTransactionEntryConverter(ordermanagementEntryConverter);

        unit.authorizePayment(orderModel, amount);

        verify(ordermanagementEntryConverter).convert(entryModel);
    }

    @Test
    public void authorizePaymentSecureCodeAndCustomFieldsArgumentsTest() {
        Map<String, String> customFields = new HashMap<>();
        CartModel cartModel = mock(CartModel.class);
        BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);

        doReturn(true).when(unit).hasCheckoutCart();
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(brainTreeConfigService.isOneOfPaymentMethodsEnabled()).thenReturn(true);

        unit.authorizePayment(SECURITY_CODE, customFields);

        verify(brainTreeTransactionService).createAuthorizationTransaction(customFields);
    }

    @Test
    public void capturePaymentMethodOrderAndAmountArgumentTest() {
        OrderModel orderModel = mock(OrderModel.class);
        BigDecimal amount = mock(BigDecimal.class);
        PaymentTransactionEntryModel entryModel = mock(PaymentTransactionEntryModel.class);
        PaymentTransactionEntryData entryData = mock(PaymentTransactionEntryData.class);

        when(brainTreeTransactionService.createAuthorizationTransactionForCapture(orderModel, amount)).thenReturn(entryModel);
        when(ordermanagementEntryConverter.convert(entryModel)).thenReturn(entryData);
        unit.setOrdermanagementPaymentTransactionEntryConverter(ordermanagementEntryConverter);

        unit.capturePayment(orderModel, amount);

        verify(ordermanagementEntryConverter).convert(entryModel);
    }

    @Test
    public void setPaymentDetailsShouldReturnFalseIfCurrentUserIsNotTheCartUser() {
        UserModel userModel = mock(UserModel.class);
        when(sessionCardModel.getUser()).thenReturn(userModel);

        Assert.assertFalse(unit.setPaymentDetails(PAYMENT_INFO_ID, PAYMENT_METHOD_NONCE, "", false));
    }

    @Test
    public void setPaymentDetailsShouldSetNonceForPaymentInfoAndSavePaymentInfo() {
        CustomerModel customerModel = mock(CustomerModel.class);
        when(sessionCardModel.getUser()).thenReturn(customerModel);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        when(brainTreePaymentService.completeCreateSubscriptionWithCvvVerification(customerModel, PAYMENT_INFO_ID, ""))
            .thenReturn(paymentInfoModel);

        assertTrue(unit.setPaymentDetails(PAYMENT_INFO_ID, PAYMENT_METHOD_NONCE, "", false));
        verify(paymentInfoModel).setNonce(PAYMENT_METHOD_NONCE);
        verify(modelService).save(paymentInfoModel);
    }

    @Test
    public void setPaymentDetailsTest() {
        CustomerModel customerModel = mock(CustomerModel.class);
        CartModel cartModel = mock(CartModel.class);
        BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);

        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getUser()).thenReturn(customerModel);
        when(brainTreePaymentService.completeCreateSubscription(customerModel, PAYMENT_INFO_ID)).thenReturn(paymentInfo);

        assertTrue(unit.setPaymentDetails(PAYMENT_INFO_ID));

        verify(brainTreePaymentService).completeCreateSubscription(customerModel, PAYMENT_INFO_ID);
    }

    @Test
    public void getPayPalCheckoutDataShouldReturnNotNull() {
        BraintreePageType pageType = BraintreePageType.MARK;
        Currency currency = Currency.getInstance(USD);
        Locale locale = Locale.US;
        CustomerModel user = mock(CustomerModel.class);
        Session session = mock(Session.class);
        CurrencyModel currencyModel = mock(CurrencyModel.class);
        List<AddressModel> addressModels = new ArrayList<>();
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);
        when(sessionService.getCurrentSession()).thenReturn(session);
        when(session.getAttribute(any())).thenReturn(currencyModel);
        when(currencyModel.getName()).thenReturn(USD);

        when(i18NService.getCurrentJavaCurrency()).thenReturn(currency);
        when(i18NService.getCurrentLocale()).thenReturn(locale);
        Assert.assertNotNull(unit.getPayPalCheckoutData(pageType));
    }

    @Test
    public void populateShippingAddressTest() throws InvocationTargetException, IllegalAccessException, NoSuchMethodException {
        PayPalCheckoutData checkoutData = mock(PayPalCheckoutData.class);
        CartModel cartModel = mock(CartModel.class);
        AddressModel addressModel = mock(AddressModel.class);
        PayPalAddressData addressData = mock(PayPalAddressData.class);

        when(cartService.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getDeliveryAddress()).thenReturn(addressModel);
        when(payPalAddressDataConverter.convert(addressModel)).thenReturn(addressData);
        unit.setPayPalAddressDataConverter(payPalAddressDataConverter);

        Method populateShippingAddress = DefaultBrainTreeCheckoutFacade.class.getDeclaredMethod("populateShippingAddress", PayPalCheckoutData.class);
        populateShippingAddress.setAccessible(true);
        populateShippingAddress.invoke(unit, checkoutData);

        verify(checkoutData).setShippingAddressOverride(addressData);
    }

    @Test
    public void checkNotGenerateConfigurationDataWithBrainTreeCustomerId() {

        CustomerModel userModel = mock(CustomerModel.class);
        Currency currency = Currency.getInstance(USD);
        Locale locale = Locale.US;
        List<AddressModel> savedAddresses = new ArrayList<>();
        List<BrainTreePaymentDetailsResult> savedPaymentMethods = new ArrayList<>();
        BrainTreeFindCustomerResult customerResult = mock(BrainTreeFindCustomerResult.class);

        when(i18NService.getCurrentJavaCurrency()).thenReturn(currency);
        when(i18NService.getCurrentLocale()).thenReturn(locale);

        when(userService.getCurrentUser()).thenReturn(userModel);
        when(userModel.getAddresses()).thenReturn(savedAddresses);

        when(userModel.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        when(brainTreeUserFacade.findBrainTreeCustomer(BRAINREE_CUSTOMER_ID)).thenReturn(customerResult);
        when(customerResult.getPaymentMethods()).thenReturn(savedPaymentMethods);

        when(brainTreeConfigService.isDropInEnabled()).thenReturn(false);

        unit.generateConfigurationData(BraintreePageType.ACCOUNT);

        verify(unit).generateClientToken(false);

    }

    @Test
    public void checkGenerateConfigurationDataWithBrainTreeCustomerId() {

        CustomerModel userModel = mock(CustomerModel.class);
        Currency currency = Currency.getInstance(USD);
        Locale locale = Locale.US;
        List<AddressModel> savedAddresses = new ArrayList<>();
        List<BrainTreePaymentDetailsResult> savedPaymentMethods = new ArrayList<>();
        BrainTreeFindCustomerResult customerResult = mock(BrainTreeFindCustomerResult.class);

        when(i18NService.getCurrentJavaCurrency()).thenReturn(currency);
        when(i18NService.getCurrentLocale()).thenReturn(locale);

        when(userService.getCurrentUser()).thenReturn(userModel);
        when(userModel.getAddresses()).thenReturn(savedAddresses);

        when(userModel.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        when(brainTreeUserFacade.findBrainTreeCustomer(BRAINREE_CUSTOMER_ID)).thenReturn(customerResult);
        when(customerResult.getPaymentMethods()).thenReturn(savedPaymentMethods);

        when(brainTreeConfigService.isDropInEnabled()).thenReturn(true);

        unit.generateConfigurationData(BraintreePageType.ACCOUNT);

        verify(unit).generateClientToken(true);

    }

    @Test
    public void countDeliveryAddressTest() {
        CartModel cartModel = mock(CartModel.class);

        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);

        unit.countDeliveryAddress(COUNTRY_CODE);

        verify(braintreeDeliveryService).countDeliveryPriceForOrderAndCountyCode(cartModel, COUNTRY_CODE);
    }

    @Test
    public void getBraintreeAssetsShouldCallGetDownloadURLForEnabledPaymentMethods() {
        when(brainTreeConfigService.isGooglePayEnabled()).thenReturn(true);
        when(brainTreeConfigService.isVenmoEnabled()).thenReturn(true);
        when(brainTreeConfigService.isApplePayEnabled()).thenReturn(true);

        CatalogVersionModel catalogVersionModel = mock(CatalogVersionModel.class);
        List<CatalogVersionModel> catalogVersionModels = new ArrayList<>();
        catalogVersionModels.add(catalogVersionModel);
        MediaModel mediaModel = mock(MediaModel.class);

        when(catalogVersionService.getSessionCatalogVersions()).thenReturn(catalogVersionModels);
        when(mediaService.getMedia(catalogVersionModel, GOOGLE_IMG_CODE)).thenReturn(mediaModel);

        unit.getBraintreeAssets();
        verify(mediaModel).getDownloadURL();
    }

    @Test
    public void isPayPalCheckoutTest() {
        OrderModel orderModel = mock(OrderModel.class);
        BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);

        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getProperty(anyString())).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());

        assertTrue(unit.isPayPalCheckout(orderModel));
    }

    @Test
    public void handleOrderIntentViaSubscriptionTest() {
        OrderModel orderModel = mock(OrderModel.class);
        BrainTreePaymentInfoModel basePaymentInfo = mock(BrainTreePaymentInfoModel.class);
        BaseSiteModel baseSiteModel = mock(BaseSiteModel.class);
        CurrencyModel currencyModel = mock(CurrencyModel.class);
        CCPaymentInfoData ccPaymentInfoData = mock(CCPaymentInfoData.class);
        Map<String, String> customFields = new HashMap<>();
        BrainTreePaymentInfoModel createdPaymentInfo = mock(BrainTreePaymentInfoModel.class);

        doNothing().when(modelService).refresh(orderModel);
        when(orderModel.getPaymentInfo()).thenReturn(basePaymentInfo);

        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getUid()).thenReturn(SITE_UID);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE);
        when(brainTreeConfigService.getMerchantAccountIdByCurrentSiteNameAndCurrency(SITE_UID, CURRENCY_ISO_CODE)).thenReturn(MERCHANT_ACCOUNT_ID);

        when(brainTreePaymentFacade.completeCreateSubscriptionForMyAccount(any(), any())).thenReturn(ccPaymentInfoData);
        when(braintreePaymentInfoService.findByCustomerIdAndCode(anyString(), anyString())).thenReturn(Optional.ofNullable(createdPaymentInfo));
        when(customFieldsService.getDefaultCustomFieldsMap()).thenReturn(customFields);
        when(createdPaymentInfo.getPaymentMethodToken()).thenReturn(TOKEN);

        unit.handleOrderIntentViaSubscription(orderModel);

        verify(createdPaymentInfo).setPayPalIntent(PayPalIntent.PAYPAL_INTENT_ORDER);
        verify(createdPaymentInfo).setCustomFields(customFields);
        verify(modelService).save(createdPaymentInfo);
        verify(orderModel).setPaymentInfo(createdPaymentInfo);
        verify(modelService).save(orderModel);
        verify(brainTreeTransactionService).createOrderTransaction(orderModel, TOKEN);
    }

    @Test
    public void placeOrderByCartShouldUseOrderConverterForPlacedOrderModel() throws InvalidCartException {
        CartModel cartModel = mock(CartModel.class);
        AddressModel deliveryAddress = mock(AddressModel.class);
        CommerceOrderResult orderResult = mock(CommerceOrderResult.class);
        OrderModel orderModel = mock(OrderModel.class);

        when(orderResult.getOrder()).thenReturn(orderModel);
        when(commerceCheckoutService.placeOrder(any(CommerceCheckoutParameter.class))).thenReturn(orderResult);
        when(cartModel.getDeliveryAddress()).thenReturn(deliveryAddress);
        unit.placeOrderByCart(cartModel);
        verify(orderConverter).convert(orderModel);
    }

    @Test
    public void generateConfigurationDataForMyAccountIsCorrect() {
        when(cartService.hasSessionCart()).thenReturn(false);
        when(brainTreeConfigService.getIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeUserFacade.isAnonymousUser()).thenReturn(false);
        when(brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId()).thenReturn(true);

        CustomerModel user = mock(CustomerModel.class);
        List<AddressModel> addressModels = new ArrayList<>();
        addressModels.add(mock(AddressModel.class));
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        PayPalCheckoutData payPalCheckoutData = unit.generateConfigurationData(BraintreePageType.ACCOUNT);
        PayPalConfigurationData configurationData = payPalCheckoutData.getConfigurationData();
        verify(brainTreePaymentService).generateClientToken(StringUtils.EMPTY);
        Assert.assertEquals(PayPalIntent.PAYPAL_INTENT_SALE, configurationData.getIntent());
        Assert.assertEquals(Currency.getInstance(USD).getCurrencyCode(), configurationData.getCurrency());
        Assert.assertEquals(Locale.US.toString(), configurationData.getLocale());
        Assert.assertFalse(payPalCheckoutData.getEnableShippingAddress());
        Assert.assertFalse(payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton());
    }

    @Test
    public void generateConfigurationDataForMarkPageShouldRenderChangePaymentButton() {
        when(cartService.hasSessionCart()).thenReturn(true);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeUserFacade.isAnonymousUser()).thenReturn(false);
        when(brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId()).thenReturn(true);

        CartModel cartModel = mock(CartModel.class);
        when(cartModel.getTotalPrice()).thenReturn(DOUBLE_TEN);
        when(cartModel.getSubtotal()).thenReturn(DOUBLE_ONE);
        when(cartService.getSessionCart()).thenReturn(cartModel);

        CustomerModel user = mock(CustomerModel.class);
        when(user.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        List<AddressModel> addressModels = new ArrayList<>();
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        BrainTreeFindCustomerResult findCustomerResult = mock(BrainTreeFindCustomerResult.class);
        List<BrainTreePaymentDetailsResult> paymentMethods = new ArrayList<>();
        BrainTreePaymentDetailsResult paymentDetailsResult = mock(BrainTreePaymentDetailsResult.class);
        paymentMethods.add(paymentDetailsResult);
        when(brainTreeUserFacade.findBrainTreeCustomer(BRAINREE_CUSTOMER_ID)).thenReturn(findCustomerResult);
        when(findCustomerResult.getPaymentMethods()).thenReturn(paymentMethods);
        when(paymentDetailsResult.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());

        PayPalCheckoutData payPalCheckoutData = unit.generateConfigurationData(BraintreePageType.MARK);
        assertTrue(payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton());
        Assert.assertEquals(DOUBLE_TEN, payPalCheckoutData.getAmount());
        Assert.assertEquals(DOUBLE_ONE, payPalCheckoutData.getSubtotal());
    }

    @Test
    public void generateConfigurationDataForMarkPageShouldNotRenderChangePaymentButtonWhenAnonymousUser() {
        when(cartService.hasSessionCart()).thenReturn(false);
        when(brainTreeConfigService.isChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeUserFacade.isAnonymousUser()).thenReturn(true);
        when(brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId()).thenReturn(false);

        CustomerModel user = mock(CustomerModel.class);
        when(user.getBraintreeCustomerId()).thenReturn(null);
        List<AddressModel> addressModels = new ArrayList<>();
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        PayPalCheckoutData payPalCheckoutData = unit.generateConfigurationData(BraintreePageType.MARK);
        verify(brainTreePaymentService).generateClientToken(StringUtils.EMPTY);
        Assert.assertFalse(payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton());
    }

    @Test
    public void placeOrderForLocalPaymentReversedTest() throws InvalidCartException {
        CartModel cartModel = mock(CartModel.class);
        OrderModel orderModel = mock(OrderModel.class);
        CommerceOrderResult orderResult = mock(CommerceOrderResult.class);

        doNothing().when(unit).beforePlaceOrder(cartModel);
        when(commerceCheckoutService.placeOrder(any(CommerceCheckoutParameter.class))).thenReturn(orderResult);
        when(orderResult.getOrder()).thenReturn(orderModel);

        unit.placeOrderForLocalPaymentReversed(cartModel);

        verify(unit).afterPlaceOrder(cartModel, orderModel);
    }

    @Test
    public void processLocalPaymentReversedWebhookTest() {
        CartModel cartModel = mock(CartModel.class);
        PaymentTransactionEntryModel entryModel = mock(PaymentTransactionEntryModel.class);

        when(brainTreeTransactionService.createTransactionEntryForLocalPaymentReversed(cartModel, PaymentTransactionType.AUTHORIZATION)).thenReturn(entryModel);
        when(brainTreeTransactionService.createTransactionEntryForLocalPaymentReversed(cartModel, PaymentTransactionType.CAPTURE)).thenReturn(entryModel);
        when(brainTreeTransactionService.createTransactionEntryForLocalPaymentReversed(cartModel, PaymentTransactionType.REFUND_FOLLOW_ON)).thenReturn(entryModel);

        unit.processLocalPaymentReversedWebhook(cartModel);

        verify(brainTreeTransactionService).createPaymentTransactionForLocalPaymentReversed(cartModel, Arrays.asList(entryModel, entryModel, entryModel));
    }

    @Test
    public void getPayPalConfigurationDataForConnectWithPayPalButtonTest() {
        String clientID = "clientID";
        String connectScript = "connectScript";
        String connectScopes = "connectScopes";
        String connectResponseType = "connectResponseType";
        String connectReturnUrl = "connectReturnUrl";
        String environmentName = "environmentName";
        Environment environment = mock(Environment.class);
        boolean connectAddPaymentMethodFlow = true;
        boolean EnableLoginViaPayPalToTheExistingUser = true;

        when(brainTreeConfigService.getClientID()).thenReturn(clientID);
        when(brainTreeConfigService.getPayPalConnectScript()).thenReturn(connectScript);
        when(brainTreeConfigService.getPayPalConnectScopes()).thenReturn(connectScopes);
        when(brainTreeConfigService.getPayPalConnectResponseType()).thenReturn(connectResponseType);
        when(brainTreeConfigService.getPayPalConnectReturnUrl()).thenReturn(connectReturnUrl);
        when(environment.getEnvironmentName()).thenReturn(environmentName);
        when(brainTreeConfigService.getEnvironmentType()).thenReturn(environment);
        when(brainTreeConfigService.isPayPalConnectAddPaymentMethodFlow()).thenReturn(connectAddPaymentMethodFlow);
        when(brainTreeConfigService.isEnableLoginViaPayPalToTheExistingUser()).thenReturn(EnableLoginViaPayPalToTheExistingUser);

        PayPalCheckoutData result = unit.getPayPalConfigurationDataForConnectWithPayPalButton();

        assertEquals(clientID, result.getConfigurationData().getClient_id());
        assertEquals(connectScript, result.getConfigurationData().getPayPalConnectScript());
        assertEquals(connectScopes, result.getConfigurationData().getPayPalConnectScopes());
        assertEquals(connectResponseType, result.getConfigurationData().getPayPalConnectResponseType());
        assertEquals(connectReturnUrl, result.getConfigurationData().getPayPalConnectReturnUrl());
        assertEquals(environmentName, result.getConfigurationData().getEnvironment());
        assertEquals(connectAddPaymentMethodFlow, result.getConfigurationData().isPayPalConnectAddPaymentMethodFlow());
        assertEquals(EnableLoginViaPayPalToTheExistingUser, result.getConfigurationData().isEnableLoginViaPayPalToTheExistingUser());

    }
}
