/**
 *
 */
package com.braintree.facade.backoffice.impl;

import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.model.BrainTreePaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

@UnitTest
public class DefaultBraintreeBackofficeMultiCaptureFacadeTest {

    private static final String PAYPAL_INTENT = "paypalIntent";
    private static final String PAYMENT_PROVIDER_PROPERTY = "paymentProvider";
    private static final String TRANSACTION_ID_1 = "transaction1";
    private static final String TRANSACTION_ID_2 = "transaction2";

    @Mock
    private ModelService modelService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Spy
    @InjectMocks
    private DefaultBraintreeBackofficeMultiCaptureFacade unit;


    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void isMultiCapturePossibleShouldReturnFalseIfOrderModelNull() {
        Assert.assertFalse(unit.isMultiCapturePossible(null));
    }

    @Test
    public void isMultiCapturePossibleShouldReturnFalseIfPaymentInfoNotBraintreeType() {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        Assert.assertFalse(unit.isMultiCapturePossible(orderModel));
    }

    @Test
    public void isMultiCapturePossibleShouldReturnFalseWhenOrderNotHaveTransactions() {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);

        when(paymentInfoModel.getPayPalIntent()).thenReturn(PAYPAL_INTENT);
        when(paymentInfoModel.getProperty(PAYMENT_PROVIDER_PROPERTY)).thenReturn(PAYMENT_PROVIDER_PROPERTY);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        Assert.assertFalse(unit.isMultiCapturePossible(orderModel));
    }

    @Test
    public void getMultiCaptureableTransactionsShouldGetTypeAndStatusFromTransactionEntry() {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        PaymentTransactionModel transactionModel = Mockito.mock(PaymentTransactionModel.class);
        List<PaymentTransactionModel> transactionModelList = new ArrayList<>();
        transactionModelList.add(transactionModel);

        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        List<PaymentTransactionEntryModel> entryModels = new ArrayList<>();
        entryModels.add(entryModel);

        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionModel.getEntries()).thenReturn(entryModels);
        when(orderModel.getPaymentTransactions()).thenReturn(transactionModelList);

        unit.getMultiCaptureableTransactions(orderModel);
        verify(entryModel).getType();
        verify(entryModel).getTransactionStatus();
    }

    @Test
    public void getAuthorizationAmountByTransactionIdShouldReturnCorrectAmount() {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        List<PaymentTransactionEntryModel> multiCaptureTransactions = new ArrayList<>();
        PaymentTransactionEntryModel transaction1 = Mockito.mock(PaymentTransactionEntryModel.class);
        when(transaction1.getRequestId()).thenReturn(TRANSACTION_ID_1);
        when(transaction1.getAmount()).thenReturn(BigDecimal.TEN);
        PaymentTransactionEntryModel transaction2 = Mockito.mock(PaymentTransactionEntryModel.class);
        when(transaction2.getRequestId()).thenReturn(TRANSACTION_ID_2);
        when(transaction2.getAmount()).thenReturn(BigDecimal.ONE);
        multiCaptureTransactions.add(transaction1);
        multiCaptureTransactions.add(transaction2);

        doReturn(multiCaptureTransactions).when(unit).getMultiCaptureableTransactions(orderModel);

        BigDecimal result = unit
            .getAuthorizationAmountByTransactionId(orderModel, TRANSACTION_ID_1);

        Assert.assertEquals(BigDecimal.TEN, result);
    }

    @Test
    public void getAuthorizationAmountByTransactionIdShouldReturnZeroIfDontFindAuthorizeTransaction() {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        List<PaymentTransactionEntryModel> multiCaptureTransactions = new ArrayList<>();
        PaymentTransactionEntryModel transaction1 = Mockito.mock(PaymentTransactionEntryModel.class);
        when(transaction1.getRequestId()).thenReturn(TRANSACTION_ID_1);
        when(transaction1.getAmount()).thenReturn(BigDecimal.TEN);
        multiCaptureTransactions.add(transaction1);

        doReturn(multiCaptureTransactions).when(unit).getMultiCaptureableTransactions(orderModel);

        BigDecimal result = unit
            .getAuthorizationAmountByTransactionId(orderModel, TRANSACTION_ID_2);

        Assert.assertEquals(BigDecimal.ZERO, result);
    }

}
