/**
 *
 */
package com.braintree.facade.backoffice.impl;

import static com.braintree.constants.BraintreeConstants.PayPalIntent.PAYPAL_INTENT_ORDER;
import static com.braintree.constants.BraintreeConstants.PayPalIntent.PAYPAL_INTENT_SALE;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.Map;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultBraintreeBackofficeAuthorizeFacadeTest {

    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final BigDecimal AMOUNT = BigDecimal.TEN;
    private static final Double TOTAL_PRICE = 200d;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;

    @Mock
    private ModelService modelService;

    @Mock
    private BraintreeOrderBackofficeUtilService backofficeUtilService;

    @InjectMocks
    private DefaultBraintreeBackofficeAuthorizeFacade unit;


    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void isAuthorizationPossobleShouldReturnFalseIfOrderModelNull() {
        Assert.assertFalse(unit.isAuthorizePossible(null));
    }

    @Test
    public void authorizationPaymentShouldUseTransactionServiceForCreatingAuthorizationTransaction() {
        final AbstractOrderModel orderModel = Mockito.mock(AbstractOrderModel.class);
        final Map<String, String> customFields = Mockito.mock(Map.class);
        unit.authorizePayment(orderModel, customFields, BigDecimal.TEN);
        verify(brainTreeTransactionService).createAuthorizationTransaction(orderModel, customFields, AMOUNT);
    }

    @Test
    public void isAuthorizePossibleShouldCheckOrderOnIntent() {
        final OrderModel orderModel = Mockito.mock(OrderModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);

        when(paymentInfoModel.getPaymentProvider()).thenReturn(PAYMENT_PROVIDER);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(orderModel.getTotalPrice()).thenReturn(TOTAL_PRICE);

        unit.isAuthorizePossible(orderModel);
        verify(paymentInfoModel).getPayPalIntent();
    }

    @Test
    public void isAuthorizePossibleShouldReturnNullIfOrderNull() {
        Assert.assertFalse(unit.isAuthorizePossible(null));
    }

    @Test
    public void isAuthorizePossibleShouldReturnTrueForPayPalWithOrderIntent() {
        final OrderModel orderModel = Mockito.mock(OrderModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(paymentInfoModel.getPayPalIntent()).thenReturn(PAYPAL_INTENT_ORDER);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        when(backofficeUtilService.isOrderAmountCaptured(orderModel)).thenReturn(false);
        when(backofficeUtilService.isOrderVoided(orderModel)).thenReturn(false);
        when(backofficeUtilService.paidByCard(orderModel)).thenReturn(false);

        Assert.assertTrue(unit.isAuthorizePossible(orderModel));
    }

    @Test
    public void isAuthorizePossibleShouldReturnFalseForPayPalWithOrderSale() {
        final OrderModel orderModel = Mockito.mock(OrderModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(paymentInfoModel.getPayPalIntent()).thenReturn(PAYPAL_INTENT_SALE);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        when(backofficeUtilService.isOrderAmountCaptured(orderModel)).thenReturn(false);
        when(backofficeUtilService.isOrderVoided(orderModel)).thenReturn(false);
        when(backofficeUtilService.paidByCard(orderModel)).thenReturn(false);

        Assert.assertFalse(unit.isAuthorizePossible(orderModel));
    }

    @Test
    public void isAuthorizePossibleShouldReturnFalseForVenmoWithOrderIntent() {
        final OrderModel orderModel = Mockito.mock(OrderModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BrainTreePaymentMethod.VENMOACCOUNT.getCode());
        when(paymentInfoModel.getPayPalIntent()).thenReturn(PAYPAL_INTENT_ORDER);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        when(backofficeUtilService.isOrderAmountCaptured(orderModel)).thenReturn(false);
        when(backofficeUtilService.isOrderVoided(orderModel)).thenReturn(false);
        when(backofficeUtilService.paidByCard(orderModel)).thenReturn(false);

        Assert.assertFalse(unit.isAuthorizePossible(orderModel));
    }

}
